
static GLboolean
CheckArray(GLenum array, GLint arrayIndex, GLint maxIndex)
{
   struct array_info {
      GLenum array;
      GLenum bufferQuery, pointerQuery, sizeQuery, typeQuery, strideQuery;
   };

   static const struct array_info info[] = {
      {
         GL_VERTEX_ARRAY,
         GL_VERTEX_ARRAY_BUFFER_BINDING_ARB,
         GL_VERTEX_ARRAY_POINTER,
         GL_VERTEX_ARRAY_SIZE,
         GL_VERTEX_ARRAY_TYPE,
         GL_VERTEX_ARRAY_STRIDE
      },
      {
         GL_NORMAL_ARRAY,
         GL_NORMAL_ARRAY_BUFFER_BINDING_ARB,
         GL_NORMAL_ARRAY_POINTER,
         3/*always*/,
         GL_NORMAL_ARRAY_TYPE,
         GL_NORMAL_ARRAY_STRIDE
      },
      {
         GL_COLOR_ARRAY,
         GL_COLOR_ARRAY_BUFFER_BINDING_ARB,
         GL_COLOR_ARRAY_POINTER,
         GL_COLOR_ARRAY_SIZE,
         GL_COLOR_ARRAY_TYPE,
         GL_COLOR_ARRAY_STRIDE
      },
      {
         GL_SECONDARY_COLOR_ARRAY,
         GL_SECONDARY_COLOR_ARRAY_BUFFER_BINDING_ARB,
         GL_SECONDARY_COLOR_ARRAY_POINTER,
         GL_SECONDARY_COLOR_ARRAY_SIZE,
         GL_SECONDARY_COLOR_ARRAY_TYPE,
         GL_SECONDARY_COLOR_ARRAY_STRIDE
      },
      {
         GL_FOG_COORDINATE_ARRAY,
         GL_FOG_COORDINATE_ARRAY_BUFFER_BINDING_ARB,
         GL_FOG_COORDINATE_ARRAY_POINTER,
         1/*always*/,
         GL_FOG_COORDINATE_ARRAY_TYPE,
         GL_FOG_COORDINATE_ARRAY_STRIDE
      },
      {
         GL_TEXTURE_COORD_ARRAY,
         GL_TEXTURE_COORD_ARRAY_BUFFER_BINDING_ARB,
         GL_TEXTURE_COORD_ARRAY_POINTER,
         GL_TEXTURE_COORD_ARRAY_SIZE,
         GL_TEXTURE_COORD_ARRAY_TYPE,
         GL_TEXTURE_COORD_ARRAY_STRIDE
      },
      {
         GL_VERTEX_ATTRIB_ARRAY_ENABLED,
         GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING,
         GL_VERTEX_ATTRIB_ARRAY_POINTER,
         GL_VERTEX_ATTRIB_ARRAY_SIZE,
         GL_VERTEX_ATTRIB_ARRAY_TYPE,
         GL_VERTEX_ATTRIB_ARRAY_STRIDE
      },
      { 0, 0, 0, 0, 0, 0 }
   };

   const GLboolean isTextureArray = (array == GL_TEXTURE_COORD_ARRAY);
   const GLboolean isAttribArray = (array == GL_VERTEX_ATTRIB_ARRAY_ENABLED);
   GLenum bufferQuery, pointerQuery, sizeQuery, typeQuery, strideQuery;
   GLint enabled;
   GLuint i;

   /* get/find GL array query tokens */
   for (i = 0; info[i].array; i++) {
      if (info[i].array == array) {
         bufferQuery = info[i].bufferQuery;
         pointerQuery = info[i].pointerQuery;
         sizeQuery = info[i].sizeQuery;
         typeQuery = info[i].typeQuery;
         strideQuery = info[i].strideQuery;
         break;
      }        
   }


   if (isAttribArray)
      glGetVertexAttribivARB(arrayIndex, array, &enabled);
   else
      enabled = glIsEnabled(array);

   if (enabled) {
      GLint buf, bufSave;
      GLint bufSize;

      if (isAttribArray)
         glGetVertexAttribivARB(arrayIndex, bufferQuery, &buf);
      else
         glGetIntegerv(bufferQuery, &buf);

      if (buf) {
         GLint mapped;
         GLvoid *ptr;
         GLint size, type, stride;
         GLint curTexUnitSave;
         GLint typeSize, elemSize;

         /* this array lives in a VBO */
         glGetIntegerv(GL_ARRAY_BUFFER_BINDING_ARB, &bufSave);

         /* get buffer size */
         glBindBufferARB(GL_ARRAY_BUFFER_ARB, buf);
         glGetBufferParameterivARB(GL_ARRAY_BUFFER_ARB, GL_BUFFER_SIZE_ARB,
                                   &bufSize);

         /* check that it's not mapped */
         glGetBufferParameterivARB(GL_ARRAY_BUFFER_ARB,
                                   GL_BUFFER_MAPPED_ARB, &mapped);
         glBindBufferARB(GL_ARRAY_BUFFER_ARB, bufSave);
         if (mapped) {
            fprintf(stderr, "glDrawElements(): array 0x%x buffer is mapped.\n",
                    array);
            return GL_FALSE;
         }

         if (isTextureArray) {
            /* save/set tex unit */
            glGetIntegerv(GL_CLIENT_ACTIVE_TEXTURE, &curTexUnitSave);
            glClientActiveTexture(GL_TEXTURE0 + arrayIndex);
         }

         /* get array element size, type, stride info */
         if (isAttribArray) {
            glGetVertexAttribPointerv(arrayIndex, pointerQuery, &ptr);
            if (sizeQuery <= 4)
               size = sizeQuery;
            else
               glGetVertexAttribiv(arrayIndex, sizeQuery, &size);
            glGetVertexAttribiv(arrayIndex, typeQuery, &type);
            glGetVertexAttribiv(arrayIndex, strideQuery, &stride);
         }
         else {
            glGetPointerv(pointerQuery, &ptr);
            if (sizeQuery <= 4)
               size = sizeQuery;
            else
               glGetIntegerv(sizeQuery, &size);
            glGetIntegerv(typeQuery, &type);
            glGetIntegerv(strideQuery, &stride);
         }

         if (isTextureArray) {
            /* restore tex unit */
            glClientActiveTexture(curTexUnitSave);
         }

         /* get datatype's size in bytes */
         switch (type) {
         case GL_BYTE:
         case GL_UNSIGNED_BYTE:
            typeSize = 1;
            break;
         case GL_SHORT:
         case GL_UNSIGNED_SHORT:
            typeSize = 2;
            break;
         case GL_INT:
         case GL_UNSIGNED_INT:
         case GL_FLOAT:
            typeSize = 4;
            break;
         default:
            fprintf(stderr, "glDrawElements(): Unexpected array type: 0x%x\n",
                    type);
            return GL_FALSE;
         }

         /* compute element size, actual stride */
         elemSize = size * typeSize;
         if (stride == 0)
            stride = elemSize;

         /* check that the last array element is not out of VBO's bounds */
         if (buf && (long) ptr + stride * maxIndex + elemSize > bufSize) {
            fprintf(stderr,
                    "glDrawElements(): array 0x%x[%d]'s element %d is out of "
                    "bounds of buffer %d whose size is %d.\n",
                    array, arrayIndex, maxIndex, buf, bufSize);
            return GL_FALSE;
         }
      }

      /* XXX we could also scan the vertex data for nan/inf/etc */
   }

   return GL_TRUE;
}


/**
 * Check the current vertex, color, texcoord, attrib, etc. arrays.
 */
static GLboolean
CheckArrays(GLint maxIndex)
{
   GLint numTexUnits, numAttribs;
   GLint i;
   GLboolean ok = GL_TRUE;

   glGetIntegerv(GL_MAX_TEXTURE_UNITS, &numTexUnits);
   glGetIntegerv(GL_MAX_VERTEX_ATTRIBS_ARB, &numAttribs);

   ok |= CheckArray(GL_VERTEX_ARRAY, 0, maxIndex);
   ok |= CheckArray(GL_NORMAL_ARRAY, 0, maxIndex);
   ok |= CheckArray(GL_COLOR_ARRAY, 0, maxIndex);
   ok |= CheckArray(GL_SECONDARY_COLOR_ARRAY, 0, maxIndex);
   ok |= CheckArray(GL_FOG_COORDINATE_ARRAY, 0, maxIndex);
   for (i = 0; i < numTexUnits; i++) {
      ok |= CheckArray(GL_TEXTURE_COORD_ARRAY, i, maxIndex);
   }
   for (i = 0; i < numAttribs; i++) {
      ok |= CheckArray(GL_VERTEX_ATTRIB_ARRAY_ENABLED, i, maxIndex);
   }

   return ok;
}


/**
 * Find the min/max indices for a glDraw[Range]Elements() call.
 */
static GLboolean
GetMinMaxIndices(GLsizei count, GLenum type, const GLvoid *indices,
                 GLint *min, GLint *max)
{
   GLint minIndex, maxIndex;
   GLint indexBuffer = 0;
   const GLvoid *indicesMap;
   GLint i;

   glGetIntegerv(GL_ELEMENT_ARRAY_BUFFER_BINDING_ARB, &indexBuffer);
   if (indexBuffer) {
      /* indices are in a buffer object */
      /* check that buffer is not mapped */
      GLint mapped;
      glGetBufferParameterivARB(GL_ELEMENT_ARRAY_BUFFER_ARB,
                                GL_BUFFER_MAPPED_ARB, &mapped);
      if (mapped) {
         fprintf(stderr, "glDrawElements(): element buffer is mapped.\n");
         return GL_FALSE;
      }
      /* map it now (unmapped below) */
      indicesMap = glMapBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB,
                                  GL_READ_ONLY_ARB);
      if (!indicesMap) {
         fprintf(stderr, "glDrawElements(): unable to map index buffer\n");
         return GL_FALSE;
      }

      /* indices is an offset into the index buffer */
      indicesMap = (const void *) ((GLubyte *) indicesMap + (long) indices);
   }
   else {
      /* user-space indices */
      indicesMap = indices;
   }

   /* find min/max indices */
   switch (type) {
   case GL_UNSIGNED_BYTE:
      {
         const GLubyte *ub = (const GLubyte *) indicesMap;
         minIndex = maxIndex = ub[0];
         for (i = 1; i < count; i++) {
            if (ub[i] < minIndex)
               minIndex = ub[i];
            if (ub[i] > maxIndex)
               maxIndex = ub[i];
         }
      }
      break;
   case GL_UNSIGNED_SHORT:
      {
         const GLushort *us = (const GLushort *) indicesMap;
         minIndex = maxIndex = us[0];
         for (i = 1; i < count; i++) {
            if (us[i] < minIndex)
               minIndex = us[i];
            if (us[i] > maxIndex)
               maxIndex = us[i];
         }
      }
      break;
   case GL_UNSIGNED_INT:
      {
         const GLuint *ui = (const GLuint *) indicesMap;
         minIndex = maxIndex = ui[0];
         for (i = 1; i < count; i++) {
            if (ui[i] < minIndex)
               minIndex = ui[i];
            if (ui[i] > maxIndex)
               maxIndex = ui[i];
         }
      }
      break;
   default:
      fprintf(stderr, "glDrawElements(): Bad type=0x%x\n", type);
      return GL_FALSE;
   }

   if (indexBuffer) {
      glUnmapBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB);
   }

   *min = minIndex;
   *max = maxIndex;

   return GL_TRUE;
}


/**
 * Do sanity checking for a glDrawElements call.
 */
static GLboolean
CheckDrawElements(GLenum mode, GLsizei count, GLenum type,
                  const GLvoid *indices)
{
   GLint minIndex, maxIndex;

   if (!GetMinMaxIndices(count, type, indices, &minIndex, &maxIndex))
      return GL_FALSE;

   return CheckArrays(maxIndex);
}


/**
 * Do sanity checking for a glDrawRangeElements call.
 */
static GLboolean
CheckDrawRangeElements(GLenum mode, GLuint start, GLuint end,
                       GLsizei count, GLenum type, const GLvoid *indices)
{
   GLint minIndex, maxIndex;

   if (!GetMinMaxIndices(count, type, indices, &minIndex, &maxIndex))
      return GL_FALSE;

   if (minIndex < (GLint) start) {
      fprintf(stderr,
              "glDrawRangeElements(): minIndex = %d but start = %u\n",
              minIndex, start);
      return GL_FALSE;
   }

   if (maxIndex < (GLint) end) {
      fprintf(stderr,
              "glDrawRangeElements(): maxIndex = %d but end = %u\n",
              maxIndex, start+end);
      return GL_FALSE;
   }

   return CheckArrays(maxIndex);
}


/**
 * Do sanity checking for a glDrawArrays call.
 */
static GLboolean
CheckDrawArrays(GLenum mode, GLint first, GLsizei count)
{
   return CheckArrays(count - 1);
}
