/* msm-binder.c
 *
 * Copyright (c) 2009, Code Aurora Forum. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Code Aurora nor
 *       the names of its contributors may be used to endorse or promote
 *       products derived from this software without specific prior written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NON-INFRINGEMENT ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#include <pthread.h>
#include <sys/socket.h>
#include <sys/un.h>

#include "msm.h"

static pthread_t binderid;

static pthread_attr_t attr;

struct cmsghdr *cmptr;

static uid_t *tptr;

#define QLEN 10
#define CONTROLLEN  CMSG_LEN(sizeof(int))
//#define offsetof(TYPE, MEMBER) ((int)&((TYPE *)0)->MEMBER)
#define MAXLINE 4096
#define STALE   30

static int
MSMBinderSocket(const char *name, gid_t gid)
{
   int fd = socket(AF_UNIX, SOCK_STREAM, 0);

   struct sockaddr_un un;

   int len, ret;

   if (fd < 0) {
      ErrorF("%s: ERROR socket(): %m\n", __FUNCTION__);
      return -1;
   }

   unlink(name);

   memset(&un, 0, sizeof(un));
   un.sun_family = AF_UNIX;
   strcpy(un.sun_path, name);

   len = offsetof(struct sockaddr_un, sun_path) + strlen(name);

   ret = bind(fd, (struct sockaddr *)&un, len);

   if (ret < 0) {
      ErrorF("%s: ERROR bind() %m\n", __FUNCTION__);
      close(fd);
      return -1;
   }

   /* Set the permissions on the file to the desired group - the
      effective owner remains the same */

   if (chown(name, geteuid(), gid))
	ErrorF("%s: ERROR chown(): %m\n", __FUNCTION__);

   /* Set the mode 0660 - read write by user and owner */
   if (chmod(name, 0660))
	ErrorF("%s: chmod(): %m\n", __FUNCTION__);

   ret = listen(fd, QLEN);

   if (ret < 0) {
      ErrorF("%s: ERROR listen() %m\n", __FUNCTION__);
      close(fd);
      return -1;
   }

   return fd;
}

static int
MSMBinderAccept(int fd, uid_t * uidptr)
{
   socklen_t len;

   int cfd;

   time_t staletime;

   struct sockaddr_un un;

   struct stat s;

   len = sizeof(un);

   cfd = accept(fd, (struct sockaddr *)&un, &len);

   if (cfd < 0) {
      /* Don't log a message on EINTR */
      if (errno != EINTR)
	ErrorF("%s: ERROR accept() %m\n", __FUNCTION__);
      return cfd;
   }

   if (stat(un.sun_path, &s) < 0) {
      close(cfd);
      ErrorF("%s: ERROR stat() %m\n", __FUNCTION__);
      return -1;
   }

   if ((s.st_mode & (S_IRWXG | S_IRWXO)) || (s.st_mode & S_IRWXU) != S_IRWXU) {
      close(cfd);
      return -1;
   }

   staletime = time(NULL) - STALE;

   if (s.st_atime < staletime ||
      s.st_ctime < staletime || s.st_mtime < staletime) {
      close(cfd);
      return -1;
   }

   if (uidptr != NULL)
      *uidptr = s.st_uid;

   unlink(un.sun_path);

   return cfd;
}

static int
MSMBinderGetFD(int fd)
{
   int newfd = -1, nr, status;

   char *ptr;
   int *iptr;

   char buf[MAXLINE];

   struct iovec iov[1];

   struct msghdr msg;

   status = -1;
   for (;;) {
      iov[0].iov_base = buf;
      iov[0].iov_len = sizeof(buf);
      msg.msg_iov = iov;
      msg.msg_iovlen = 1;
      msg.msg_name = NULL;
      msg.msg_namelen = 0;
      if (cmptr == NULL && (cmptr = malloc(CONTROLLEN)) == NULL)
	 return (-1);
      msg.msg_control = cmptr;
      msg.msg_controllen = CONTROLLEN;
      if ((nr = recvmsg(fd, &msg, 0)) < 0) {
	 ErrorF("%s: recvmsg() %m\n", __FUNCTION__);
      } else if (nr == 0) {
	 return (-1);
      }
      for (ptr = buf; ptr < &buf[nr];) {
	 if (*ptr++ == 0) {
	    if (ptr != &buf[nr - 1])
	       ErrorF("%s: message format error\n", __FUNCTION__);
	    status = *ptr & 0xFF;
	    if (status == 0) {
	       if (msg.msg_controllen != CONTROLLEN)
		  ErrorF("%s: status = 0 but no fd\n", __FUNCTION__);
	       iptr = (int *) CMSG_DATA(cmptr);
	       newfd = *iptr;
	    } else {
	       newfd = -status;
	    }
	    nr -= 2;
	 }
      }
      if (status >= 0) {
	 return (newfd);
      }
   }
}

void *
MSMBinderProc(void *arg)
{
   MSMPtr pMsm = (MSMPtr) arg;

   int fd = MSMBinderSocket("/var/tmp/hwsocket", pMsm->socketGID);

   if (fd == -1) {
      ErrorF("%s: Unable to start the binder thread.\n", __FUNCTION__);
      return NULL;
   }

   while (1) {
      int child;

      int pfd;

      child = MSMBinderAccept(fd, tptr);

      if (child == -1)
	 continue;

      pfd = MSMBinderGetFD(child);

      /* FIXME: Probably need a mutex here */

      pMsm->pfd = pfd;

      close(child);
   }
}

void
MSMBinderInit(MSMPtr pMsm)
{
   pthread_attr_init(&attr);
   pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);

   if (pthread_create(&binderid, &attr, MSMBinderProc, pMsm))
      ErrorF("%s: Unable to create the binder thread\n", __FUNCTION__);
}
