#define _ISOC99_SOURCE
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <gtk/gtk.h>
#include "rk.h"

#define NPARAM 2

/* Backing pixmap for drawing area */
static GdkPixmap *pixmap = NULL;
static GtkWidget *drawing_area;
static GtkWidget *params[NPARAM], *zoom_param;
static GtkAdjustment *adj[NPARAM], *zoom_adj;
static double dir_angle; /* origin is x0, y0 */
double values[NPARAM];
double zoom=1.0;

static void draw_init(void);
static void draw_main(GdkPixmap *pixmap, GdkGC *gc, int width, int height);

static void update(void)
{
  GtkWidget *widget;
  GdkGC *gc;
  int width, height;
  int i;

  if (!pixmap) return;
  widget=drawing_area;
  width=widget->allocation.width;
  height=widget->allocation.height;
  gc=widget->style->black_gc;
  for (i=0; i<NPARAM; i++)
    values[i]=adj[i]->value;
  zoom=exp2(-zoom_adj->value);
  gdk_draw_rectangle (pixmap,
		      widget->style->white_gc,
		      TRUE,
		      0, 0,
		      widget->allocation.width,
		      widget->allocation.height);
  draw_main(pixmap, gc, width, height);
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  pixmap, 0, 0, 0, 0, width, height);
}

/* Create a new backing pixmap of the appropriate size */
static gint configure_event( GtkWidget         *widget,
                             GdkEventConfigure *event )
{
  if (pixmap)
    gdk_pixmap_unref(pixmap);

  pixmap = gdk_pixmap_new(widget->window,
			  widget->allocation.width,
			  widget->allocation.height,
			  -1);
  update();
  return TRUE;
}

/* Redraw the screen from the backing pixmap */
static gint expose_event( GtkWidget      *widget,
                          GdkEventExpose *event )
{
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);

  return FALSE;
}

void quit ()
{
  gtk_exit (0);
}

int main( int   argc, 
          char *argv[] )
{
  GtkWidget *window;
  GtkWidget *vbox;

  int i;

  gtk_init (&argc, &argv);

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_name (window, "Test Input");

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_widget_show (vbox);

  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (quit), NULL);

  /* Create the drawing area */

  drawing_area = gtk_drawing_area_new ();
  gtk_drawing_area_size (GTK_DRAWING_AREA (drawing_area), 200, 200);
  gtk_box_pack_start (GTK_BOX (vbox), drawing_area, TRUE, TRUE, 0);

  gtk_widget_show (drawing_area);

  /* Signals used to handle backing pixmap */

  gtk_signal_connect (GTK_OBJECT (drawing_area), "expose_event",
		      (GtkSignalFunc) expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT(drawing_area),"configure_event",
		      (GtkSignalFunc) configure_event, NULL);

  /* Event signals */

  gtk_widget_set_events (drawing_area, GDK_EXPOSURE_MASK);

  /* Parameters */  
  for (i=0; i<NPARAM; i++)
    {
      adj[i]=GTK_ADJUSTMENT(gtk_adjustment_new(0, -0.5, 0.5, 0.01, 0.05, 0.0));
      params[i]=gtk_hscale_new(adj[i]);
      gtk_scale_set_digits(GTK_SCALE(params[i]), 2);
      gtk_signal_connect(GTK_OBJECT(adj[i]), "value_changed", GTK_SIGNAL_FUNC(update), NULL);
      gtk_range_set_update_policy(GTK_RANGE(params[i]), GTK_UPDATE_CONTINUOUS);
      gtk_box_pack_start(GTK_BOX(vbox), params[i], FALSE, FALSE, 0);
      gtk_widget_show(params[i]);
    }

  zoom_adj=GTK_ADJUSTMENT(gtk_adjustment_new(0, -1.0, 4.0, 0.1, 0.5, 0.0));
  zoom_param=gtk_hscale_new(zoom_adj);
  gtk_scale_set_digits(GTK_SCALE(zoom_param), 2);
  gtk_signal_connect(GTK_OBJECT(zoom_adj), "value_changed", GTK_SIGNAL_FUNC(update), NULL);
  gtk_range_set_update_policy(GTK_RANGE(zoom_param), GTK_UPDATE_CONTINUOUS);
  gtk_box_pack_start(GTK_BOX(vbox), zoom_param, FALSE, FALSE, 0);
  gtk_widget_show(zoom_param);
  
  gtk_widget_show (window);

  gtk_main ();

  return 0;
}

#define MINDIV 2
#define MAXCOUNT 50
#define XDIV 32
#define YDIV 32
#define XMIN (-zoom)
#define YMIN (-zoom)
#define XMAX (zoom)
#define YMAX (zoom)
#define XSTEP ((XMAX-XMIN)/XDIV)
#define YSTEP ((YMAX-YMIN)/YDIV)
#define XDEN (1.0/XSTEP)
#define YDEN (1.0/YSTEP)
#define ARROW_LEN 8
#define ARROW_ANGLE 0.3
#define MIN_STEP 5 /* minimum steps to draw arrows */
char pr[XDIV][YDIV]; /* footprint */
char xr[XDIV][YDIV], nr[XDIV][YDIV]; /* used by get_point() */

static inline int transx(double x)
{
  return lrint((x-XMIN)*XDEN);
}

static inline int transy(double y)
{
  return lrint((y-YMIN)*YDEN);
}

/* returns 0 for success, -1 for out-of-bound or already-gone-to (x,y) */
static inline int footprint(int px, int py)
{
  if (px>=0 && px<XDIV && py>=0 && py<YDIV && !pr[px][py])
    {
      pr[px][py]=1;
      return 0;
    }
  else return -1;
}

/* returns 0 for success, -1 for fail */
static int get_point(double *x, double *y)
{
  int div, i, j;
  int mdiv, mi=0, mj=0;
  int found;
  
  mdiv=0;
  div=1;
  memcpy(xr, pr, sizeof(pr));
  while (1)
    {
      if (div>XDIV) break;
      found=0;
      for (i=0; i<=XDIV-div; i++)
	for (j=0; j<=YDIV-div; j++)
	  if (xr[i][j]==0)
	    {
	      found=1;
	      mdiv=div;
	      mi=i;
	      mj=j;
	      break;
	    }
      if (!found) break;
      for (i=0; i<XDIV-div; i++)
	for (j=0; j<YDIV-div; j++)
	  nr[i][j]=(xr[i][j]+xr[i][j+1]+xr[i+1][j]+xr[i+1][j+1]) ? 1 : 0;
      memcpy(xr, nr, sizeof(nr));
      div++;
    }
  if (mdiv>=MINDIV)
    {
      *x=XMIN+(mi+(mdiv-1)*0.5)*XSTEP;
      *y=YMIN+(mj+(mdiv-1)*0.5)*YSTEP;
      return 0;
    }
  else return -1;
}

static void draw_init(void)
{
  int i, j;
  for (i=0; i<XDIV; i++)
    for (j=0; j<YDIV; j++)
      pr[i][j]=0;
}

static int do_loop(GdkPixmap *pixmap, GdkGC *gc, int width, int height, double x0, double y0, double dt)
{
  int i, count=0, steps=0;
  int px, py;
  double x, y, lx, ly;
  double xa[5], ya[5];
  double gxden, gyden;

  x=x0;
  y=y0;
  gxden=width/(XMAX-XMIN);
  gyden=height/(YMAX-YMIN);
  while (1)
    {
      steps++;
      px=transx(x);
      py=transy(y);
      if (footprint(px, py)<0 && steps>1) break;
      lx=x;
      ly=y;
      count=0;
      while (count<MAXCOUNT)
	{
	  count++;
	  xa[0]=x;
	  ya[0]=y;
	  for (i=1; i<=4; i++)
	    {
	      xa[i]=xa[0]+dx(xa[i-1], ya[i-1])*dt;
	      ya[i]=ya[0]+dy(xa[i-1], ya[i-1])*dt;
	    }
	  x=(xa[1]+2.0*xa[2]+xa[3]-xa[4])*(1/3.0);
	  y=(ya[1]+2.0*ya[2]+ya[3]-ya[4])*(1/3.0);
	  if (transx(x)!=px) break;
	  if (transy(y)!=py) break;
	}
      if (dt>0 && steps==1)
	dir_angle=atan2(y-y0, x-x0);
      gdk_draw_line(pixmap, gc,
		    lrint((lx-XMIN)*gxden), lrint((ly-YMIN)*gyden),
		    lrint((x-XMIN)*gxden), lrint((y-YMIN)*gyden));
    }
  return steps;
}

static void draw_main(GdkPixmap *pixmap, GdkGC *gc, int width, int height)
{
  double x0, y0, dt;
  int px0, py0, px1, py1, px2, py2, px3, py3;
  int step1, step2;

  dt=0.05;
  x0=0;
  y0=0;
  draw_init();
  do {
      step1=do_loop(pixmap, gc, width, height, x0, y0, dt);
      step2=do_loop(pixmap, gc, width, height, x0, y0, -dt);
      if (step1>=MIN_STEP && step2>=MIN_STEP) /* draw an arrow */
	{
	  px0=lrint((x0-XMIN)/(XMAX-XMIN)*width);
	  py0=lrint((y0-YMIN)/(YMAX-YMIN)*height);
	  px1=px0+lrint(ARROW_LEN*cos(dir_angle));
	  py1=py0+lrint(ARROW_LEN*sin(dir_angle));	  
	  px2=px1-lrint(ARROW_LEN*cos(dir_angle-ARROW_ANGLE));
	  py2=py1-lrint(ARROW_LEN*sin(dir_angle-ARROW_ANGLE));
	  px3=px1-lrint(ARROW_LEN*cos(dir_angle+ARROW_ANGLE));
	  py3=py1-lrint(ARROW_LEN*sin(dir_angle+ARROW_ANGLE));
	  gdk_draw_line(pixmap, gc, px1, py1, px2, py2);
	  gdk_draw_line(pixmap, gc, px1, py1, px3, py3);
	}
    }
  while (get_point(&x0, &y0)>=0);
}





