/*
*  RAL -- Rubrica Addressbook Library
*  file: work.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include <glib.h>
#include <glib-object.h>

#include "work.h"
#include "utils.h"


enum {
  WORK_ASSIGNMENT = 1,
  WORK_ORG,
  WORK_DEP,
  WORK_SUBDEP,
  WORK_MANAGER,
  WORK_MPHONE,
  WORK_COLLABORATOR,
  WORK_CPHONE
};

struct _RWorkPrivate {
  gchar* assignment;            /* contact role             */
  gchar* org;                   /* organization             */
  gchar* dep;                   /* department               */
  gchar* subdep;                /* sub department           */
  gchar* manager;               /* contact's manager        */
  gchar* mphone;                /* manager telephone        */
  gchar* collaborator;          /* contact's collaborator   */
  gchar* cphone;                /* collaborator's telephone */

  gboolean dispose_has_run;
};

#define R_WORK_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE((o),  \
			           R_WORK_TYPE, RWorkPrivate))


static void r_work_class_init   (RWorkClass* klass);
static void r_work_init         (RWork* obj);

static void r_work_dispose      (RWork* obj);
static void r_work_finalize     (RWork* obj);


static void r_work_set_property (GObject* obj, guint property_id,
				 GValue* value, GParamSpec* spec);
static void r_work_get_property (GObject* obj, guint property_id,
				 GValue* value, GParamSpec* spec);


GType
r_work_get_type()
{
  static GType work_type = 0;
  
  if (!work_type)
    {
      static const GTypeInfo work_info =
	{
	  sizeof(RWorkClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_work_class_init,
	  NULL,
	  NULL,
	  sizeof(RWork),
	  0,
	  (GInstanceInitFunc) r_work_init
	};

      work_type = g_type_register_static (G_TYPE_OBJECT, "RWork",
					  &work_info, 0);
    }
  
  return work_type;
}


static void
r_work_class_init(RWorkClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  class  = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) r_work_dispose;
  class->finalize = (GObjectFinalizeFunc) r_work_finalize;
  
  class->set_property = (gpointer) r_work_set_property;
  class->get_property = (gpointer) r_work_get_property;

  g_type_class_add_private (klass, sizeof(RWorkPrivate));

  /**
   * RWork:assignment
   *
   * the contact's assignment
   */
  pspec = g_param_spec_string("assignment",
			      "the contact's assignment",
			      "the contact's assignment",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, WORK_ASSIGNMENT, pspec);

  /**
   * RWork:organization:
   *
   * The organization the contact works for
   */
  pspec = g_param_spec_string("organization",
			      "organization",
			      "The organization the contact works for",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, WORK_ORG, pspec);

  /**
   * RWork:department:
   *
   * The department where contact works
   */
  pspec = g_param_spec_string("department",
			      "department",
			      "The department where contact works",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, WORK_DEP, pspec);

  /**
   * RWork:sub-department:
   *
   * The sub department where contact works
   */
  pspec = g_param_spec_string("sub-department",
			      "sub department",
			      "The sub department where contact works",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, WORK_SUBDEP, pspec);

  /**
   * RWork:manager-name:
   *
   * the contact's manager
   */
  pspec = g_param_spec_string("manager-name",
			      "contact's manager",
			      "the contact's manager ",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, WORK_MANAGER, pspec);

  /**
   * RWork:manager-phone:
   *
   * the manager's telephone number
   */
  pspec = g_param_spec_string("manager-phone",
			      "manager's phone",
			      "the manager's telephone number",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, WORK_MPHONE, pspec);

  /**
   * RWork:collaborator:
   *
   * contact's collaborator
   */
  pspec = g_param_spec_string("collaborator",
			      "contact's collaborator",
			      "contact's collaborator",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, WORK_COLLABORATOR, pspec);

  /**
   * RWork:collaborator-phone:
   * 
   * collaborator's phone
   */
  pspec = g_param_spec_string("collaborator-phone",
			      "collaborator's phone",
			      "collaborator's phone",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, WORK_CPHONE, pspec);
}


static void
r_work_init(RWork* self)
{
  self->priv = R_WORK_GET_PRIVATE(self);

  self->priv->assignment   = NULL;
  self->priv->org          = NULL;
  self->priv->dep          = NULL;
  self->priv->subdep       = NULL;
  self->priv->manager      = NULL;
  self->priv->mphone       = NULL;
  self->priv->collaborator = NULL;
  self->priv->cphone       = NULL;
  
  self->priv->dispose_has_run = FALSE;
}



static void 
r_work_dispose (RWork* self)
{
  g_return_if_fail(IS_R_WORK(self));

  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;
}


static void 
r_work_finalize (RWork* self)
{
  g_return_if_fail(IS_R_WORK(self));

  r_utils_free_string(self->priv->assignment);
  r_utils_free_string(self->priv->org);
  r_utils_free_string(self->priv->dep);
  r_utils_free_string(self->priv->subdep);
  r_utils_free_string(self->priv->manager);
  r_utils_free_string(self->priv->mphone);
  r_utils_free_string(self->priv->collaborator);
  r_utils_free_string(self->priv->cphone);

  self->priv->assignment   = NULL;
  self->priv->org          = NULL;
  self->priv->dep          = NULL;
  self->priv->subdep       = NULL;
  self->priv->manager      = NULL;
  self->priv->mphone       = NULL;
  self->priv->collaborator = NULL;
  self->priv->cphone       = NULL;  
}


static void 
r_work_set_property (GObject* obj, guint property_id,
		     GValue* value, GParamSpec* spec)
{
  RWork* self = (RWork*) obj;
  RWorkPrivate* priv = R_WORK_GET_PRIVATE(self);
 
  switch (property_id)
    {
    case WORK_ASSIGNMENT:
      g_free(priv->assignment);
      priv->assignment = g_value_dup_string(value);
      break;

    case WORK_ORG:
      g_free(priv->org);
      priv->org = g_value_dup_string(value);
      break;

    case WORK_DEP:
      g_free(priv->dep);
      priv->dep = g_value_dup_string(value);
      break;

    case WORK_SUBDEP:
      g_free(priv->subdep);
      priv->subdep = g_value_dup_string(value);
      break;

    case WORK_MANAGER:
      g_free(priv->manager);
      priv->manager = g_value_dup_string(value);
      break;

    case WORK_MPHONE:
      g_free(priv->mphone);
      priv->mphone = g_value_dup_string(value);
      break;

    case WORK_COLLABORATOR:
      g_free(priv->collaborator);
      priv->collaborator = g_value_dup_string(value);
      break;

    case WORK_CPHONE:
      g_free(priv->cphone);
      priv->cphone = g_value_dup_string(value);
      break;

    default:
      break;
    }
}


static void 
r_work_get_property (GObject* obj, guint property_id,
		     GValue* value, GParamSpec* spec)
{
  RWork* self = (RWork*) obj;
  RWorkPrivate* priv = R_WORK_GET_PRIVATE(self);
  
  switch (property_id)
    {
    case WORK_ASSIGNMENT:
      g_value_set_string(value, priv->assignment);
      break;

    case WORK_ORG:
      g_value_set_string(value, priv->org);
      break;

    case WORK_DEP:
      g_value_set_string(value, priv->dep);
      break;

    case WORK_SUBDEP:
      g_value_set_string(value, priv->subdep);
      break;

    case WORK_MANAGER:
      g_value_set_string(value, priv->manager);
      break;

    case WORK_MPHONE:
      g_value_set_string(value, priv->mphone);
      break;

    case WORK_COLLABORATOR:
      g_value_set_string(value, priv->collaborator);
      break;

    case WORK_CPHONE:
      g_value_set_string(value, priv->cphone);
      break;

    default:
      break;
    }
}


/*   Public
*/


/**
 * r_work_new
 *
 * create a new #RWork
 *
 * returns: a new #RWork*
 */
RWork*
r_work_new(void)
{
  RWork* work;

  work = g_object_new(r_work_get_type(), NULL);

  return work;
}

/**
 * r_work_free
 * @work: a #RWork
 *
 * free the memory owned by object
 */
void
r_work_free(RWork* work)
{
  g_return_if_fail(IS_R_WORK(work));

  g_object_unref(work);   
}


gboolean 
r_work_have_data (RWork* work)
{
  RWorkPrivate* priv;
  
  g_return_val_if_fail(IS_R_WORK(work), FALSE);
  
  priv = R_WORK_GET_PRIVATE(work);
  
  if (priv->assignment || priv->org || priv->dep || priv->subdep ||
      priv->manager || priv->mphone || priv->collaborator || priv->cphone)
    return TRUE;

  return FALSE;
}


gboolean 
r_work_search (RWork* work, const gchar* str)
{
  RWorkPrivate* priv;

  g_return_val_if_fail(IS_R_WORK(work), FALSE);
  g_return_val_if_fail(str != NULL, FALSE);

  priv = R_WORK_GET_PRIVATE(work);

  if (priv->assignment && g_strrstr(priv->assignment, str))
    return TRUE;

  if (priv->org && g_strrstr(priv->org, str))
    return TRUE;

  if (priv->dep && g_strrstr(priv->dep, str))
    return TRUE;

  if (priv->subdep && g_strrstr(priv->subdep, str))
    return TRUE;

  if (priv->manager && g_strrstr(priv->manager, str))
    return TRUE;

  if (priv->mphone && g_strrstr(priv->mphone, str))
    return TRUE;

  if (priv->collaborator && g_strrstr(priv->collaborator, str))
    return TRUE;

  if (priv->cphone && g_strrstr(priv->cphone, str))
    return TRUE;

  return FALSE;
}




/**
 * r_work_copy
 * @work: a #RWork
 *
 * copy the given #RWork
 *
 * returns: a new allocated #RWork* 
 */
RWork* 
r_work_copy (RWork* work)
{
  RWork* new;
  gchar *assignment, *org, *dep, *subdep;
  gchar *manager, *mphone, *collaborator, *cphone;

  g_return_val_if_fail(IS_R_WORK(work), NULL);
  
  new = r_work_new();

  g_object_get(G_OBJECT(work), "assignment", &assignment, 
	       "organization", &org, "department", &dep, 
	       "sub-department", &subdep, "manager-name", &manager, 
	       "manager-phone", &mphone, "collaborator", &collaborator, 
	       "collaborator-phone", &cphone, NULL);

  g_object_set(G_OBJECT(new), "assignment", assignment, 
	       "organization", org, "department", dep, 
	       "sub-department", subdep, "manager-name", manager, 
	       "manager-phone", mphone, "collaborator", collaborator, 
	       "collaborator-phone", cphone, NULL);

  return new;
}

