/* read_dna.c       2020-05-02 */

/* Copyright 2013-2020 Emmanuel Paradis */

/* This file is part of the R-package `ape'. */
/* See the file ../COPYING for licensing issues. */

#include <R.h>
#include <Rinternals.h>

/* translation table CHAR -> DNAbin */
static unsigned char tab_trans[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0-9 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 10-19 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 20-29 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 30-39 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, /* 40-49 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 50-59 */
	0x00, 0x00, 0x00, 0x02, 0x00, 0x88, 0x70, 0x28, 0xd0, 0x00, /* 60-69 */
	0x00, 0x48, 0xb0, 0x00, 0x00, 0x50, 0x00, 0xa0, 0xf0, 0x00, /* 70-79 */
	0x00, 0x00, 0xc0, 0x60, 0x18, 0x00, 0xe0, 0x90, 0x00, 0x30, /* 80-89 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x70, 0x28, /* 90-99 */
	0xd0, 0x00, 0x00, 0x48, 0xb0, 0x00, 0x00, 0x50, 0x00, 0xa0, /* 100-109 */
	0xf0, 0x00, 0x00, 0x00, 0xc0, 0x60, 0x18, 0x00, 0xe0, 0x90, /* 110-119 */
	0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 120-129 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 130-139 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 140-149 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 150-159 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 160-169 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 170-179 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 180-189 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 190-199 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 200-209 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 210-219 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 220-229 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 230-239 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 240-249 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00}; /* 250-255 */

/* translation table DNAbin -> CHAR */
static const unsigned char tab_trans_rev[] = {
	0x00, 0x00, 0x3f, 0x00, 0x2d, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0-9 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 10-19 */
	0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, /* 20-29 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 30-39 */
	0x43, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x00, /* 40-49 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 50-59 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 60-69 */
	0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 70-79 */
	0x4b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 80-89 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x00, 0x00, 0x00, /* 90-99 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 100-109 */
	0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 110-119 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 120-129 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, /* 130-139 */
	0x00, 0x00, 0x00, 0x00, 0x57, 0x00, 0x00, 0x00, 0x00, 0x00, /* 140-149 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 150-159 */
	0x4d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 160-169 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, /* 170-179 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 180-189 */
	0x00, 0x00, 0x52, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 190-199 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x00, /* 200-209 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 210-219 */
	0x00, 0x00, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00, 0x00, 0x00, /* 220-229 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 230-239 */
	0x4e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 240-249 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00}; /* 250-255 */

/* translation table CHAR -> AAbin */
static unsigned char tab_trans_aminoacid[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0-9 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 10-19 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 20-29 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 30-39 */
	0x00, 0x00, 0x2a, 0x00, 0x00, 0x2d, 0x00, 0x00, 0x00, 0x00, /* 40-49 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 50-59 */
	0x00, 0x00, 0x00, 0x3f, 0x00, 0x41, 0x41, 0x43, 0x44, 0x45, /* 60-69 */
	0x46, 0x47, 0x48, 0x49, 0x00, 0x4b, 0x4c, 0x4d, 0x4e, 0x00, /* 70-79 */
	0x50, 0x51, 0x52, 0x53, 0x54, 0x00, 0x56, 0x57, 0x58, 0x59, /* 80-89 */
	0x5a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x61, 0x62, 0x63, /* 90-99 */
	0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x00, 0x6b, 0x6c, 0x6d, /* 100-109 */
	0x6e, 0x00, 0x70, 0x71, 0x72, 0x73, 0x74, 0x00, 0x76, 0x77, /* 110-119 */
	0x78, 0x79, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 120-129 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 130-139 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 140-149 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 150-159 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 160-169 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 170-179 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 180-189 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 190-199 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 200-209 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 210-219 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 220-229 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 230-239 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 240-249 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00}; /* 250-255 */

static const unsigned char hook = 0x3e;
static const unsigned char lineFeed = 0x0a;
/* static const unsigned char space = 0x20; */

SEXP rawStreamToDNAorAAbin(SEXP x, SEXP DNA)
{
	int k, startOfSeq;
	long i, j, n;
	unsigned char *xr, *rseq, *buffer, tmp, *TAB_TRANS;
	SEXP obj, nms, seq;

	PROTECT(x = coerceVector(x, RAWSXP));
	PROTECT(DNA = coerceVector(DNA, INTSXP));
	if (INTEGER(DNA)[0]) TAB_TRANS = tab_trans;
	else TAB_TRANS = tab_trans_aminoacid;

	double N = XLENGTH(x);
	xr = RAW(x);

/* do a 1st pass to find the number of sequences

   this code should be robust to '>' present inside
   a label or in the header text before the sequences */

	n = 0;
	k = 0; /* use k as a flag */
	if (xr[0] == hook) {
		k = 1;
		startOfSeq = 0;
	}
	for (i = 1; i < N; i++) {
		if (k && xr[i] == lineFeed) {
			n++;
			k = 0;
		} else if (xr[i] == hook) {
			if (!n) startOfSeq = i;
			k = 1;
		}
	}

	if (n == 0) {
	    PROTECT(obj = allocVector(INTSXP, 1));
	    INTEGER(obj)[0] = 0;
	    UNPROTECT(3);
	    return obj;
	}

	PROTECT(obj = allocVector(VECSXP, n));
	PROTECT(nms = allocVector(STRSXP, n));

/* Refine the way the size of the buffer is set? */
	buffer = (unsigned char *)R_alloc(N, sizeof(unsigned char));

	i = (long) startOfSeq;
	j = 0; /* gives the index of the sequence */
	while (i < N) {
		/* 1st read the label... */
		i++;
		k = 0;
		while (xr[i] != lineFeed) buffer[k++] = xr[i++];
		buffer[k] = '\0';
		SET_STRING_ELT(nms, j, mkChar((char *)buffer));
		/* ... then read the sequence */
		n = 0;
		while (i < N && xr[i] != hook) {
			tmp = TAB_TRANS[xr[i++]];
/* If we are sure that the FASTA file is correct (ie, the sequence on
   a single line and only the IUAPC code (plus '-' and '?') is used,
   then the following check would not be needed; additionally the size
   of tab_trans could be restriced to 0-121. This check has the
   advantage that all invalid characters are simply ignored without
   causing error -- except if '>' occurs in the middle of a sequence. */
			if (tmp) buffer[n++] = tmp;
		}
		PROTECT(seq = allocVector(RAWSXP, n));
		rseq = RAW(seq);
		for (k = 0; k < n; k++) rseq[k] = buffer[k];
		SET_VECTOR_ELT(obj, j, seq);
		UNPROTECT(1);
		j++;
	}
	setAttrib(obj, R_NamesSymbol, nms);
	UNPROTECT(4);
	return obj;
}

static const int BUFF = 1e9;

#define WRITELABELS\
    o[0] = hook; /* start with ">" */\
    p = RAW(VECTOR_ELT(labels, i));  \
    nchr = LENGTH(VECTOR_ELT(labels, i)); \
    for (k = 1, w = 0; w < nchr; k++, w++) o[k] = p[w]; \
    o[k++] = lineFeed; \
    fwrite(o, 1, k, fl)

SEXP writeDNAbinToFASTA(SEXP x, SEXP FILENAME, SEXP n, SEXP s, SEXP labels)
{
    int i, w, k, nchr;
    const char *filename;
    FILE *fl;
    unsigned char *p, *px, *o;

    /* IMPORTANT: two distinct pointers *p and *px must be used,
       otherwise, this does not work correctly */

    PROTECT(s = coerceVector(s, INTSXP));
    int S = INTEGER(s)[0];

    if (S != -1) /* x is a matrix */
	PROTECT(x = coerceVector(x, RAWSXP));
    else /* x is a list */
	PROTECT(x = coerceVector(x, VECSXP));

    PROTECT(labels = coerceVector(labels, VECSXP));
    PROTECT(FILENAME = coerceVector(FILENAME, STRSXP));
    PROTECT(n = coerceVector(n, INTSXP));

    int nseq = INTEGER(n)[0];
    filename = CHAR(STRING_ELT(FILENAME, 0));
    fl = fopen(filename, "a+");

    o = (unsigned char*)R_alloc(BUFF, sizeof(unsigned char)); /* the output stream */

    SEXP res;
    PROTECT(res = allocVector(INTSXP, 1));
    INTEGER(res)[0] = 0;

    if (S != -1) { /* x is a matrix */
	px = RAW(x);
	for (i = 0; i < nseq; i++) {
	    WRITELABELS;
	    w = i;
	    k = 0;
	    while (k < S) {
		o[k++] = tab_trans_rev[px[w]];
		w = w + nseq;
		//if (!((k + 1) % (COLW + 1))) o[k++] = lineFeed;
	    }
	    //if (o[k - 1] != 0x0a) o[k++] = lineFeed;
	    o[k++] = lineFeed;
	    fwrite(o, 1, k, fl);
	}
    } else { /* x is a list */
	for (i = 0; i < nseq; i++) {
	    WRITELABELS;
	    int seql = XLENGTH(VECTOR_ELT(x, i));
	    p = RAW(VECTOR_ELT(x, i));
	    /* w: position where to start copy the bases to the output stream
	       k: position in the output stream */
	    for (k = 0, w = 0; w < seql; w++)
		o[k++] = tab_trans_rev[p[w]];
		//if (!((k + 1) % (COLW + 1))) o[k++] = lineFeed;
		//}
		//if (o[k - 1] != 0x0a) o[k++] = lineFeed;
	    o[k++] = lineFeed;
	    fwrite(o, 1, k, fl);
	}
    }

    fclose(fl);
    UNPROTECT(6);
    return res;
}

SEXP writeAAbinToFASTA(SEXP x, SEXP FILENAME, SEXP n, SEXP s, SEXP labels)
{
    int i, w, k, nchr;
    const char *filename;
    FILE *fl;
    unsigned char *p, *px, *o;

    /* IMPORTANT: two distinct pointers *p and *px must be used,
       otherwise, this does not work correctly */

    PROTECT(s = coerceVector(s, INTSXP));
    int S = INTEGER(s)[0];

    if (S != -1) /* x is a matrix */
	PROTECT(x = coerceVector(x, RAWSXP));
    else /* x is a list */
	PROTECT(x = coerceVector(x, VECSXP));

    PROTECT(labels = coerceVector(labels, VECSXP));
    PROTECT(FILENAME = coerceVector(FILENAME, STRSXP));
    PROTECT(n = coerceVector(n, INTSXP));

    int nseq = INTEGER(n)[0];
    filename = CHAR(STRING_ELT(FILENAME, 0));
    fl = fopen(filename, "a+");

    o = (unsigned char*)R_alloc(BUFF, sizeof(unsigned char)); /* the output stream */

    SEXP res;
    PROTECT(res = allocVector(INTSXP, 1));
    INTEGER(res)[0] = 0;

    if (S != -1) { /* x is a matrix */
	px = RAW(x);
	for (i = 0; i < nseq; i++) {
	    WRITELABELS;
	    w = i;
	    k = 0;
	    while (k < S) {
		o[k++] = px[w];
		w = w + nseq;
	    }
	    o[k++] = lineFeed;
	    fwrite(o, 1, k, fl);
	}
    } else { /* x is a list */
	for (i = 0; i < nseq; i++) {
	    WRITELABELS;
	    int seql = XLENGTH(VECTOR_ELT(x, i));
	    p = RAW(VECTOR_ELT(x, i));
	    /* w: position where to start copy the bases to the output stream
	       k: position in the output stream */
	    for (k = 0, w = 0; w < seql; w++)
		o[k++] = p[w];
	    o[k++] = lineFeed;
	    fwrite(o, 1, k, fl);
	}
    }

    fclose(fl);
    UNPROTECT(6);
    return res;
}

#undef WRITELABELS

SEXP charVectorToDNAbinVector(SEXP x)
{
    SEXP res;
    const char *xr;
    unsigned char *rs;

    PROTECT(x = coerceVector(x, STRSXP));
    xr = CHAR(STRING_ELT(x, 0));		\
    int n = strlen(xr);
    PROTECT(res = allocVector(RAWSXP, n));
    rs = RAW(res);

    for (long i = 0; i < n; i++) rs[i] = tab_trans[(unsigned char) xr[i]];

    UNPROTECT(2);
    return res;
}
