///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file FindGrains.h
 * \brief Contains the definition of the CrystalAnalysis::FindGrains class.
 */

#ifndef __FIND_GRAINS_H
#define __FIND_GRAINS_H

#include <crystalanalysis/CrystalAnalysis.h>
#include <atomviz/atoms/AtomsObject.h>

namespace CrystalAnalysis {

/**
 * \brief This helper class is used by the AnalyzeMicrostructureModifier to
 *        decompose the microstructure into individual grains.
 *
 * \author Alexander Stukowski
 */
class CRYSTALANALYSIS_DLLEXPORT FindGrains
{
public:

	/**
	 * \brief This structures stores information about a single grain that has been extracted by the algorithm.
	 */
	class GrainInfo
	{
	public:

		/// The constructor for a new grain.
		GrainInfo(int id = -1) : _atomCount(0), _crystallineCount(0), _id(id) {
			// Generate a random color for this grain.
			FloatType hue = (FloatType)rand() / (FloatType)RAND_MAX;
			FloatType saturation = 1.0 - 0.3 * (FloatType)rand() / (FloatType)RAND_MAX;
			FloatType value = 1.0 - 0.2 * (FloatType)rand() / (FloatType)RAND_MAX;
			_grainColor = Color::fromHSV(hue, saturation, value);
		}

		/// Returns the unique identifier of the grain.
		int id() const { return _id; }

		/// Returns the total number of atoms that belong to the grain.
		int atomCount() const { return _atomCount; }

		/// Returns the number of crystalline atoms that belong to the grain.
		int crystallineCount() const { return _crystallineCount; }

		/// Returns the color that has randomly be assigned to this grain.
		const Color& color() const { return _grainColor; }

		/// Returns the average lattice orientation of the grain.
		const Tensor2& averageOrientation() const { return _averageOrientation; }

		/// Compares the grains with respect to their size (number of atoms).
		bool operator<(const GrainInfo& other) const { return atomCount() < other.atomCount(); }

	private:

		/// The unique identifier of the grain.
		int _id;

		/// The total number of atoms that belong to the grain.
		int _atomCount;

		/// The number of crystalline atoms that belong to the grain.
		int _crystallineCount;

		/// The random color assigned to this grain.
		Color _grainColor;

		/// The average lattice orientation of the grain.
		Tensor2 _averageOrientation;

		friend class FindGrains;
	};

	/**
	 * \brief This structures stores information about a single grain boundary.
	 */
	class GrainBoundaryInfo
	{
	public:

		/// The constructor for a new grain boundary.
		GrainBoundaryInfo(int id = -1) : _id(id), _numBonds(0) {}

		/// Returns the unique identifier of the grain boundary.
		int id() const { return _id; }

		/// Returns the identifier of the first grain.
		int grainA() const { return _grains[0]; }

		/// Returns the identifier of the second grain.
		int grainB() const { return _grains[1]; }

	private:

		/// The unique identifier of the grain boundary.
		int _id;

		/// The two grains.
		int _grains[2];

		/// The number of nearest-neighbor bonds this boundary consists of.
		int _numBonds;

		friend class FindGrains;
	};

public:

	/// \brief Default constructor.
	FindGrains();

	/// \brief Performs the grain cluster analysis.
	///
	/// \param input The input atoms.
	/// \param outputClusterChannel The output channel that will contain the grain cluster number for each atom.
	/// \param nearestNeighborCutoff The cutoff radius used to find nearest neighbor atoms.
	/// \param misorientationThreshold The grain misorientation threshold angle in radians.
	/// \param minCrystallineAtoms The minimum number of crystalline atoms that must belong
	///                            to a cluster to consider it a grain.
	/// \param misorientationChannel Optional output channel for atomic misorientation angles.
	///
	/// \throw Exception on error.
	/// \return \c false when the operation has been canceled by the user; \c true if the analysis has been completed.
	///
	/// The input AtomsObject must contain the DataChannel::DeformationGradientChannel.
	/// The information records for the identified grains can be accessed via the grains() method.
	bool performAnalysis(AtomsObject* input, DataChannel* outputClusterChannel, FloatType nearestNeighborCutoff,
			FloatType misorientationThreshold, int minCrystallineAtoms, DataChannel* misorientationChannel, bool suppressDialogs);

	/// \brief Returns the list of identified grains.
	const QVector<GrainInfo>& grains() const { return _grains; }

	/// \brief Returns the list of identified grain boundaries.
	const QVector<GrainBoundaryInfo>& grainBoundaries() const { return _grainBoundaries; }

	/// \brief Stores the grain information records in the given output stream.
	void saveToStream(ObjectSaveStream& stream);

	/// \brief Loads the grain information records from the given input stream.
	void loadFromStream(ObjectLoadStream& stream);

private:

	/// \brief Calculates the misorientation angle between two crystal orientations.
	///        Takes the symmetry of the cubic lattice into account.
	FloatType calculateMisorientation(const Tensor2& F1inverse, Tensor2& F2);

	/// The list of identified grains.
	QVector<GrainInfo> _grains;

	/// The list of identified grain boundaries.
	QVector<GrainBoundaryInfo> _grainBoundaries;

	/// The list of point group rotation for the cubic lattice.
	QVector<Tensor2> _pointGroupRotations;
};

};	// End of namespace CrystalAnalysis

#endif // __FIND_GRAINS_H
