///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file BezierShape.h
 * \brief Contains definition of Core::BezierShape class.
 */

#ifndef __OVITO_BEZIER_SHAPE_H
#define __OVITO_BEZIER_SHAPE_H

#include <core/Core.h>
#include "BezierCurve.h"

namespace Core {

/**
 * \brief A bezier shape is a collection of BezierCurve objects.
 *
 * \author Alexander Stukowski
 * \sa BezierCurve
 */
class CORE_DLLEXPORT BezierShape
{
public:

	/// \brief Creates an empty bezier shape with no curves.
	BezierShape() {}

	/// \brief Copy constructor.
	BezierShape(const BezierShape& shape) : _curves(shape._curves) {}

	/// \brief Returns the list of curves in this shape.
	/// \return A read-only list of curves.
	const QVector<BezierCurve>& curves() const { return _curves; }

	/// \brief Returns the list of curves in this shape.
	/// \return A mutable list of curves.
	QVector<BezierCurve>& curves() { return _curves; }

	/// \brief Computes the bounding box of the shape.
	/// \return The bounding box of the shape.
	Box3 boundingBox() const {
		Box3 bbox;
		for(QVector<BezierCurve>::const_iterator curve = curves().begin(); curve != curves().end(); ++curve)
			bbox.addBox(curve->boundingBox());
		return bbox;
	}

	/// \brief Saves the shape to the given output stream.
    /// \param stream The destination stream.
	/// \sa loadFromStream()
	void saveToStream(SaveStream& stream) const;

	/// \brief Loads the shape from the given input stream.
    /// \param stream The source stream.
	/// \sa saveToStream()
	void loadFromStream(LoadStream& stream);

private:

	/// The list of curves that make up the shape.
	QVector<BezierCurve> _curves;
};


};	// End of namespace Core

#endif // __OVITO_BEZIER_SHAPE_H
