///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <base/Base.h>
#include <base/io/LoadStream.h>
#include <base/utilities/Exception.h>

namespace Base {

/******************************************************************************
* Opens the stream for reading.
******************************************************************************/
LoadStream::LoadStream(QDataStream& source) : is(source), _isOpen(false)
{
	OVITO_ASSERT_MSG(!is.device()->isSequential(), "LoadStream constructor", "LoadStream class requires a seekable input stream.");
    if(is.device()->isSequential())
		throw Exception("LoadStream class requires a seekable input stream.");
    
	// Read file header. 

	_isOpen = true;

	// Check magic numbers.
	quint32 magic1, magic2;
	*this >> magic1 >> magic2;
	*this >> fileFormat;
	*this >> numFPBits;
	
	_isOpen = false;

	if(magic1 != 0x0FACC5AA || magic2 != 0x0AFCCA5A)
		throw Exception(tr("Unknown file format."));

	// Check file format version.
	if(fileFormat > OVITO_FILE_FORMAT_VERSION)
		throw Exception(tr("Unsupported file format version: %1. This file has been written with a newer program version. Please upgrade to the newest program version.").arg(fileFormat));

	is.setVersion(QDataStream::Qt_4_3);

	_isOpen = true;

	VerboseLogger() << "File format:" << endl;
	if(fileFormat >= 11) {
		// Read application name.
		*this >> applicationName;

		// Read application version.
		*this >> applicationMajorVersion;
		*this >> applicationMinorVersion;
		*this >> applicationRevisionVersion;

		VerboseLogger() << "  Application: " << applicationName << endl;
		VerboseLogger() << "  Application version: " << applicationMajorVersion << "." << applicationMinorVersion << "." << applicationRevisionVersion << endl;
	}
	VerboseLogger() << "  File format version: " << fileFormat << endl;
	VerboseLogger() << "  Floating-point precision: " << (numFPBits*8) << endl;
}

/******************************************************************************
* Closes the stream.
******************************************************************************/
void LoadStream::close()
{
	if(isOpen()) {
		_isOpen = false;
		if(!backpatchPointers.empty())
			throw Exception(tr("Deserialization error: Not all pointers in the input file have been resolved."));
	}
}

/******************************************************************************
* Loads an array of bytes from the input stream.
******************************************************************************/
void LoadStream::read(void* buffer, size_t numBytes)
{
	if(is.readRawData((char*)buffer, numBytes) != numBytes) {
		if(is.atEnd())
            throw Exception(tr("Unexpected end of file."));
		else
			throw Exception(tr("Failed to read data from input file."));
	}
	if(!chunks.empty()) {
		qint64 chunkEnd = chunks.top().second;
		OVITO_ASSERT_MSG(chunkEnd >= is.device()->pos(), "LoadStream::Read", "Tried to read past end of file chunk.");
		if(chunkEnd < is.device()->pos())
			throw Exception(tr("Invalid file format."));
	}
}

/******************************************************************************
* Opens the next chunk in the stream.
******************************************************************************/
quint32 LoadStream::openChunk()
{    	
	quint32 chunkId;
	quint32 chunkSize; 
	*this >> chunkId >> chunkSize;
	chunks.push(make_pair(chunkId, (qint64)chunkSize + is.device()->pos()));
	return chunkId;
}

/******************************************************************************
* Opens the next chunk and throws an exception if the id doesn't match.
******************************************************************************/
void LoadStream::expectChunk(quint32 chunkId)
{
	quint32 cid = openChunk();
	if(cid != chunkId) {
        Exception ex(tr("Invalid file structure. This error might be caused by old files that are no longer supported by the newer program version."));
        ex.appendDetailMessage(tr("Expected chunk ID %1 (0x%2) but found chunk ID %3 (0x%4).").arg(chunkId).arg(chunkId, 0, 16).arg(cid).arg(cid, 0, 16));
		throw ex;
	}
}

/******************************************************************************
* Opens the next chunk and throws an exception if the id is not in a given range.
******************************************************************************/
quint32 LoadStream::expectChunkRange(quint32 chunkBaseId, quint32 maxVersion)
{
	quint32 cid = openChunk();
	if(cid < chunkBaseId) {
        Exception ex(tr("Invalid file structure. This error might be caused by old files that are no longer supported by the newer program version."));
        ex.appendDetailMessage(tr("Expected chunk ID range %1-%2 (0x%3-0x%4) but found chunk ID %5 (0x%6).").arg(chunkBaseId).arg(chunkBaseId, 0, 16).arg(chunkBaseId+maxVersion).arg(chunkBaseId+maxVersion, 0, 16).arg(cid).arg(cid, 0, 16));
		throw ex;
	}
	else if(cid > chunkBaseId + maxVersion) {
        Exception ex(tr("Unexpected chunk ID. This error might be caused by files that have been written by a newer program version."));
        ex.appendDetailMessage(tr("Expected chunk ID range %1-%2 (0x%3-0x%4) but found chunk ID %5 (0x%6).").arg(chunkBaseId).arg(chunkBaseId, 0, 16).arg(chunkBaseId+maxVersion).arg(chunkBaseId+maxVersion, 0, 16).arg(cid).arg(cid, 0, 16));
		throw ex;
	}
	else return cid - chunkBaseId;
}

/******************************************************************************
* Closes the current chunk.
******************************************************************************/
void LoadStream::closeChunk()
{
	OVITO_ASSERT(!chunks.empty());
	qint64 chunkEnd = chunks.top().second;
	OVITO_ASSERT_MSG(chunkEnd >= is.device()->pos(), "LoadStream::closeChunk()", "Read past end of chunk.");
	if(is.device()->pos() > chunkEnd)
		throw Exception(tr("File parsing error: Read past end of chunk."));
	chunks.pop();

	qint64 currentPos = filePosition();
	if(currentPos > chunkEnd)
		throw Exception(tr("Read past end of file chunk."));

    // Go to end of chunk
	if(currentPos != chunkEnd) {
		if(!is.device()->seek(chunkEnd))
			throw Exception(tr("Failed to seek in input file."));
	}

	// Check end code.
	quint32 code;
	*this >> code;
	if(code != 0x0FFFFFFF)
		throw Exception(tr("Invalid file structure."));
}

/******************************************************************************
* Reads a pointer to an object of type T from the input stream.
* This method will patch the address immediately if it is available, 
* otherwise it will happen later when it is known. 
******************************************************************************/
quint64 LoadStream::readPointer(void** patchPointer) 
{
	quint64 id;
	*this >> id;
	if(id == 0) { *patchPointer = NULL; }
	else {
		if(pointerMap.size() > id && resolvedPointers.test(id))
			*patchPointer = pointerMap[id];
		else
			backpatchPointers.insert(make_pair(id, patchPointer));
	}
	return id;
}

/******************************************************************************
* Resolves an ID with the real pointer.
* This method will backpatch all registered pointers with the given ID.
******************************************************************************/
void LoadStream::resolvePointer(quint64 id, void* pointer)
{	
	OVITO_ASSERT(id != 0);
	OVITO_ASSERT(id >= resolvedPointers.size() || !resolvedPointers.test(id));
	if(id >= pointerMap.size()) {
		pointerMap.resize(id+1);
		resolvedPointers.resize(id+1);
	}
	pointerMap[id] = pointer;
	resolvedPointers.set(id);

	// Backpatch pointers.
	multimap<quint64, void**>::iterator first = backpatchPointers.find(id);
	if(first == backpatchPointers.end()) return;
	multimap<quint64, void**>::iterator last = first;
	for(; last != backpatchPointers.end(); ++last) {
		if(last->first != id) break;
		*(last->second) = pointer;
	}

	backpatchPointers.erase(first, last);
}

};	// End of namespace Base

