/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var constantFunction = require( '@stdlib/utils/constant-function' );
var degenerate = require( './../../../../../base/dists/degenerate/mgf' ).factory;
var isnan = require( '@stdlib/math/base/assert/is-nan' );
var sinc = require( '@stdlib/math/base/special/sinc' );
var exp = require( '@stdlib/math/base/special/exp' );
var abs = require( '@stdlib/math/base/special/abs' );


// MAIN //

/**
* Returns a function for evaluating the moment-generating function (MGF) of a logistic distribution with mean `mu` and scale parameter `s`.
*
* @param {number} mu - mean
* @param {NonNegativeNumber} s - scale parameter
* @returns {Function} MGF
*
* @example
* var mgf = factory( 10.0, 0.5 );
*
* var y = mgf( 0.5 );
* // returns ~164.846
*
* y = mgf( 2.0 );
* // returns Infinity
*/
function factory( mu, s ) {
	if ( isnan( mu ) || isnan( s ) || s < 0.0 ) {
		return constantFunction( NaN );
	}
	if ( s === 0.0 ) {
		return degenerate( mu );
	}
	return mgf;

	/**
	* Evaluates the moment-generating function (MGF) for a logistic distribution.
	*
	* @private
	* @param {number} t - input value
	* @returns {number} evaluated MGF
	*
	* @example
	* var y = mgf( 0.5 );
	* // returns <number>
	*/
	function mgf( t ) {
		var st = s * t;
		if ( abs( st ) > 1.0 ) {
			return NaN;
		}
		return exp( mu * t ) / sinc( st );
	}
}


// EXPORTS //

module.exports = factory;
