// Copyright (c) 2004-2008 MySQL AB, 2008-2009 Sun Microsystems, Inc.
//
// MySQL Connector/NET is licensed under the terms of the GPLv2
// <http://www.gnu.org/licenses/old-licenses/gpl-2.0.html>, like most 
// MySQL Connectors. There are special exceptions to the terms and 
// conditions of the GPLv2 as it is applied to this software, see the 
// FLOSS License Exception
// <http://www.mysql.com/about/legal/licensing/foss-exception.html>.
//
// This program is free software; you can redistribute it and/or modify 
// it under the terms of the GNU General Public License as published 
// by the Free Software Foundation; version 2 of the License.
//
// This program is distributed in the hope that it will be useful, but 
// WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
// or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License 
// for more details.
//
// You should have received a copy of the GNU General Public License along 
// with this program; if not, write to the Free Software Foundation, Inc., 
// 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA

using System;
using System.Data;
using System.IO;
using NUnit.Framework;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Threading;

namespace MySql.Data.MySqlClient.Tests
{
	[TestFixture]
	public class TableCaching : BaseTest
	{
        [Test]
        public void SimpleTableCaching()
        {
            execSQL("CREATE TABLE test (id INT, name VARCHAR(20), name2 VARCHAR(20))");
            execSQL("INSERT INTO test VALUES (1, 'boo', 'hoo'), (2, 'first', 'last'), (3, 'fred', 'flintstone')");

            MySqlTrace.Listeners.Clear();
            MySqlTrace.Switch.Level = SourceLevels.All;
            GenericListener listener = new GenericListener();
            MySqlTrace.Listeners.Add(listener);

            string connStr = GetConnectionString(true) + ";logging=true;table cache=true";
            using (MySqlConnection c = new MySqlConnection(connStr))
            {
                c.Open();

                MySqlCommand cmd = new MySqlCommand("test", c);
                cmd.CommandType = CommandType.TableDirect;
                ConsumeReader(cmd);
                // now run the query again but this time it shouldn't generate a call to the database
                ConsumeReader(cmd);
            }

            Assert.AreEqual(1, listener.Find("Resultset Opened: field(s) = 3"));
        }

        [Test]
        public void ConnectionStringExpiry()
        {
            execSQL("CREATE TABLE test3 (id INT, name VARCHAR(20), name2 VARCHAR(20))");
            execSQL("INSERT INTO test3 VALUES (1, 'boo', 'hoo'), (2, 'first', 'last'), (3, 'fred', 'flintstone')");

            MySqlTrace.Listeners.Clear();
            MySqlTrace.Switch.Level = SourceLevels.All;
            GenericListener listener = new GenericListener();
            MySqlTrace.Listeners.Add(listener);

            string connStr = GetConnectionString(true) + ";logging=true;table cache=true;default table cache age=1";
            using (MySqlConnection c = new MySqlConnection(connStr))
            {
                c.Open();

                MySqlCommand cmd = new MySqlCommand("test3", c);
                cmd.CommandType = CommandType.TableDirect;
                ConsumeReader(cmd);
                Thread.Sleep(1500);
                // now run the query again but this time it should generate a call to the database
                // since our next query is past the cache age of 1 second
                ConsumeReader(cmd);
            }

            Assert.AreEqual(2, listener.Find("Resultset Opened: field(s) = 3"));
        }

        [Test]
        public void SettingAgeOnCommand()
        {
            execSQL("CREATE TABLE test2 (id INT, name VARCHAR(20), name2 VARCHAR(20))");
            execSQL("INSERT INTO test2 VALUES (1, 'boo', 'hoo'), (2, 'first', 'last'), (3, 'fred', 'flintstone')");

            MySqlTrace.Listeners.Clear();
            MySqlTrace.Switch.Level = SourceLevels.All;
            GenericListener listener = new GenericListener();
            MySqlTrace.Listeners.Add(listener);

            string connStr = GetConnectionString(true) + ";logging=true;table cache=true;default table cache age=1";
            using (MySqlConnection c = new MySqlConnection(connStr))
            {
                c.Open();

                MySqlCommand cmd = new MySqlCommand("test2", c);
                cmd.CommandType = CommandType.TableDirect;
                cmd.CacheAge = 20;
                ConsumeReader(cmd);
                Thread.Sleep(1000);
                // now run the query again but this time it shouldn't generate a call to the database
                // since we have overriden the connection string cache age of 1 second and set it
                // to 20 seconds on our command
                ConsumeReader(cmd);
            }

            Assert.AreEqual(1, listener.Find("Resultset Opened: field(s) = 3"));
        }

        private void ConsumeReader(MySqlCommand cmd)
        {
            using (MySqlDataReader reader = cmd.ExecuteReader())
            {
                reader.Read();
                Assert.AreEqual(1, reader.GetInt32(0));
                Assert.AreEqual("boo", reader.GetString(1));
                Assert.AreEqual("hoo", reader.GetString(2));
                reader.Read();
                Assert.AreEqual(2, reader.GetInt32(0));
                Assert.AreEqual("first", reader.GetString(1));
                Assert.AreEqual("last", reader.GetString(2));
                reader.Read();
                Assert.AreEqual(3, reader.GetInt32(0));
                Assert.AreEqual("fred", reader.GetString(1));
                Assert.AreEqual("flintstone", reader.GetString(2));
                Assert.IsFalse(reader.Read());
            }
        }
    }
}
