# filestore.py
#  Copyright 2004 Daniel Burrows
#
#
#  This maps filenames to active file objects.  It is responsible for
#  managing file objects that exist during the program's execution (as
#  opposed to the cache).
#
#  This could be merged with the cache, but then I'd have to write
#  custom picklers to only pickle the cache data, which doesn't work
#  with cPickle, and the only real advantage is that you don't have to
#  maintain the cache data separately from the store.  Given that
#  invalid cache info is explicitly allowed (it just has to be
#  recalculated on startup), this is not a huge deal.
#
#  Circular symlinks will be broken.  If a file has multiple names,
#  some of the names will be arbitrarily discarded.

import cache
import musicfile
import os
import os.path
import sets
import sys

class FileStoreFileOperationError(Exception):
    """An error that indicates that an operation failed on some files.
    The recommended (multi-line) error message is stored in the
    strerror attribute."""

    def __init__(self, failed, strerror):
        self.failed=failed
        self.__strerror=strerror

    def __getattr__(self, name):
        if name == 'strerror':
            def make_error(x):
                fn,strerror=x

                if strerror == None:
                    return fn
                else:
                    return '%s (%s)'%(fn,strerror)

            self.strerror='%s%s'%(self.__strerror,'\n'.join(map(make_error, self.failed)))
            return self.strerror
        else:
            raise AttributeError, name

class SaveError(FileStoreFileOperationError):
    """An error that indicates that some files failed to be saved."""

    def __init__(self, failed):
        FileStoreFileOperationError.__init__(self, failed, 'Changes to the following files could not be saved:\n')

    def __str__(self):
        # Return a list of the failed files
        return 'Failed to save files: %s'%','.join(map(lambda x:x[0], self.failed))

class LoadError(FileStoreFileOperationError):
    """An error that indicates that some files failed to be saved."""

    def __init__(self, failed):
        FileStoreFileOperationError.__init__(self, failed, 'The following files could not be read:\n')

    def __str__(self):
        # Return a list of the failed files
        return 'Failed to load files: %s'%','.join(map(lambda x:x[0], self.failed))

class NotDirectoryError(Exception):
    """This error is raised when a non-directory is passed to the add_dir method."""

    def __init__(self, dir):
        Exception.__init__(self, dir)
        self.strerror='%s is not a directory'%dir

def fname_ext(fn):
    "Returns the extension of the given filename, or None if it has no extension."
    if not '.' in fn or fn.rfind('.')==len(fn)-1:
        return None

    return fn[fn.rfind('.')+1:]

class FileStore:
    """A collection of music files, indexed by name.  Files are added
    to the store using the add_dir and add_file functions.  Each file
    has exactly one corresponding 'file object' in the store, whose
    lifetime is equal to that of the store itself."""
    def __init__(self, cache):
        """Initializes an empty store attached to the given cache."""

        self.files={}
        self.modified_files=sets.Set()
        self.inodes=sets.Set()
        self.cache=cache

    def add_dir(self, dir, callback=lambda cur,max:None, set=None):
        """Adds the given directory and any files recursively
        contained inside it to this file store.  'set' may be a
        mutable set; file objects added as a result of this operation
        will be placed in 'set'."""

        if not os.path.isdir(dir):
            raise NotDirectoryError(dir)

        candidates=[]
        self.__find_files(dir, sets.Set(), candidates, callback)

        cur=0
        max=len(candidates)
        failed=[]
        for fn in candidates:
            callback(cur, max)
            cur+=1
            try:
                self.add_file(fn, set)
            except LoadError,e:
                failed+=e.failed
        callback(max, max)

        self.cache.flush()

        if failed <> []:
            raise LoadError(failed)

    # Finds all files in the given directory and subdirectories,
    # following symlinks and avoiding cycles.
    def __find_files(self, dir, seen_dirs, output, callback=lambda cur,max:None):
        """Returns a list of all files contained in the given directory and
        subdirectories which have an extension that we recognize.  The
        result is built in the list 'output'."""
        assert(os.path.isdir(dir))

        dir_ino=os.stat(dir).st_ino

        callback(None, None)

        if dir_ino not in seen_dirs and os.access(dir, os.R_OK|os.X_OK):
            seen_dirs.add(dir_ino)

            for name in os.listdir(dir):
                fullname=os.path.join(dir, name)

                if os.path.isfile(fullname) and musicfile.file_types.has_key(fname_ext(fullname)):
                    output.append(fullname)
                elif os.path.isdir(fullname):
                    self.__find_files(fullname, seen_dirs, output, callback)

        return output

    def add_file(self, fn, set=None):
        """Adds the given file to the store.  If an exception is
        raised when we try to open the file, print it and
        continue. 'set' may be a mutable set, in which case any file
        object which is successfully created will be added to it."""
        fn=os.path.normpath(fn)
        if self.files.has_key(fn):
            # We've already got one!  (it's very nice, too)
            set.add(self.files[fn])
            return

        st=os.stat(fn)
        file_ino=st.st_ino
        if file_ino not in self.inodes and os.access(fn, os.R_OK):
            self.inodes.add(file_ino)

            # Find the file extension and associated handler
            ext=fname_ext(fn)
            if musicfile.file_types.has_key(ext):
                try:
                    try:
                        cacheinf=self.cache.get(fn, st)
                    except:
                        cacheinf=None

                    new_file=musicfile.file_types[ext](self, fn, cacheinf)

                    self.files[fn]=new_file
                    self.cache.put(new_file, st)
                    if set <> None:
                        set.add(new_file)
                except EnvironmentError,e:
                    raise LoadError([(fn, e.strerror)])
                except musicfile.MusicFileError,e:
                    raise LoadError([(fn, e.strerror)])
                except:
                    raise LoadError([(fn, None)])

    def commit(self, callback=lambda cur,max:None, S=None):
        """Commit all changes to files in the set S to the store.  If
        S is not specified or None, it defaults to the entire store."""
        if S == None:
            modified=self.modified_files
        else:
            modified=S & self.modified_files

        cur=0
        max=len(modified)
        failed=[]
        for f in modified:
            callback(cur, max)
            cur+=1
            try:
                f.commit()
            except EnvironmentError,e:
                failed.append((f.fn,e.strerror))
            except musicfile.MusicFileError,e:
                failed.append((f.fn,e.strerror))
            except:
                failed.append((f.fn,None))
            try:
                cache.put(f)
            except:
                pass
        callback(max, max)

        if failed <> []:
            raise SaveError(failed)

    def revert(self, callback=lambda cur,max:None, S=None):
        """Revert all modifications to files in the set S.  If S is
        not specified or None, it defaults to the entire store."""
        if S == None:
            modified=self.modified_files
        else:
            modified=S & self.modified_files

        cur=0
        max=len(modified)
        for f in modified:
            callback(cur, max)
            cur+=1
            f.revert()
        callback(max, max)

    def set_modified(self, file, modified):
        """Updates whether the given file is known to be modified."""
        if modified:
            self.modified_files.add(file)
        else:
            self.modified_files.remove(file)

    def modified_count(self, S=None):
        """Returns the number of files in the set S that are modified.
        If S is not specified or None, it defaults to the entire
        store."""

        if S == None:
            modified=self.modified_files
        else:
            modified=S & self.modified_files

        return len(modified)


