/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009-2011 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdlib.h>
#include <sys/types.h>
#include <sys/sysctl.h>
#include <string.h>
#include <errno.h>
#include <vm/vm_param.h>
#include <sys/stat.h>
#include <limits.h>
#include <fcntl.h>

#include "utils.h"

#define S_MIB_SIZE 3

static void assign(struct sa_swap* __restrict__ dst, struct xswdev* __restrict__ src) SA_NONNULL;

extern long pagesize;
__thread int s_mib[S_MIB_SIZE];
__thread struct xswdev* buffer_s;
__thread size_t buffer_s_size;
__thread uint16_t number_swaps;

int sa_open_swap() {
	size_t m_mib_size = S_MIB_SIZE;
	if (sysctlnametomib("vm.swap_info", s_mib, &m_mib_size) == -1)
		return ENOSYS;

	return 0;
}

int sa_count_swaps(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	*number = 0;
	struct xswdev xsw;
	size_t size = sizeof xsw;
	for (s_mib[S_MIB_SIZE - 1] = 0; sysctl(s_mib, S_MIB_SIZE, &xsw, &size, NULL, 0) != -1; ++s_mib[S_MIB_SIZE - 1])
		++(*number);

	return 0;
}

int sa_reset_swaps() {
	size_t len = 0;
	size_t size = sizeof(struct xswdev);
	s_mib[S_MIB_SIZE - 1] = 0;
	number_swaps = 0;

	while (1) {
		len += sizeof(struct xswdev);
		if (len > buffer_s_size) {
			safe_realloc(&buffer_s, &len);
			buffer_s_size = len;
			if (buffer_s == NULL)
				return ENOMEM;
		}

		if (sysctl(s_mib, S_MIB_SIZE, &buffer_s[number_swaps], &size, NULL, 0) == -1)
			break;

		++s_mib[S_MIB_SIZE - 1];
		++number_swaps;
	}

	return 0;
}

int sa_get_swap(uint16_t index, struct sa_swap* dst) {
	if (dst == NULL)
		return EINVAL;
	if (index >= number_swaps)
		return ENODEV;

	assign(dst, &buffer_s[index]);
	return 0;
}

int sa_get_swaps(struct sa_swap* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	*written = 0;
	uint16_t i;
	for (i = 0; i < number_swaps; ++i) {
		if (i == dst_size)
			return ENOMEM;

		assign(&dst[i], &buffer_s[i]);
		++(*written);
	}
	return 0;
}

static void assign(struct sa_swap* __restrict__ dst, struct xswdev* __restrict__ src) {
	devname_r(src->xsw_dev, S_IFCHR, dst->name, SA_SWAP_NAME);
	dst->total = (uint64_t) src->xsw_nblks * pagesize;
	dst->free = (uint64_t) (src->xsw_nblks - src->xsw_used) * pagesize;
}
