/*==============================================================================
** Lynkeos
** $Id: fourier.c,v 1.10 2005/01/27 23:14:32 j-etienne Exp $
** Created by Jean-Etienne LAMIAUD on Aug 5, 2003.
**------------------------------------------------------------------------------
** Copyright (c) 2003-2005. Jean-Etienne LAMIAUD
**------------------------------------------------------------------------------
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
**
**------------------------------------------------------------------------------
*/

#include <stdlib.h>
#include <assert.h>

#include <sys/sysctl.h>

#ifdef GNUSTEP
#else
#include <CarbonCore/Multiprocessing.h>
#endif

#include <pthread.h>

#include "fourier.h"

u_short numberOfCpus;

static unsigned fftwDefaultFlag;
// Mutex used to protect every call to FFTW except fftw_execute
static pthread_mutex_t fftwLock;

void initializeProcessing()
{
   // First evaluate if the CPU has Altivec capacity

#ifdef GNUSTEP
  int hasVectorUnit = 0;
  int error = 1;
#else
   int selectors[2] = { CTL_HW, HW_VECTORUNIT };
   int hasVectorUnit = 0;
   size_t length = sizeof(hasVectorUnit);
   int error = sysctl(selectors, 2, &hasVectorUnit, &length, NULL, 0);
#endif
   
   if ( error == 0 && hasVectorUnit )
      fftwDefaultFlag = 0;
   else
      fftwDefaultFlag = FFTW_NO_SIMD;

#ifdef GNUSTEP
   //For now, on linux/GNUstep you, use only one processor ...
   numberOfCpus = 1;
#else
   // Then read the number of CPUs we are running on
   if ( MPLibraryIsLoaded() )
      numberOfCpus = MPProcessors();
   else
      numberOfCpus = 1;
#endif

   // Create a lock for FFTW non thread safe functions
   pthread_mutex_init( &fftwLock, NULL );

   // Prepare FFTW to work with threads
   fftw_init_threads();
}

/*=============================================================================
** FFT_DATA_INIT
**-----------------------------------------------------------------------------
**
** Purpose       : Initialize the FFT structures
**
**-----------------------------------------------------------------------------
**
** Entry     : spectrum
**
** Output    : None
**
**=============================================================================
*/
inline void FFT_DATA_INIT( FFT_DATA *d )
{
   d->spectrum = NULL;
   d->goal = 0;
}

/*=============================================================================
** free_spectrum
**-----------------------------------------------------------------------------
**
** Purpose       : free the spectrums planes
**
**-----------------------------------------------------------------------------
**
** Entry     : spectrum
**
** Output    : None
**
**=============================================================================
*/
void free_spectrum( FFT_DATA *s )
{
   pthread_mutex_lock( &fftwLock );

   if ( s->spectrum != NULL )
      FFT_FREE( s->spectrum );
   s->spectrum = NULL;
   if ( s->goal & FOR_DIRECT )
      FFT_DESTROY_PLAN( s->direct );
   if ( s->goal & FOR_INVERSE )
      FFT_DESTROY_PLAN( s->inverse );
   s->goal = 0;

   pthread_mutex_unlock( &fftwLock );
}

/*=============================================================================
** allocate_spectrum
**-----------------------------------------------------------------------------
**
** Purpose       : Allocate the spectrums planes
**
**-----------------------------------------------------------------------------
**
** Entry     : RGB spectrum
**             square side
**
** Output    : None
**
**=============================================================================
*/
void allocate_spectrum( FFT_DATA *s, u_short w, u_short h, u_char nplanes, 
                        u_char goal )
{
   int sizes[2];

   free_spectrum( s );

   pthread_mutex_lock( &fftwLock );

   s->spectrum = FFT_MALLOC( nplanes*sizeof(COMPLEX)*(w/2+1)*h );
   assert( s->spectrum != NULL );
   s->w = w;
   s->h = h;
   s->pw = (w/2+1)*sizeof(COMPLEX)/sizeof(REAL);
   s->nplanes = nplanes;
   
   s->goal = goal;

   sizes[0] = h;
   sizes[1] = w;

   if ( goal & FOR_DIRECT )
      s->direct = FFT_PLAN_R2C( 2, sizes, nplanes,
                                (REAL*)s->spectrum, NULL, 1, s->pw*h,
                                s->spectrum, NULL, 1, (w/2+1)*h,
                                fftwDefaultFlag | FFTW_MEASURE );
   if ( goal & FOR_INVERSE )
      s->inverse = FFT_PLAN_C2R( 2, sizes,  nplanes,
                                 s->spectrum,  NULL, 1, (w/2+1)*h,
                                 (REAL*)s->spectrum,  NULL, 1, s->pw*h,
                                 fftwDefaultFlag | FFTW_MEASURE );

   pthread_mutex_unlock( &fftwLock );
}

/*=============================================================================
** log_2
**-----------------------------------------------------------------------------
**
** Purpose   : Base 2 logarithm
**
**-----------------------------------------------------------------------------
**
** Entry     : Value
**
** Output    : Its integer base 2 log
**
**=============================================================================
*/
short log_2( short val )
{
   short i, n;

   assert( val != 0 );

   for( i = val, n = 0; i != 0; i >>= 1, n++ )  /* Search for last bit set */
      ;

   return( n-1 );
}

/*=============================================================================
** fourier
**-----------------------------------------------------------------------------
**
** Purpose   : Direct color Fourier transform 
**
**-----------------------------------------------------------------------------
**
** Entry     : Sample color array
**
** Output    : Transformed color array
**
**=============================================================================
*/
void fourier( FFT_DATA sample )
{
   assert( sample.goal & FOR_DIRECT );
   FFT_EXECUTE( sample.direct );
}

/*=============================================================================
** fourier_inverse
**-----------------------------------------------------------------------------
**
** Purpose   : Inverse Fourier transform
**
**-----------------------------------------------------------------------------
**
** Entry     : Spectrum sample array
**
** Output    : Transformed array
**
**=============================================================================
*/
void fourier_inverse( FFT_DATA sample, REAL *vmin, REAL *vmax )
{
   const REAL area = sample.w*sample.h;
   REAL min = HUGE, max = -HUGE;
   REAL *result;
   int x, y, c;

   assert( sample.goal & FOR_INVERSE );
   FFT_EXECUTE( sample.inverse );

   result = (REAL*)sample.spectrum;

   for( y = 0; y < sample.h; y++ )
   {
      for( x = 0; x < sample.w; x++ )
      {
         for( c = 0; c < sample.nplanes; c++ )
         {
            u_long i = (y+c*sample.h)*sample.pw+x;
            result[i] /= area;

            /* Update the range */
            if ( result[i] < min )
               min = result[i];
            if ( result[i] > max )
               max = result[i];
         }
      }
   }

   if ( vmin != NULL )
      *vmin = min;
   if ( vmax != NULL )
      *vmax = max;
}

inline void *rgbPlane( FFT_DATA sample, u_char c )
{
   return( &sample.spectrum[(c)*sample.h*(sample.w/2+1)] );
}

inline REAL *colorValue( FFT_DATA sample, u_short x, u_short y, 
                               u_char c )
{
   return( &((REAL*)sample.spectrum)[((y)+(c)*sample.h)*sample.pw+x] );
}
