#!/usr/bin/python
# -*- coding: UTF-8 -*-

"""
	Qt version of Lybniz, a	simple function graph plotter
	© Thomas Führinger, Sam Tygier 2005-2008
	http://lybniz2.sourceforge.net/

	Requires PyQt 4.3
	Released under the terms of the revised BSD license
	Modified: 2008-05-01
"""

import sys
from math import *
from PyQt4 import QtCore, QtGui, Qt, QtXml

#import lybniz_rc

def marks(min_val, max_val, minor=1):
    "Yield positions of scale marks between min and max. For making minor marks, set minor to the number of minors you want between majors"
    try:
    	min_val = float(min_val)
    	max_val = float(max_val)
    except:
    	print "needs 2 numbers"
    	raise ValueError

    if(min_val >= max_val):
    	print "min bigger or equal to max"
    	raise ValueError

    a = 0.2 # tweakable control for when to switch scales
            # big a value results in more marks

    a = a + log10(minor)

    width = max_val - min_val
    log10_range = log10(width)

    interval = 10 ** int(floor(log10_range - a))
    lower_mark = min_val - fmod(min_val, interval)

    if lower_mark < min_val:
    	lower_mark += interval

    a_mark = lower_mark
    while a_mark <= max_val:
    	if abs(a_mark) < interval / 2:
    		a_mark = 0
    	yield a_mark
    	a_mark += interval

class RenderArea(QtGui.QWidget):

    def __init__(self, parent = None):
        QtGui.QWidget.__init__(self, parent)
        self.parent = parent

        self.pen = QtGui.QPen()
        self.dash_pen = QtGui.QPen(QtCore.Qt.DashLine)
        self.bluePen = QtGui.QPen()
        self.bluePen.setColor(QtCore.Qt.blue)
        self.redPen = QtGui.QPen()
        self.redPen.setColor(QtCore.Qt.red)
        self.greenPen = QtGui.QPen()
        self.greenPen.setColor(QtCore.Qt.green)

        self.scale_style = "dec"
        self.connect_points = True
        
    def resizeEvent(self, event):
        self.image = QtGui.QImage(event.size(), QtGui.QImage.Format_RGB32)
        self.image.fill(QtGui.qRgb(255, 255, 255))
        self.plot()
        QtGui.QWidget.resizeEvent(self, event)
        
    def replot(self):
        self.image = QtGui.QImage(self.size(), QtGui.QImage.Format_RGB32)
        self.image.fill(QtGui.qRgb(255, 255, 255))
        self.plot()
        self.update()

    def plot(self):
        painter = QtGui.QPainter()
        painter.begin(self.image)
        painter.setPen(self.pen)

        # where to put the numbers
        numbers_x_pos = -10
        numbers_y_pos = 10

        # where to center the axis
        center_x_pix = int(round(self.canvas_x(0)))
        center_y_pix = int(round(self.canvas_y(0)))
        if (center_x_pix < 5): center_x_pix = 5
        if (center_x_pix < 20): numbers_x_pos = 10
        if (center_y_pix < 5): center_y_pix = 5
        if (center_x_pix > self.image.width()- 5): center_x_pix = self.image.width() - 5
        if (center_y_pix > self.image.height() - 5): center_y_pix = self.image.height() - 5
        if (center_y_pix > self.image.height() - 20): numbers_y_pos = -10

        # draw cross
        painter.drawLine(QtCore.QLine(center_x_pix, 0, center_x_pix, self.image.height()))
        painter.drawLine(QtCore.QLine(0, center_y_pix, self.image.width(), center_y_pix))

        factor = 1
        if (self.scale_style == "rad"): factor = pi

        for i in marks(self.x_min / factor, self.x_max / factor):
            label = "%g" % i
            if (self.scale_style == "rad"): label += u"π"
            i = i * factor

            painter.drawLine(QtCore.QLine(int(round(self.canvas_x(i))), center_y_pix - 5, int(round(self.canvas_x(i))), center_y_pix + 5))
            
            if numbers_y_pos < 0:
                adjust = painter.boundingRect(self.image.rect(), QtCore.Qt.AlignLeft, label).height()
            else: adjust = 0
            painter.drawText(QtCore.QPoint(int(round(self.canvas_x(i))), center_y_pix + numbers_y_pos - adjust + 10), label)

        for i in marks(self.y_min, self.y_max):
            label = "%g" % i

            painter.drawLine(QtCore.QLine(center_x_pix - 5, int(round(self.canvas_y(i))), center_x_pix + 5, int(round(self.canvas_y(i)))))

            if numbers_x_pos < 0:
                adjust = painter.boundingRect(self.image.rect(), QtCore.Qt.AlignLeft, label).width()
            else: adjust = 0
            painter.drawText(QtCore.QPoint(center_x_pix + numbers_x_pos - adjust, int(round(self.canvas_y(i)))), label)

        # minor marks
        for i in marks(self.x_min / factor, self.x_max / factor, minor=10):
            i = i * factor
            painter.drawLine(int(round(self.canvas_x(i))), center_y_pix - 2, int(round(self.canvas_x(i))), center_y_pix + 2)

        for i in marks(self.y_min, self.y_max, minor=10):
            label = "%g" % i
            painter.drawLine(center_x_pix - 2, int(round(self.canvas_y(i))), center_x_pix + 2, int(round(self.canvas_y(i))))

        plots = []
        # precompile the functions
        try:
            compiled_y1 = compile(self.y1, "", "eval")
            plots.append((compiled_y1, 0, self.bluePen))
        except:
        	compiled_y1 = None
        try:
            compiled_y2 = compile(self.y2, "", "eval")
            plots.append((compiled_y2, 1, self.redPen))
        except:
        	compiled_y2 = None
        try:
            compiled_y3 = compile(self.y3, "", "eval")
            plots.append((compiled_y3, 2, self.greenPen))
        except:
        	compiled_y3 = None

        self.prev_y = [None, None, None]

        if len(plots) != 0:
            for i in xrange(0, self.image.width()):
                x = self.graph_x(i + 1)
                for e in plots:
                    try:
                        y = eval(e[0])
                        y_c = round(self.canvas_y(y))

                        if y_c < 0 or y_c > self.image.height():
                            raise ValueError

                        painter.setPen(e[2])

                        if self.connect_points and self.prev_y[e[1]] is not None:
                            painter.drawLine(QtCore.QLine(i, self.prev_y[e[1]], i + 1, y_c))
                        else:
                            painter.drawPoint(QtCore.QPoint(i + 1, y_c))
                        self.prev_y[e[1]] = y_c
                    except:
        				#print "Error at %d: %s" % (x, sys.exc_value)
        				self.prev_y[e[1]] = None

        painter.end()


    def paintEvent(self, event):
        painter = QtGui.QPainter()
        painter.begin(self)
        painter.drawImage(QtCore.QPoint(0, 0), self.image)
        painter.end()
        
    def mousePressEvent(self, event):
        self.selection = QtCore.QRect(QtCore.QPoint(event.pos()), QtCore.QPoint(event.pos()))
        self.orig_image = QtGui.QImage(self.image)
        
    def mouseMoveEvent(self, event):
        self.image = QtGui.QImage(self.orig_image)
        self.selection.setBottomRight(QtCore.QPoint(event.pos()))
        painter = QtGui.QPainter()
        painter.begin(self.image)
        painter.setPen(self.dash_pen)
        painter.drawRect(self.selection)
        self.old_rect = self.selection
        painter.end()
        self.update()
        
    def mouseReleaseEvent(self, event):
        if self.selection.width() <> 0 and self.selection.height() <> 0:
            xmi, xma = self.graph_x(min(self.selection.left(), self.selection.right())), self.graph_x(max(self.selection.left(), self.selection.right()))
            ymi, yma = self.graph_y(max(self.selection.bottom(), self.selection.top())), self.graph_y(min(self.selection.bottom(), self.selection.top()))
            self.x_min, self.x_max, self.y_min, self.y_max = xmi, xma, ymi, yma
            
            self.replot()
            self.parent.parameter_entries_repopulate()

    def canvas_x(self, x):
        "Calculate position on canvas to point on graph"
        return (x - self.x_min) * self.image.width() / (self.x_max - self.x_min)

    def canvas_y(self, y):
        return (self.y_max - y) * self.image.height() / (self.y_max - self.y_min)

    def canvas_point(self, x, y):
        return (self.canvas_x(x), self.canvas_y(y))

    def graph_x(self, x):
        "Calculate position on graph from point on canvas"
        return x  * (self.x_max - self.x_min) / self.image.width() + self.x_min

    def graph_y(self, y):
        return self.y_max - (y * (self.y_max - self.y_min) / self.image.height())


class Graph(QtGui.QMdiSubWindow):
    sequence_number = 1

    def __init__(self, parent=None):
        QtGui.QMdiSubWindow.__init__(self, parent)

        self.setAttribute(QtCore.Qt.WA_DeleteOnClose)
        self.setWindowIcon(QtGui.QIcon(":/images/sheet.png"))
        self.resize(QtCore.QSize(500, 400))
        self.is_untitled = True

    def new_file(self):
        self.current_file = self.tr("Graph%1.lyb").arg(Graph.sequence_number)
        Graph.sequence_number += 1
        self.setWindowTitle(self.current_file)

        self.y1_line_edit = QtGui.QLineEdit()
        self.y2_line_edit = QtGui.QLineEdit()
        self.y3_line_edit = QtGui.QLineEdit()

        layout = QtGui.QGridLayout()

        layout.addWidget(QtGui.QLabel(self.tr("<font color='blue'>y1 = </font>")), 0, 0)
        layout.addWidget(QtGui.QLabel(self.tr("<font color='red'>y2 = </font>")), 1, 0)
        layout.addWidget(QtGui.QLabel(self.tr("<font color='green'>y3 = </font>")), 2, 0)

        self.y1_line_edit = QtGui.QLineEdit()
        self.y2_line_edit = QtGui.QLineEdit()
        self.y3_line_edit = QtGui.QLineEdit()
        self.x_min_line_edit = QtGui.QLineEdit("-5.0")
        self.x_max_line_edit = QtGui.QLineEdit("5.0")
        self.y_min_line_edit = QtGui.QLineEdit("-3.0")
        self.y_max_line_edit = QtGui.QLineEdit("3.0")
        
        self.connect(self.y1_line_edit, QtCore.SIGNAL("returnPressed()"), self.plot)
        self.connect(self.y2_line_edit, QtCore.SIGNAL("returnPressed()"), self.plot)
        self.connect(self.y3_line_edit, QtCore.SIGNAL("returnPressed()"), self.plot)
        self.render_area = RenderArea(self)

        layout.addWidget(self.y1_line_edit, 0, 1)
        layout.addWidget(self.y2_line_edit, 1, 1)
        layout.addWidget(self.y3_line_edit, 2, 1)

        layout.addWidget(QtGui.QLabel(self.tr("  X min ")), 0, 2)
        layout.addWidget(QtGui.QLabel(self.tr("  X max ")), 0, 4)

        layout.addWidget(self.x_min_line_edit, 0, 3)
        layout.addWidget(self.x_max_line_edit, 0, 5)

        layout.addWidget(QtGui.QLabel(self.tr("  Y min ")), 1, 2)
        layout.addWidget(QtGui.QLabel(self.tr("  Y max ")), 1, 4)

        layout.addWidget(self.y_min_line_edit, 1, 3)
        layout.addWidget(self.y_max_line_edit, 1, 5)

        layout.addWidget(self.render_area, 3, 0, 3, 6)

        layout.setColumnStretch(1, 1)
        layout.setColumnMinimumWidth(1, 100)
        layout.setRowStretch(3, 1)
        layout.setRowMinimumHeight(3, 100)

        w = QtGui.QWidget()
        w.setLayout(layout)
        self.setWidget(w)
        self.plot()
        self.modified = False


    def file_open(self, root, file_name):
           
        e = root.firstChildElement("parameters")
        self.y1_line_edit.setText(e.firstChildElement("y1").text())
        self.y2_line_edit.setText(e.firstChildElement("y2").text())
        self.y3_line_edit.setText(e.firstChildElement("y3").text())
        self.x_min_line_edit.setText(e.firstChildElement("x_min").text())
        self.x_max_line_edit.setText(e.firstChildElement("x_max").text())
        self.y_min_line_edit.setText(e.firstChildElement("y_min").text())
        self.y_max_line_edit.setText(e.firstChildElement("y_max").text())
        
        e = root.firstChildElement("settings")
        self.render_area.connect_points = bool(e.firstChildElement("connect_points").text())
        self.render_area.scale_style = e.firstChildElement("scale_style").text()
        self.plot()
        
        self.x_min = e.firstChildElement("x_min").text()
        self.x_max = e.firstChildElement("x_max").text()
        self.y_min = e.firstChildElement("y_min").text()
        self.y_max = e.firstChildElement("y_max").text()
        self.move(QtCore.QPoint(int(e.firstChildElement("x").text()), int(e.firstChildElement("y").text())))
        self.resize(QtCore.QSize(int(e.firstChildElement("w").text()), int(e.firstChildElement("h").text())))
                
        self.current_file = file_name
        self.setWindowTitle(self.current_file)
        self.is_untitled = False
        self.modified = False


    def save(self):
        "saves function definitions, returns True if sucessful"
        if self.is_untitled:
            return self.save_as()
        else:            
            doc = QtXml.QDomDocument("Lybniz")
            root = doc.createElement("graph")
            
            e = doc.createElement("parameters")
            root.appendChild(e)
            
            tag = doc.createElement("y1")
            e.appendChild(tag)
            t = doc.createTextNode(self.y1_line_edit.text())
            tag.appendChild(t)
            
            tag = doc.createElement("y2")
            e.appendChild(tag)
            t = doc.createTextNode(self.y2_line_edit.text())
            tag.appendChild(t)
            
            tag = doc.createElement("y3")
            e.appendChild(tag)
            t = doc.createTextNode(self.y3_line_edit.text())
            tag.appendChild(t)
            
            tag = doc.createElement("x_min")
            e.appendChild(tag)
            t = doc.createTextNode(self.x_min_line_edit.text())
            tag.appendChild(t)
            
            tag = doc.createElement("x_max")
            e.appendChild(tag)
            t = doc.createTextNode(self.x_max_line_edit.text())
            tag.appendChild(t)
            
            tag = doc.createElement("y_min")
            e.appendChild(tag)
            t = doc.createTextNode(self.y_min_line_edit.text())
            tag.appendChild(t)
            
            tag = doc.createElement("y_max")
            e.appendChild(tag)
            t = doc.createTextNode(self.y_max_line_edit.text())
            tag.appendChild(t)
            
            e = doc.createElement("settings")
            root.appendChild(e)
            
            tag = doc.createElement("x")
            e.appendChild(tag)
            t = doc.createTextNode(str(self.pos().x()))
            tag.appendChild(t)
            
            tag = doc.createElement("y")
            e.appendChild(tag)
            t = doc.createTextNode(str(self.pos().y()))
            tag.appendChild(t)
            
            tag = doc.createElement("w")
            e.appendChild(tag)
            t = doc.createTextNode(str(self.size().width()))
            tag.appendChild(t)
            
            tag = doc.createElement("h")
            e.appendChild(tag)
            t = doc.createTextNode(str(self.size().height()))
            tag.appendChild(t)
            
            tag = doc.createElement("x_min")
            e.appendChild(tag)
            t = doc.createTextNode(str(self.x_min))
            tag.appendChild(t)
            
            tag = doc.createElement("x_max")
            e.appendChild(tag)
            t = doc.createTextNode(str(self.x_max))
            tag.appendChild(t)
            
            tag = doc.createElement("y_min")
            e.appendChild(tag)
            t = doc.createTextNode(str(self.y_min))
            tag.appendChild(t)
            
            tag = doc.createElement("y_max")
            e.appendChild(tag)
            t = doc.createTextNode(str(self.y_max))
            tag.appendChild(t)
            
            tag = doc.createElement("scale_style")
            e.appendChild(tag)
            t = doc.createTextNode(self.render_area.scale_style)
            tag.appendChild(t)
            
            tag = doc.createElement("connect_points")
            e.appendChild(tag)
            t = doc.createTextNode(str(self.render_area.connect_points))
            tag.appendChild(t)
            
            doc.appendChild(root)
        
            file = QtCore.QFile(self.current_file)
            if not file.open(QtCore.QIODevice.WriteOnly | QtCore.QIODevice.Text):
                return False
            file.writeData(str(doc.toString(4)))
            file.close()
            self.modified = False
            return True
        

    def save_as(self):
        file_name = QtGui.QFileDialog.getSaveFileName(self, "Save Function Definitions", self.current_file, "Lybniz (*.lyb)")
        if file_name:
            self.current_file = file_name
            self.setWindowTitle(self.current_file)
            self.is_untitled = False
            return self.save()
        else:
            return False


    def save_graph(self):
        file_name = QtGui.QFileDialog.getSaveFileName(self, "Save Graph", "Graph.png", "Image (*.png)")
        if not file_name:
            return
        pixmap = QtGui.QPixmap(self.render_area.width(),self.render_area.height())
        self.render_area.render(pixmap)
        pixmap.save(file_name, "PNG")


    def plot(self):
        self.render_area.x_min = float(eval(str(self.x_min_line_edit.text())))
        self.render_area.x_max = float(eval(str(self.x_max_line_edit.text())))
        self.render_area.y_min = float(eval(str(self.y_min_line_edit.text())))
        self.render_area.y_max = float(eval(str(self.y_max_line_edit.text())))
        self.render_area.y1 = str(self.y1_line_edit.text())
        self.render_area.y2 = str(self.y2_line_edit.text())
        self.render_area.y3 = str(self.y3_line_edit.text())
        
        # keep for zoom reset
        self.x_min = self.x_min_line_edit.text()
        self.x_max = self.x_max_line_edit.text()
        self.y_min = self.y_min_line_edit.text()
        self.y_max = self.y_max_line_edit.text()
        self.render_area.replot()
        
        self.modified = True


    def zoom_in(self):
        "Narrow the plotted section by half"

        center_x = (self.render_area.x_min + self.render_area.x_max) / 2
        center_y = (self.render_area.y_min + self.render_area.y_max) / 2
        range_x = (self.render_area.x_max - self.render_area.x_min)
        range_y = (self.render_area.y_max - self.render_area.y_min)


        self.render_area.x_min = center_x - (range_x / 4)
        self.render_area.x_max = center_x + (range_x / 4)
        self.render_area.y_min = center_y - (range_y / 4)
        self.render_area.y_max = center_y + (range_y / 4)

        self.parameter_entries_repopulate()
        self.render_area.replot()
        self.modified = True


    def zoom_out(self):
        "Double the plotted section"

        center_x = (self.render_area.x_min + self.render_area.x_max) / 2
        center_y = (self.render_area.y_min + self.render_area.y_max) / 2
        range_x = (self.render_area.x_max - self.render_area.x_min)
        range_y = (self.render_area.y_max - self.render_area.y_min)


        self.render_area.x_min = center_x - range_x
        self.render_area.x_max = center_x + range_x
        self.render_area.y_min = center_y - range_y
        self.render_area.y_max = center_y + range_y

        self.parameter_entries_repopulate()
        self.render_area.replot()
        self.modified = True
        

    def zoom_reset(self):
        self.x_min_line_edit.setText(self.x_min)
        self.x_max_line_edit.setText(self.x_max)
        self.y_min_line_edit.setText(self.y_min)
        self.y_max_line_edit.setText(self.y_max)
        self.plot()


    def parameter_entries_repopulate(self):
        self.x_min_line_edit.setText(QtCore.QString("%1").arg(self.render_area.x_min))
        self.x_max_line_edit.setText(QtCore.QString("%1").arg(self.render_area.x_max))
        self.y_min_line_edit.setText(QtCore.QString("%1").arg(self.render_area.y_min))
        self.y_max_line_edit.setText(QtCore.QString("%1").arg(self.render_area.y_max))
        
    
    def evaluate(self):
        d = EvaluateDialog(self)
        d.show()


    def closeEvent(self, event):
        if self.modified: 
            reply = QtGui.QMessageBox.question(self, self.tr("Save Graph"),
                                               self.tr("Save the changes to '%1'?").arg(self.current_file), QtGui.QMessageBox.Yes,
                                               QtGui.QMessageBox.No,
                                               QtGui.QMessageBox.Cancel)
            if reply == QtGui.QMessageBox.Yes and self.save():
                event.accept()
            elif reply == QtGui.QMessageBox.No:
                event.accept()
            else:
                event.ignore()
        else:
            event.accept()


class MainWindow(QtGui.QMainWindow):
    def __init__(self):
        QtGui.QMainWindow.__init__(self)

        self.setWindowTitle("Lybniz")
        self.setWindowIcon(QtGui.QIcon(":/images/lybniz.png"))
        self.mdi_area = QtGui.QMdiArea()
        self.connect(self.mdi_area, QtCore.SIGNAL("subWindowActivated(QMdiSubWindow *)"), self.update_menus)
        self.window_mapper = QtCore.QSignalMapper(self)
        self.connect(self.window_mapper, QtCore.SIGNAL("mapped(QWidget *)"), self.mdi_area, QtCore.SLOT("setActiveSubWindow (QWidget *)"))
                     
        self.sub_window_maximized = True
        self.setCentralWidget(self.mdi_area)

        self.create_actions()
        self.create_menus()
        self.create_tool_bars()
        self.create_status_bar()

        self.read_settings()

        child = self.create_MdiChild()
        child.new_file()
        child.y1_line_edit.setText("sin(x)")
        child.plot()
        child.modified = False

        if self.sub_window_maximized:
            child.showMaximized()
        else:
            child.show()

    def new_file(self):
        child = self.create_MdiChild()
        child.new_file()
        child.show()
        

    def create_MdiChild(self):
        child = Graph(self)
        self.mdi_area.addSubWindow(child)
        return child


    def closeEvent(self, event):
        self.mdi_area.closeAllSubWindows()
        if self.mdi_area.activeSubWindow():
            event.ignore()
        else:
            self.write_settings()
            event.accept()


    def about(self):
        QtGui.QMessageBox.about(self, self.tr("About Lybniz"),
            self.tr("<b>Lybniz</b><br><br>"
                    "<em>Qt Version 0.9</em><br><br>"
                    "Simple function graph plotter<br>"
                    "by Thomas F&uuml;hringer and Sam Tygier<br>"
                    "http://lybniz2.sourceforge.net/"))


    def create_actions(self):
        self.new_act = QtGui.QAction(QtGui.QIcon(":/images/filenew.png"), self.tr("&New"), self)
        self.new_act.setShortcut(self.tr("Ctrl+N"))
        self.new_act.setStatusTip(self.tr("Create a new file"))
        self.connect(self.new_act, QtCore.SIGNAL("triggered()"), self.new_file)

        self.open_act = QtGui.QAction(QtGui.QIcon(":/images/fileopen.png"), self.tr("&Open"), self)
        self.open_act.setShortcut(self.tr("Ctrl+O"))
        self.open_act.setStatusTip(self.tr("Open a file with function definitions"))
        self.connect(self.open_act, QtCore.SIGNAL("triggered()"), self.file_open)

        self.save_act = QtGui.QAction(QtGui.QIcon(":/images/filesave.png"), self.tr("&Save"), self)
        self.save_act.setShortcut(self.tr("Ctrl+S"))
        self.save_act.setStatusTip(self.tr("Save the function definitions"))
        self.connect(self.save_act, QtCore.SIGNAL("triggered()"), self.save)

        self.save_as_act = QtGui.QAction(QtGui.QIcon(":/images/filesaveas.png"), self.tr("Save &As"), self)
        self.save_as_act.setShortcut(self.tr("Shift+Ctrl+S"))
        self.save_as_act.setStatusTip(self.tr("Save the function definitions under a new name"))
        self.connect(self.save_as_act, QtCore.SIGNAL("triggered()"), self.save_as)

        self.save_graph_act = QtGui.QAction(self.tr("Save &Graph"), self)
        self.save_graph_act.setShortcut(self.tr("Ctrl+G"))
        self.save_graph_act.setStatusTip(self.tr("Save the graph bitmap."))
        self.connect(self.save_graph_act, QtCore.SIGNAL("triggered()"), self.save_graph)

        self.close_act = QtGui.QAction(self.tr("&Close"), self)
        self.close_act.setShortcut(self.tr("Ctrl+F4"))
        self.close_act.setStatusTip(self.tr("Close file"))
        self.connect(self.close_act, QtCore.SIGNAL("triggered()"), self.file_close)

        self.exit_act = QtGui.QAction(QtGui.QIcon(":/images/fileclose.png"), self.tr("E&xit"), self)
        self.exit_act.setShortcut(self.tr("Ctrl+Q"))
        self.exit_act.setStatusTip(self.tr("Exit Lybniz"))
        self.connect(self.exit_act, QtCore.SIGNAL("triggered()"), self, QtCore.SLOT("close()"))

        self.plot_act = QtGui.QAction(QtGui.QIcon(":/images/apply.png"), self.tr("P&lot"), self)
        self.plot_act.setShortcut(self.tr("Ctrl+L"))
        self.plot_act.setStatusTip(self.tr("Plot the graph"))
        self.connect(self.plot_act, QtCore.SIGNAL("triggered()"), self.plot)

        self.evaluate_act = QtGui.QAction(QtGui.QIcon(":/images/run.png"), self.tr("&Evaluate"), self)
        self.evaluate_act.setShortcut(self.tr("Ctrl+E"))
        self.evaluate_act.setStatusTip(self.tr("Evaluate functions"))
        self.connect(self.evaluate_act, QtCore.SIGNAL("triggered()"), self.evaluate)

        self.zoom_in_act = QtGui.QAction(QtGui.QIcon(":/images/viewmag+.png"), self.tr("Zoom &In"), self)
        self.zoom_in_act.setShortcut(self.tr("Ctrl++"))
        self.zoom_in_act.setStatusTip(self.tr("Zoom into a quarter of the area"))
        self.connect(self.zoom_in_act, QtCore.SIGNAL("triggered()"), self.zoom_in)

        self.zoom_out_act = QtGui.QAction(QtGui.QIcon(":/images/viewmag-.png"), self.tr("Zoom &Out"), self)
        self.zoom_out_act.setShortcut(self.tr("Ctrl+-"))
        self.zoom_out_act.setStatusTip(self.tr("Zoom out to see four times the area"))
        self.connect(self.zoom_out_act, QtCore.SIGNAL("triggered()"), self.zoom_out)

        self.zoom_reset_act = QtGui.QAction(QtGui.QIcon(":/images/viewmag1.png"), self.tr("Zoom &Reset"), self)
        self.zoom_reset_act.setShortcut(self.tr("Ctrl+R"))
        self.zoom_reset_act.setStatusTip(self.tr("Reset zoom factor to original"))
        self.connect(self.zoom_reset_act, QtCore.SIGNAL("triggered()"), self.zoom_reset)

        self.connect_points_act = QtGui.QAction(self.tr("&Connect Points"), self)
        self.connect_points_act.setCheckable(True)
        self.connect_points_act.setChecked(True)
        self.connect(self.connect_points_act, QtCore.SIGNAL("triggered()"), self.connect_points)

        self.decimal_act = QtGui.QAction(self.tr("Decimal"), self)
        self.decimal_act.setCheckable(True)
        self.decimal_act.setChecked(True)
        self.connect(self.decimal_act, QtCore.SIGNAL("triggered()"), self.decimal)

        self.radians_act = QtGui.QAction(self.tr("Radians"), self)
        self.radians_act.setCheckable(True)
        self.connect(self.radians_act, QtCore.SIGNAL("triggered()"), self.radians)

        self.style_group = QtGui.QActionGroup(self)
        self.style_group.addAction(self.decimal_act)
        self.style_group.addAction(self.radians_act)

        self.tile_act = QtGui.QAction(self.tr("&Tile"), self)
        self.tile_act.setStatusTip(self.tr("Tile the windows"))
        self.connect(self.tile_act, QtCore.SIGNAL("triggered()"), self.mdi_area.tileSubWindows)

        self.cascade_act = QtGui.QAction(self.tr("&Cascade"), self)
        self.cascade_act.setStatusTip(self.tr("Cascade the windows"))
        self.connect(self.cascade_act, QtCore.SIGNAL("triggered()"), self.mdi_area.cascadeSubWindows)

        self.separator_act = QtGui.QAction(self)
        self.separator_act.setSeparator(True)

        #self.help_act = QtGui.QAction(QtGui.QIcon(":/images/help_index.png"), self.tr("&Help"), self)
        #self.help_act.setStatusTip(self.tr("How to use Lybniz"))

        self.about_act = QtGui.QAction(QtGui.QIcon(":/images/info.png"), self.tr("&About"), self)
        self.about_act.setStatusTip(self.tr("Show the About Box"))
        self.connect(self.about_act, QtCore.SIGNAL("triggered()"), self.about)


    def create_menus(self):
        self.file_menu = self.menuBar().addMenu(self.tr("&File"))
        self.file_menu.addAction(self.new_act)
        self.file_menu.addAction(self.open_act)
        self.file_menu.addAction(self.save_act)
        self.file_menu.addAction(self.save_as_act)
        self.file_menu.addAction(self.save_graph_act)
        self.file_menu.addAction(self.close_act)
        self.file_menu.addSeparator()
        self.file_menu.addAction(self.exit_act)

        self.graph_menu = self.menuBar().addMenu(self.tr("&Graph"))
        self.graph_menu.addAction(self.plot_act)
        self.graph_menu.addAction(self.evaluate_act)
        self.graph_menu.addAction(self.zoom_in_act)
        self.graph_menu.addAction(self.zoom_out_act)
        self.graph_menu.addAction(self.zoom_reset_act)
        self.graph_menu.addAction(self.connect_points_act)

        self.style_menu = self.graph_menu.addMenu(self.tr("Scale Style"))
        self.style_menu.addAction(self.decimal_act)
        self.style_menu.addAction(self.radians_act)
        
        self.window_menu = self.menuBar().addMenu(self.tr("&Window"))
        self.connect(self.window_menu, QtCore.SIGNAL("aboutToShow()"), 
                     self.update_window_menu)

        self.help_menu = self.menuBar().addMenu(self.tr("&Help"))
        #self.help_menu.addAction(self.help_act)
        self.help_menu.addAction(self.about_act)
        
    def update_window_menu(self):
        self.window_menu.clear()
        self.window_menu.addAction(self.tile_act)
        self.window_menu.addAction(self.cascade_act)
        self.window_menu.addAction(self.separator_act)
        
        windows = self.mdi_area.subWindowList()
        
        self.separator_act.setVisible(len(windows) != 0)
        
        i = 0

        for child in windows:
            if i < 9:
                text = self.tr("&%1 %2").arg(i + 1).arg(QtCore.QFileInfo(child.current_file).fileName())
                
            else:
                text = self.tr("%1 %2").arg(i + 1).arg(QtCore.QFileInfo(child.current_file).fileName())

            i += 1
            
            action = self.window_menu.addAction(text)
            action.setCheckable(True)
            action.setChecked(child == self.mdi_area.activeSubWindow())
            self.connect(action, QtCore.SIGNAL("triggered()"), self.window_mapper, QtCore.SLOT("map()"))
            self.window_mapper.setMapping(action, child)

    def update_menus(self):
        has_mdi_child = (self.mdi_area.activeSubWindow() is not None)
        self.save_as_act.setEnabled(has_mdi_child)
        self.save_act.setEnabled(has_mdi_child)
        self.save_graph_act.setEnabled(has_mdi_child)
        self.close_act.setEnabled(has_mdi_child)
        self.plot_act.setEnabled(has_mdi_child)
        self.evaluate_act.setEnabled(has_mdi_child)
        self.zoom_in_act.setEnabled(has_mdi_child)
        self.zoom_out_act.setEnabled(has_mdi_child)
        self.zoom_reset_act.setEnabled(has_mdi_child)
        self.graph_menu.setEnabled(has_mdi_child)
        self.window_menu.setEnabled(has_mdi_child)

        if has_mdi_child:
            self.connect_points_act.setChecked(self.mdi_area.activeSubWindow().render_area.connect_points)

            if self.mdi_area.activeSubWindow().render_area.scale_style == "rad":
                self.radians_act.setChecked(True)
            else:
                self.decimal_act.setChecked(True)

    def create_tool_bars(self):
        self.tool_bar = self.addToolBar(self.tr("Tools"))
        self.tool_bar.addAction(self.new_act)
        self.tool_bar.addAction(self.open_act)
        self.tool_bar.addAction(self.save_act)
        self.tool_bar.addSeparator()
        self.tool_bar.addAction(self.plot_act)
        self.tool_bar.addAction(self.evaluate_act)
        self.tool_bar.addSeparator()
        self.tool_bar.addAction(self.zoom_in_act)
        self.tool_bar.addAction(self.zoom_out_act)
        self.tool_bar.addAction(self.zoom_reset_act)

    def create_status_bar(self):
        self.statusBar().showMessage(self.tr(""))

    def read_settings(self):
        settings = QtCore.QSettings("Free Software", "Lybniz")
        self.move(settings.value("pos", QtCore.QVariant(QtCore.QPoint(200, 200))).toPoint())
        self.resize(settings.value("size", QtCore.QVariant(QtCore.QSize(400, 400))).toSize())
        self.sub_window_maximized = settings.value("sub_window_maximized", QtCore.QVariant(True)).toBool()

    def write_settings(self):
        settings = QtCore.QSettings("Free Software", "Lybniz")
        settings.setValue("pos", QtCore.QVariant(self.pos()))
        settings.setValue("size", QtCore.QVariant(self.size()))
        if self.mdi_area.activeSubWindow() is None:
            settings.setValue("sub_window_maximized", QtCore.QVariant(self.sub_window_maximized))
        else:
            settings.setValue("sub_window_maximized", QtCore.QVariant(self.mdi_area.activeSubWindow().isMaximized()))


    def file_open(self):
        file_name = QtGui.QFileDialog.getOpenFileName(self, "Open Graph", "", "Lybniz (*.lyb)")        
        if not file_name:
            return
        
        file = QtCore.QFile(file_name)
        if not file.open(QtCore.QIODevice.ReadOnly):
            return
            
        xml = file.readData(5000)
        file.close()
        
        doc = QtXml.QDomDocument("Lybniz")
        ok, errorStr, errorLine, errorColumn = doc.setContent(xml, True)
        if not ok:
            QtGui.QMessageBox.information(self.window(), self.tr("File Open Error"),
                                          self.tr("Parse error at line %1, column %2:\n%3").arg(errorLine).arg(errorColumn).arg(errorStr))
            return
        
        root = doc.documentElement()
        if root.tagName() != "graph":
            QtGui.QMessageBox.information(self.window(), self.tr("Lybniz"), self.tr("The file is not a Lybniz file."))
            return
        
        child = self.create_MdiChild()
        child.new_file()
        child.file_open(root, file_name)
        child.show()

    def save(self):
        self.mdi_area.activeSubWindow().save()

    def save_as(self):
        self.mdi_area.activeSubWindow().save_as()

    def save_graph(self):
        self.mdi_area.activeSubWindow().save_graph()
        
    def file_close(self):
        self.mdi_area.activeSubWindow().close()

    def plot(self):
        self.mdi_area.activeSubWindow().plot()
        
    def evaluate(self):
        self.mdi_area.activeSubWindow().evaluate()

    def zoom_in(self):
        self.mdi_area.activeSubWindow().zoom_in()

    def zoom_out(self):
        self.mdi_area.activeSubWindow().zoom_out()


    def zoom_reset(self):
        self.mdi_area.activeSubWindow().zoom_reset()


    def connect_points(self):
        self.mdi_area.activeSubWindow().render_area.connect_points = not self.mdi_area.activeSubWindow().render_area.connect_points
        self.mdi_area.activeSubWindow().plot()

    def decimal(self):
        self.mdi_area.activeSubWindow().render_area.scale_style = "dec"
        self.mdi_area.activeSubWindow().plot()

    def radians(self):
        self.mdi_area.activeSubWindow().render_area.scale_style = "rad"
        self.mdi_area.activeSubWindow().plot()


class EvaluateDialog(QtGui.QDialog):
    "Evaluate a given x for the three functions"
    
    def __init__(self, parent):
        QtGui.QDialog.__init__(self, parent)
        				
        self.setWindowTitle("Evaluate")

        layout = QtGui.QGridLayout()

        layout.addWidget(QtGui.QLabel("x ="), 0, 0)
        layout.addWidget(QtGui.QLabel("<font color='blue'>y1 = </font>"), 1, 0)
        layout.addWidget(QtGui.QLabel("<font color='red'>y2 = </font>"), 2, 0)
        layout.addWidget(QtGui.QLabel("<font color='green'>y2 = </font>"), 3, 0)

        self.x_line_edit = QtGui.QLineEdit()
        
        self.y1_line_edit = QtGui.QLineEdit()
        self.y2_line_edit = QtGui.QLineEdit()
        self.y3_line_edit = QtGui.QLineEdit()
        self.y1_line_edit.setReadOnly(True)
        self.y2_line_edit.setReadOnly(True)
        self.y3_line_edit.setReadOnly(True)
        layout.addWidget(self.x_line_edit, 0, 1)
        layout.addWidget(self.y1_line_edit, 1, 1)
        layout.addWidget(self.y2_line_edit, 2, 1)
        layout.addWidget(self.y3_line_edit, 3, 1)
        self.setLayout(layout)
        #self.connect(self.x_line_edit, QtCore.SIGNAL("textEdited(QString *)"), self.update_ys) # did not work (inexplicably) so I chose below
        self.connect(self.x_line_edit, QtCore.SIGNAL("returnPressed()"), self.update_ys)
        
    def update_ys(self, t=None):
        for e in ((self.parentWidget().y1_line_edit.text(), self.y1_line_edit), (self.parentWidget().y2_line_edit.text(), self.y2_line_edit), (self.parentWidget().y3_line_edit.text(), self.y3_line_edit)):
            try:
                x = float(eval(str(self.x_line_edit.text())))
                e[1].setText(str(eval(str(e[0]))))
            except:
                if len(e[0]) > 0:
                    e[1].setText("Error: %s" % sys.exc_value)
                else:
                    e[1].setText("")


qt_resource_data = "\
\x00\x00\x06\x37\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x05\xfe\x49\x44\x41\x54\x58\x85\xd5\x57\x5b\x6c\x14\x55\
\x18\xfe\xfe\x33\x67\x76\x67\x2f\xdd\xed\x14\x29\x8b\xdb\x14\xca\
\x75\x6d\xb1\x0f\x5a\x2f\x14\x8d\xd7\x07\x84\x40\x45\x82\xb7\xc4\
\x18\x78\x51\xdf\x95\x68\xd0\xc4\x44\x13\x8d\x97\xf8\x46\xf4\xc5\
\x07\x9a\x78\x8d\x46\x14\x85\x04\x35\x44\x43\x6b\xc5\xf8\xd0\xa6\
\xb5\x68\xcb\x22\xb2\xa5\x94\xb2\xdb\xbd\x74\x67\x76\x2e\xe7\xf8\
\x30\xdd\xd2\xee\x45\x11\x35\xc6\x2f\xf9\xb3\x3b\x67\xfe\xf9\xff\
\x6f\xce\x7f\x39\xff\x90\x94\x12\xff\x25\xd8\x7f\xea\xfd\x7f\x4b\
\x60\xdf\x1b\xbd\xa4\x6f\xd8\xb1\x24\xb8\x76\x6b\x47\x70\xed\xd6\
\x0e\x7d\xc3\x8e\x25\xfb\xde\xe8\xa5\x2b\xb1\x45\x97\x9b\x03\x4b\
\x3a\x77\x86\xf2\x85\xe2\x03\x52\xca\x6d\x2c\x1c\xbf\x9b\xc2\xb1\
\xb0\x1a\x6a\x02\x00\xd8\xb3\x69\xc8\xc2\x64\x41\x14\x52\x5f\x12\
\xd1\x67\x0d\xe1\xe0\xfb\x17\x07\x3f\x9a\xfd\x47\x08\xac\xec\x7e\
\x44\x9d\x98\x9c\x7e\x0c\x5a\xd3\xb3\xe1\x6b\xb6\x2f\xbb\xf6\xfa\
\x4d\x58\xb5\x22\x0e\x46\x80\xed\x7a\x3a\xaa\x02\x08\x09\x9c\xfa\
\x75\x02\x43\x3f\x1e\x47\x61\xe4\xe0\x79\x98\xe9\x17\xaf\x8e\x5d\
\xf5\xd6\xe9\xbe\x5e\xfb\x8a\x09\x84\xd7\x6f\x5b\x56\x72\xe4\x87\
\xbe\xf6\x87\x6e\xed\xba\x6d\x3b\xd6\xc5\x43\xb8\x90\x67\x08\xf9\
\x81\xe5\x11\x20\xec\x07\x88\x80\xbc\x09\x4c\x64\x81\x9c\x09\x2c\
\x8f\x0a\xfc\x3c\x61\xe0\x87\x63\x9f\xc0\x1a\x79\xf7\x5b\x3f\xa7\
\x5d\x85\x93\x9f\x9d\xff\xcb\x04\x82\x6b\xb6\xae\x76\xd4\xe8\xb1\
\xe0\xc6\xa7\x5a\x36\xdf\xb2\x01\xa6\xab\x22\x1e\x05\xae\x89\x79\
\x4e\x6b\x41\x4a\x60\x28\x05\x4c\xe4\x81\x10\xb7\x71\xe4\xdb\x21\
\x14\xbf\x7b\xed\x2c\xb7\x67\x6e\x2f\x8e\x7d\x31\x7e\xd9\x04\xa2\
\xed\xdb\x75\xc3\x51\xfb\xb4\x3b\x5e\x4e\x6c\xb9\xa9\x0d\x69\x83\
\xe3\xe6\x95\xde\x1b\x97\xc1\x2a\x48\x88\x05\x66\x72\x26\xf0\xc3\
\x19\xa0\x51\x73\xf0\xc5\x77\xa7\x60\x1e\x7b\x66\x34\xc0\xed\xee\
\xec\xc8\xa7\x99\x4a\x5f\x35\xab\xa0\x68\xd8\x6f\xfa\x6f\xda\x9b\
\xe8\x6a\x5f\x81\x8b\x06\x47\x7b\xcc\x73\x58\xb4\x3c\x89\x45\x80\
\x27\xef\x5a\x2c\xb9\x91\x83\xf3\xf7\x39\x03\x36\x2c\x07\xf2\x16\
\x47\x57\x47\x1b\x7c\x37\x3e\x99\x28\x1a\xf6\x9b\xb5\x7c\x55\x11\
\xf0\xaf\xbe\x67\x93\x12\xef\xbe\x3f\xb2\xf2\x06\x34\x47\x54\x28\
\x04\xf8\x38\x50\xb4\x2f\x49\xc9\xa9\x36\x34\x7d\x2e\xb9\x48\x87\
\x31\x20\xaa\x01\xcd\x11\x15\xd1\xd6\xeb\xa0\xc4\xbb\xef\xf7\xaf\
\xbe\x67\x53\xe5\x73\xbc\x72\x41\x48\x7a\x2e\xd0\xb9\x07\xed\x31\
\xc2\x6f\x33\x84\x15\x3a\x60\x56\xe4\xf1\x4f\x93\xc0\xa3\x2f\x7d\
\x05\xcb\xcc\xcf\xaf\x59\xfe\x0e\xa8\x15\x7a\x4d\x21\x60\x68\x02\
\x68\x8f\x11\x06\x3a\xf7\xc0\x98\xe8\x7f\x0e\xc0\xe6\xba\x04\x1a\
\x12\x3d\x3a\x6b\x5a\x7f\xa7\x12\x89\x23\xa2\x79\xd9\x3d\x63\x78\
\xf7\x34\x15\x50\xe6\xf6\x8b\x11\x10\x5d\x7b\xd7\x22\x67\xa6\x0d\
\x4c\x17\xbc\x10\x48\x09\x84\x35\x20\xe4\xf3\xfe\x47\x42\x2a\x94\
\x48\x1c\xac\x69\xfd\x9d\x0d\x89\x1e\x3d\x3f\x7a\x70\x3e\x17\x16\
\x11\x28\x59\xd6\x66\x35\xbe\x51\xd5\x03\xde\x36\xab\x1c\xc8\x5b\
\x9e\x00\x80\x4f\x01\x02\x1c\xe8\x6a\x05\x9e\xdf\xb2\xf8\x6d\x1f\
\x7d\xfe\x23\x9c\xd5\x77\xce\x5f\x67\x4c\x80\x13\x20\x09\x28\xb9\
\x04\x3d\x00\x58\xf1\x8d\x6a\x69\x68\x74\x33\x80\x77\xcb\x7a\x15\
\x39\x20\x3b\x99\xbe\x06\x01\x15\x70\x84\xe7\x70\xa1\x00\x80\xe1\
\x78\x31\xae\x84\x9d\xfd\xad\x4a\x9f\x31\xef\xd7\x11\x40\x40\x05\
\x98\xbe\x06\x80\xec\xac\x1b\x02\x29\xb1\x94\x34\x1d\x9c\x79\xcc\
\xca\x4e\x2b\xc1\x6b\xd4\x0e\xa3\xfa\xfa\x34\xf7\x0c\x69\x3a\xa4\
\xc4\xd2\xba\x04\xca\x70\x85\x17\xef\x7a\x0d\xa7\x16\x01\x22\xaf\
\x25\xd7\x82\x94\x9e\xcd\x9a\xb6\x2a\x8c\x5c\x90\x66\x06\xa6\xd3\
\x06\xc6\xea\x1b\xe4\x35\xd6\xd9\x5c\xb9\xd6\x82\xed\x02\xa6\x03\
\x48\x33\x03\x22\x5c\xa8\x4b\x00\xa0\x41\x91\x19\x83\x61\x5f\x07\
\xd7\xf5\xb2\xb8\x26\x81\x3a\x3b\x50\x2f\x04\xa6\x05\x18\x36\x20\
\x32\x63\x00\x68\x70\x11\xf1\x85\x17\x7e\x9f\xef\x88\x93\xea\xb7\
\x67\x2d\x20\x6b\x56\x27\x61\x59\xd4\x3f\xc8\x81\x5a\x92\x35\x81\
\x59\x0b\x70\x52\xfd\xb6\xdf\xe7\x3b\x52\x97\x40\x7e\xf4\x60\x46\
\xa4\x4f\x7e\x2d\xf2\x29\x4c\x64\x01\x5b\x78\xa5\x58\x29\x4a\x9d\
\x10\xd4\xd2\xb5\x85\x77\x52\x8a\x7c\x0a\x22\x7d\xf2\xeb\x85\x3d\
\xa0\x8a\x80\x67\x48\xbe\x60\x0d\xbe\x8d\x19\x03\x38\x93\xae\xb3\
\x03\x35\x08\x50\x9d\x1d\x38\x93\xf6\x9a\x99\x35\xf8\x36\x18\xc9\
\x17\xaa\xfc\x55\x2e\x94\xc6\x0f\x1f\x77\x53\x7d\x1f\x58\xa9\x01\
\x8c\x4d\x03\x93\xb9\x6a\xa3\x7f\x54\x86\x0b\x65\x32\x07\x8c\x4d\
\x03\x56\x6a\x00\x6e\xaa\xef\x83\xd2\xf8\xe1\xe3\x95\xcf\xd5\xcc\
\xdb\x60\x40\x7d\xdc\x18\x78\xa5\x93\x82\xaf\x26\x06\x68\x15\x42\
\x3e\xef\x04\x2c\xe3\x6c\x06\x78\xe2\xf5\xaf\x60\x19\x97\xce\x02\
\x33\x98\x40\x74\xc1\xce\x4c\xe6\x80\x81\xd3\xc0\xec\x85\x53\x28\
\x0d\xbc\x32\x1a\x0c\xa8\x8f\xd7\xf2\xf5\xa7\x03\x89\x7f\xe3\xd3\
\x2d\x81\x58\x27\x36\xb5\x01\xed\xcb\xeb\xf7\x86\x32\xa4\x04\x46\
\xce\x01\xc7\x93\x80\x31\x39\x88\x52\xff\xcb\x7f\x7d\x20\x29\xa3\
\x3c\x92\xa9\xed\x0f\xdf\xea\x5b\x77\x1f\x96\xe9\x1a\x6e\x5c\x01\
\xb4\x35\x55\xf7\x02\xc7\x05\x92\x69\xe0\xfb\x5f\x81\xf3\x19\x13\
\xd6\xcf\x1f\xc3\x1e\x79\xe7\xca\x47\xb2\x32\x16\x0e\xa5\x7c\xed\
\xbd\xcb\x78\x4b\x37\xfc\xd1\x38\x9a\xc3\x97\x26\xa4\x42\x09\x98\
\x2a\x00\xa5\x6c\x0a\xce\xd9\x3e\x38\xbf\x7c\xf2\xf7\x86\xd2\xfd\
\xfb\xf7\xd3\x89\x13\x27\x68\x7c\x7c\x9c\x65\xb3\x59\x32\x0c\x83\
\xce\x97\x1a\x1b\x0c\x34\x3c\x28\x40\x5b\x58\x43\xfc\x36\x0a\xc5\
\x42\xa4\x79\x63\xb9\x34\xd3\x90\xb3\x93\xb3\x22\x9f\xfa\x86\x41\
\x7e\xae\x21\xf7\x5e\xb3\x9a\x29\x68\x9a\x26\x1a\x1b\x1b\x65\x4b\
\x4b\x8b\x48\x24\x12\x72\xd7\xae\x5d\xb2\xa3\xa3\xa3\x9a\xc0\xde\
\xbd\x7b\xd9\xd1\xa3\x47\x95\xa9\xa9\x29\x6e\x18\x86\xea\x38\x0e\
\x77\x5d\x57\x15\x42\x70\x29\x25\x97\x52\x72\x00\x5c\x4a\xa9\x48\
\x29\x39\x88\x14\xa8\x91\x26\xc9\x43\xcd\x00\x40\xce\xec\x14\xec\
\x5c\x06\x52\xba\x00\x5c\xc6\x98\x0b\xc0\x21\x22\x9b\x88\x1c\x22\
\x72\x14\x45\xb1\x39\xe7\x8e\xa6\x69\xb6\xae\xeb\x4e\x57\x57\x97\
\x73\xe0\xc0\x01\xc1\x87\x87\x87\x31\x3c\x3c\x4c\x33\x33\x33\xcc\
\x34\x4d\xee\xba\xae\xe2\xba\x2e\x9f\x73\xae\xce\x09\x97\x52\xaa\
\xf0\xaa\x86\x43\x4a\x0e\x2b\x6b\xa1\x34\x73\x4e\x4a\x49\xe4\x39\
\x0c\x02\x70\x00\x38\x42\x08\x87\x88\xd8\xdc\x0b\x32\xc6\x18\xb9\
\xae\xf7\x11\x51\x2a\x95\x64\x2e\x97\x13\xc9\x64\x92\xf5\xf7\xf7\
\x8b\x45\x21\xe8\xed\xed\xa5\x43\x87\x0e\x51\x32\x99\x64\x99\x4c\
\x86\x15\x8b\x45\xc5\xb2\x2c\xe6\x38\x8e\x22\x84\x60\x42\x08\x05\
\x00\x09\x21\x08\x5e\x0f\x29\xd7\x84\x04\x20\x89\x48\x12\x91\x60\
\x8c\x09\xc6\x98\x50\x14\xc5\xf5\xf9\x7c\xee\x5c\x28\x44\x6b\x6b\
\xab\xe8\xe9\xe9\x11\xbb\x77\xef\x9e\x77\x7a\xd9\x9f\x66\xff\x16\
\x7e\x07\x5a\x75\xd2\xa4\x61\x23\x98\x44\x00\x00\x00\x00\x49\x45\
\x4e\x44\xae\x42\x60\x82\
\x00\x00\x07\x6f\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x07\x36\x49\x44\x41\x54\x58\x85\xc5\x97\x6d\x6c\x53\xd7\
\x19\xc7\xff\xe7\x9c\x7b\xe3\xd8\x71\x9c\x38\xb6\x43\x03\x79\x19\
\x4b\xa3\x32\x18\xc9\x58\x8a\x16\x08\xb4\x20\x8d\x79\x03\xb4\x42\
\x04\x64\x43\xea\x26\x0d\x8a\xd4\xa9\x1f\x28\x63\x42\xda\x8b\x26\
\x44\x61\x9d\xb6\x49\x30\x69\xea\x68\x18\x94\x74\x03\x85\x69\x12\
\x9a\xe6\x01\x12\x5d\x17\xd4\xc4\x09\xe0\x05\x87\x94\x10\x92\x39\
\x8e\x21\x29\x71\x9d\x38\x8e\xe3\xb7\x7b\xef\x39\x67\x1f\xe2\x64\
\xd0\xb2\x29\x09\xd3\xf6\x97\x9e\x7b\x75\x75\x9f\x73\x9f\xdf\x7d\
\xce\x73\xde\x88\x94\x12\xff\x4f\x29\x73\xf4\x63\x00\xac\x00\x72\
\x00\xc4\x01\xa4\xff\x17\x00\x44\xd7\xf5\x15\xa1\x50\xe8\x6b\x5d\
\x5d\x5d\x5f\x0a\x06\x83\xe6\x4c\x26\x03\xa7\xd3\x89\x55\xab\x56\
\x0d\x54\x55\x55\xfd\xd5\x6e\xb7\xbf\x07\x60\xea\x69\x00\xc8\x93\
\xba\x20\x1a\x9d\x28\xe8\xef\xbf\xb7\xbf\xa5\xa5\x65\x8d\xdf\xef\
\xbf\xb9\x63\xc7\x0e\xa5\xae\xae\x2e\xdf\x6c\xb6\xd0\x60\x70\x30\
\xd1\xdc\xdc\x1c\x4f\xa7\xd3\x4b\x1b\x1b\x1b\xad\x9b\x36\x6d\xfa\
\xb9\xdd\x6e\xef\x5c\x30\x81\x94\xf2\x31\xeb\xef\xef\xb7\x9f\x3d\
\x7b\xf6\xdd\xad\x5b\xb7\x9e\x08\x85\x42\x4d\x86\xc1\x13\x86\x61\
\x48\x5d\x37\xa4\xa6\xe9\x52\xd3\x74\x99\xc9\x68\x32\x99\x4c\xb5\
\x1e\x3c\xf8\xfd\xef\x1d\x3e\x7c\xf8\xca\xe0\xe0\xe0\x96\x4f\x7e\
\x67\xae\xf6\x58\x06\xfa\xfa\xfa\x68\x6b\xeb\xb5\x5f\xb5\xb6\xfe\
\x2d\x7e\xe6\xcc\x99\xcd\x84\xd0\xea\x47\xdf\x3f\xde\x18\x90\x52\
\x66\xae\xbe\x77\xf5\x47\x1f\xf6\xdc\x76\xef\xdc\xb9\xf3\xf5\xaa\
\xaa\xaa\x9e\xf9\x26\x80\x3e\xfa\xd0\xde\xde\xbe\xb1\xa3\xc3\xeb\
\x68\x6a\x3a\xf5\x65\x21\x50\x2d\xa5\x04\xa5\x04\x94\x52\x50\x4a\
\xc1\x18\x85\xa2\x30\x30\xc6\x40\x08\xc0\xb9\x30\x6d\x78\x71\xe3\
\x1b\x8a\xa2\xfe\xe5\x83\x0f\xda\x0e\x85\xc3\x61\xfa\xef\x02\xcd\
\x09\xe0\xce\x9d\xde\x97\xf7\xec\xd9\x9b\x2f\x81\xe7\x25\x24\x08\
\x21\x90\x52\x42\x48\x01\x29\x05\x84\x90\x10\x62\x3a\x23\x8c\x31\
\x98\x4c\x2a\x40\x60\xda\xb3\xe7\x95\x86\xee\xee\x6e\x5c\xbf\x7e\
\xe3\xb9\x05\x03\x1c\x3d\xfa\xa6\x39\x1e\x9f\xac\x5c\xb9\xb2\x7a\
\x23\x37\x04\x20\x25\x84\x98\x0e\x2a\x85\x04\x17\x12\x22\x0b\x33\
\x0d\x22\x20\x25\x60\xca\x51\x41\x19\x5d\xb7\x64\x49\x69\xa0\xb7\
\xb7\xf7\x85\x05\x03\x0c\x0c\xdc\x7b\xa6\xb4\xb4\xac\x4c\x08\x69\
\xe1\x9c\x43\xd7\x0d\x08\x21\xc0\xb9\x00\x17\x02\x82\xcf\xd8\xbf\
\xc0\x66\xee\x39\xaa\x8a\x2f\xd6\xd6\x7e\x26\x1c\x0e\x7f\x76\xc1\
\x00\xc1\x60\x90\x51\xca\x2c\x06\xe7\xe0\x9c\x23\xa3\xe9\xd0\xb4\
\x69\x08\x91\x85\x78\xcc\x38\xcf\xc2\x71\x08\x21\x91\x97\x67\x79\
\x6e\x7c\x7c\x7c\xae\x13\xdb\xac\x66\x1b\x3c\x78\x30\x1c\x1e\x1d\
\x1d\x85\xaa\x28\xd0\x74\x1d\xdc\xe0\x48\x1a\x1c\xaa\xaa\x40\x55\
\x18\x66\xc6\x02\x01\xf0\xa9\x99\x43\x4a\x8c\x8e\x8e\x96\x4f\x4e\
\xc6\x2e\x2c\x38\x03\xfd\xfd\x7d\x93\x7e\xbf\xdf\x1f\x89\x44\x66\
\xab\x9d\x10\x20\x9d\xd6\x10\x8f\x27\x91\x4e\x65\xa0\xeb\x06\x0c\
\x83\x83\x1b\xd9\x0c\x18\x1c\x99\x8c\x86\x58\x6c\x4a\xeb\xea\xea\
\x12\x81\xc0\x60\xdb\x82\x01\x00\x20\x14\x1a\x6a\xfe\xb3\xc7\xc3\
\x0d\x9d\x83\x0b\x01\x02\x32\x3d\xec\x14\x06\x2e\x04\xd2\xe9\x0c\
\x12\xc9\x34\x92\xc9\x14\x12\x89\x34\xa6\xa6\x52\x48\xa7\x35\x3c\
\x1c\xfd\xe8\xe3\xce\x8e\x8e\xe1\x68\x74\xdc\xf7\x54\x00\x53\x53\
\xf1\x0b\x7f\xb8\xd0\xe2\xf5\xfd\xbd\x0b\xba\x66\x40\xd3\x0d\x68\
\x9a\x0e\x5d\x37\xc0\x0d\x9e\x1d\x11\x8f\x14\xa5\x14\x48\x24\x12\
\xd1\xa6\xb7\x4f\xca\x9e\x9e\x9e\x9f\x04\x02\x01\x63\xbe\x00\x9f\
\x5a\x0b\xf2\xf3\x6d\x95\xf5\xf5\xeb\xae\xec\xdb\xf7\x6a\xe5\xca\
\xea\x1a\x10\x32\xdd\xeb\x52\x02\x90\x80\x9c\xbe\x40\x42\x22\x1a\
\x8d\x26\x4f\x9f\x7e\xfb\xe3\xa1\x60\x40\x8f\x44\x22\xcf\xfb\x7c\
\xbe\xd8\x53\x03\x00\x40\x6e\x6e\xee\xb3\xd5\xd5\x5f\x38\xed\x76\
\x6f\x5e\xb7\x66\xed\x5a\xe2\x72\x16\xc3\x6c\xb6\x00\x04\xd0\x75\
\x1d\x13\xd1\x28\xee\xde\xbd\x13\xf5\x78\xfe\x94\x52\x55\x96\xcf\
\x18\xcb\xd7\x34\xed\xe6\xd0\xd0\x90\xdb\xef\xf7\x8f\x3f\x35\x00\
\x00\xb8\x5c\xae\x5c\x4a\xd9\x37\x2a\x2b\x9f\x7d\xb5\xac\xbc\xa2\
\xc6\x66\xb3\x31\x55\xcd\x11\x93\x93\x31\x31\x3e\x16\x89\x8f\x8f\
\x47\xde\x4f\xa7\xd3\xc7\x4a\x4a\x4a\x56\x87\x42\xa1\x5f\x2f\x5d\
\xba\x34\xc7\x6c\x36\xf7\x04\x02\x01\xb7\xcf\xe7\x1b\x79\x6a\x80\
\x19\x9d\x3a\x75\x8a\x1c\x39\x72\xa4\xd4\xe5\x72\x55\x32\xc6\xf2\
\xad\xf9\xf9\x11\x55\x55\x6f\x5f\xbe\x74\x69\x76\x1f\xe0\x72\xb9\
\xb6\xbb\x5c\xae\x96\x65\xcb\x96\xa9\x56\xab\x75\xb0\xaf\xaf\xef\
\x2b\x9d\x9d\x9d\x03\xff\x15\x80\xb9\xca\xe1\x70\xec\x28\x2e\x2e\
\x3e\x5f\x5b\x5b\xab\x58\xad\xd6\x87\xb7\x6e\xdd\xfa\x6a\x47\x47\
\x87\x7f\xde\x00\x3f\xbb\x29\x2a\x32\x86\x58\xa9\x50\x9a\xb2\xa8\
\xd4\xbb\x7f\x15\x92\x73\x85\x70\x3a\x9d\x3b\x5d\x2e\xd7\xb9\xf5\
\xeb\xd7\x2b\x36\x9b\x6d\xc2\xeb\xf5\xbe\xd4\xd6\xd6\x76\x6d\x4e\
\x00\x47\x3b\x34\xfb\xc0\x48\xe4\x37\x2e\x55\xdb\x69\x55\x04\xe1\
\x12\x88\x6a\xca\x44\x92\x9a\x7f\x70\xea\xeb\xae\xb7\xe6\x0a\xe1\
\x70\x38\x1a\x17\x2d\x5a\xf4\x3b\xb7\xdb\xad\x14\x15\x15\xa5\x5a\
\x5b\x5b\x77\x5f\xbd\x7a\xf5\xe2\x7f\x04\x38\x7e\x3d\xc1\x6e\x04\
\xc6\xde\x5f\x5d\xa2\xac\xff\x7c\x69\x01\xd6\x94\xe5\xe2\xce\xfd\
\x31\xdc\x0a\x0b\xf4\x86\x53\x88\x70\xcb\xbe\xe6\x86\x45\x4d\xf3\
\x80\xf8\x66\x49\x49\xc9\xbb\xdb\xb6\x6d\x63\xc5\xc5\xc5\xfc\xca\
\x95\x2b\x7b\x3d\x1e\xcf\x3b\x4f\xf2\xa5\x00\xe0\xbb\x3f\xd5\x50\
\x5a\xa0\xae\x27\xb6\x67\xa0\xb1\x3c\xe4\xe5\x30\xc4\xc7\x46\x11\
\x4e\x11\x94\x2d\x59\x02\x18\x99\x37\x5f\xbf\x14\x35\xcd\x15\x60\
\x6c\x6c\xec\xfc\xc8\xc8\xc8\xb7\x3d\x1e\x8f\x48\x24\x12\xac\xb1\
\xb1\xf1\xb7\xbb\x76\xed\x3a\xf8\x44\x67\x29\x25\x76\xfc\x7e\xe8\
\xe4\x1f\xef\x19\xf2\x49\xea\x0a\x46\xe4\xae\xe6\x01\xb9\xeb\xfc\
\x83\xfa\xf9\xee\xf7\x0a\x0b\x0b\xbf\x55\x57\x57\xc7\x8f\x1f\x3f\
\x2e\x2f\x5e\xbc\x28\xf6\xee\xdd\xfb\xc6\x8d\x1b\x37\x1e\xf3\xa1\
\x00\x30\xc5\x55\x73\x5f\x8c\xa1\xb3\x37\x84\x7b\xf7\xc3\x00\x80\
\x8f\xc6\xe3\x68\xed\x1e\x44\xab\xef\x2e\x82\x91\x04\x62\xc9\x54\
\xee\x5c\x33\x30\xa3\x68\x34\xda\xdc\xdb\xdb\xfb\x8a\xc7\xe3\x11\
\xe1\x70\x98\xec\xde\xbd\xfb\x87\xe7\xce\x9d\x3b\xe1\xf5\x7a\x67\
\x7d\x14\x00\x88\x27\x12\x5d\x27\xba\xc4\xcb\xbf\x0c\xa7\xb1\xb9\
\x8a\xa0\xb9\x0c\xb8\x7c\xb3\x1f\xfb\x2e\x27\x00\x4b\x11\x16\x39\
\x0a\x79\x6e\x3a\x76\x7b\xbe\x00\x00\x30\x31\x31\x71\xba\xb0\xb0\
\x50\xb1\xd9\x6c\x6f\x0d\x0f\x0f\xd3\x81\x81\x81\x17\x0f\x1c\x38\
\x40\xbd\x5e\xaf\x00\xb2\x45\x58\xfb\x8b\x5b\x76\x51\xb0\xf8\xee\
\x87\x19\x67\xf1\xe7\x0a\x25\x1a\xf3\xfc\xf0\xdd\xbe\x8b\x8b\xf4\
\x05\x14\x38\x4b\x50\x9e\xea\x7b\x67\xe4\xa7\x1b\x5e\x13\x42\x10\
\x29\xe5\x8c\xcd\x06\x21\x84\x80\x10\x22\x09\x21\x92\x52\x2a\x19\
\x63\x52\x55\x55\x61\x32\x99\x84\xc5\x62\x91\x45\x45\x45\xa2\xbb\
\xbb\xfb\xbb\x0e\x87\xc3\x9d\x48\x24\xbe\x33\x3c\x3c\x1c\x9e\x6d\
\x5b\x53\x53\xa3\x3e\x7c\xf8\x50\xcd\xdd\xfc\xe3\x0d\xe5\x35\x6b\
\xcf\x4e\x52\x9b\x73\x7c\x22\x0e\x46\x25\x8a\x0b\xf2\x20\xa2\xf7\
\xdb\x7b\x4e\xbe\xb6\xdf\x18\x1d\x48\x61\xfa\x88\xc6\x30\x5d\xbc\
\x8f\xae\xa4\x22\x6b\x1c\x00\xa7\x94\x72\x00\x06\x00\x83\x52\x6a\
\x10\x42\x0c\x4a\xa9\xc1\x18\x33\x54\x55\xd5\xf3\xf2\xf2\x8c\xc5\
\x8b\x17\x1b\xdb\xb7\x6f\xe7\xa4\xa2\xa2\xc2\x34\x36\x36\x66\xd5\
\x34\xcd\x21\x8b\x2a\x2a\x5c\xab\x5f\xda\x56\x54\x54\x58\xc5\x39\
\xd7\x1e\xfc\xa3\xef\x66\xe2\x7a\x4b\x1b\x25\x24\x89\xe9\xf3\xa0\
\x32\x03\x21\xa5\x64\x52\x4a\x92\xfd\x73\x3e\x13\x7c\x26\x30\x21\
\xc4\x20\x84\xe8\x59\x08\x3d\x0b\xa1\x2b\x8a\x62\xe4\xe4\xe4\xe8\
\x76\xbb\x5d\xaf\xaf\xaf\x37\x88\x94\x12\x5b\xb6\x6c\xa1\xed\xed\
\xed\xd6\x74\x3a\x5d\xc0\x39\xb7\x4a\x29\x4d\xd9\xd4\xea\x94\xd2\
\x24\x63\x2c\x49\x29\xe5\x42\x08\x0a\x80\x4a\x29\xc9\x27\xfb\x9a\
\x10\x22\x00\x48\x4a\xa9\x20\x84\x08\x4a\x29\x67\x8c\x09\x45\x51\
\x78\x6e\x6e\xae\xb0\x58\x2c\xc2\xe9\x74\x8a\xe5\xcb\x97\x8b\x86\
\x86\x06\xe9\x76\xbb\xe5\x6c\x0d\xcc\xe8\xd0\xa1\x43\xe4\xda\xb5\
\x6b\x2c\x16\x8b\x51\x93\xc9\x24\xcb\xcb\xcb\xc5\xb1\x63\xc7\xf8\
\x8a\x15\x2b\x16\x52\x7f\x73\xd2\x3f\x01\xc0\x71\x6d\xc5\x8a\xda\
\xca\x92\x00\x00\x00\x00\x49\x45\x4e\x44\xae\x42\x60\x82\
\x00\x00\x08\x2b\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x07\xf2\x49\x44\x41\x54\x58\x85\xe5\x97\x6d\x50\x54\xd7\
\x19\xc7\xff\xf7\xee\xbd\xbb\x77\x61\xd9\x45\x91\x2c\x54\x50\x5e\
\x04\xab\xa4\xa0\x20\x10\xc4\xb1\x1a\xa7\xbe\xcc\xc8\x5a\x0b\x32\
\xb5\x5f\x12\xdb\x91\xda\xda\x4e\x92\xaa\x14\x4b\x66\x8c\x19\x71\
\xc6\x64\xec\x54\x4d\xa7\xf1\xa5\xa6\x8e\xa3\x91\x26\x06\x09\x45\
\x9d\x44\x7c\x21\x49\xa5\x80\xa9\x94\x6c\x79\xed\x0a\xbb\xf2\x8e\
\x0b\x2c\xbb\xec\xde\x7b\xcf\x3d\xa7\x1f\x04\x07\x15\x8c\x6d\xd2\
\xc9\x87\x9e\x99\xe7\xcb\x3d\x77\x9e\xff\xef\x3c\xe7\x39\xcf\x79\
\x0e\xc7\x18\xc3\x37\x39\xf8\x6f\x54\xfd\xff\x0b\xe0\xc0\x2f\x2c\
\x38\x52\xb4\xea\xb1\xef\x8c\xb1\xff\xb9\xd1\xdf\xbf\x3a\x9b\xba\
\xda\x1b\xd9\xf0\x3d\xc6\x3e\x2e\xcd\x9f\x3c\xc7\x7d\x1d\x49\x68\
\xb3\xd9\x92\x18\x63\xbb\x29\xa5\x44\x10\x84\xdd\xe5\xe5\xe5\x3d\
\x13\x73\x67\xce\x9c\x99\x9b\x3e\x7f\xde\xf5\x79\x89\xf3\x62\xf8\
\x9e\x7f\x01\xbd\xce\x3e\xf4\x8d\x46\x22\x7f\x0b\x03\xf0\xd5\x00\
\x6c\x36\x5b\xa6\x24\x49\x45\x89\x89\x89\x36\x8b\xc5\xc2\x57\x55\
\x55\x81\x6a\xda\x08\xc7\xf3\xfb\x96\x2f\x5f\x7e\x30\x24\xc4\x14\
\x9f\x92\xb2\xe8\xa6\xd1\x68\x9c\x25\xf8\x3d\x48\x15\xbd\x18\x6b\
\xfa\xfb\x27\x41\x3f\xd9\xbb\x7c\xc2\x87\xf0\x5f\x0a\xaf\xb2\x58\
\x2c\x45\xe9\xe9\xe9\xab\x4c\xc1\xc1\xdc\x3d\xb7\x1b\x8b\x17\x2f\
\x46\x5d\x5d\x1d\xfa\xfb\xfa\x2c\x9a\xa6\xbd\x69\x19\xf9\x4b\xf8\
\x92\x15\xfb\x7f\x26\x13\x84\xf8\x7c\x3e\x84\x85\x3d\x83\x2f\x6a\
\x6e\xde\xba\xe9\xe8\xc8\xdf\x3a\xc9\xd7\x53\x47\xa0\xa4\xa4\x84\
\xab\xad\xad\xcd\xb1\x5a\xad\xbb\xb3\xb2\xb2\x32\x17\x2d\x5a\xc4\
\x29\x8a\x82\x8a\x8a\x0a\x28\xb2\x8c\xd9\x51\x51\xe0\x79\x1e\xe7\
\xce\x9d\x43\x56\x82\x8a\x3d\xbf\x4c\x82\xb7\xdb\x81\x3b\x96\x62\
\x04\x99\x9f\x41\xcf\x5d\xc7\x67\xfd\xdd\x8e\x35\x2f\x14\xec\xf4\
\x4d\xf6\xfb\xa5\x00\x05\x05\x05\xc2\xe0\xe0\x60\x7e\x54\x54\xd4\
\xee\x35\x6b\xd6\x3c\x9b\x96\x96\x06\x45\x51\xe0\xf3\xf9\x30\x36\
\x36\x86\xfa\xfa\x7a\x34\x37\x37\x63\x74\x74\x14\xf9\xf9\xf9\x68\
\xba\x7e\x10\x05\x9b\xbe\x05\xe2\xa8\x84\xe2\xb9\x07\xa7\x61\x2b\
\x5c\xc6\xef\x9e\xbf\xd3\x72\xfb\x47\x2f\x15\x1d\x50\x1e\xf5\x3f\
\xed\x16\x6c\xde\xbc\xd9\xa0\x28\xca\x0b\x7a\xbd\xbe\x70\xfe\xfc\
\xf9\xf1\x7b\xf6\xec\x01\xcf\xf3\xf0\x78\x3c\xf0\xfb\xfd\xd0\x34\
\x0d\x84\x10\xc4\xc6\xc6\xc2\x6e\xb7\xc3\xe7\xf3\x81\xde\x7d\x17\
\xdb\x36\x45\x40\x69\xfb\x00\x7e\x8f\x0f\xdd\x5d\x06\x6a\x57\x6e\
\x17\xb3\xd8\x98\x37\x5e\x2a\x3a\x40\xa7\xd2\x79\x0c\x20\x2f\x2f\
\xcf\xc4\x71\x5c\x81\xc1\x60\xf8\x55\x48\x48\xc8\x6c\x42\x08\x9c\
\x4e\x27\xda\xdb\xdb\x91\x90\x90\x00\x4a\x29\x28\xa5\x20\x84\x80\
\x10\x02\x9d\x4e\x87\x98\xb9\x73\xf0\xbd\xf9\x9d\x58\x91\x02\xa8\
\x2d\x17\xe0\x1d\x51\x71\xc7\x65\x44\xc9\x8d\x68\x4e\x91\x42\xae\
\x7e\x5c\x52\x34\xa5\x38\x30\x45\x21\x62\x8c\xed\x31\x9b\xcd\x07\
\x75\x3a\xdd\xec\x09\x11\x59\x96\x71\xe1\xc2\x05\x28\x8a\x02\x4a\
\x29\x34\x4d\x7b\x10\x01\x4a\x02\xd8\x94\xd1\x87\x55\x69\x3c\x48\
\xeb\x7b\x18\x1e\x24\xb0\x3b\x4c\xd8\x75\x29\x12\x5f\x74\x7a\xb9\
\xfe\xfe\xfe\xa2\xe9\x37\x78\x0a\x00\x45\x51\x0e\xcb\xb2\x2c\x13\
\x42\xa0\xaa\x2a\x88\xaa\x82\x10\x82\xfa\xfa\x7a\xb4\xb4\xb4\x3c\
\x24\xee\xee\x77\x22\x51\xba\x8c\x30\xdc\x86\xe6\x28\xc7\x60\x3f\
\x87\x7a\x87\x19\x85\x95\xe1\x70\x0d\x04\xa0\x12\x02\xb7\xdb\xbd\
\x61\xed\xda\xb5\x0b\x9f\x1a\xa0\xa2\xa2\xc2\xe5\xf7\xfb\xff\x34\
\xb1\x7a\x75\x1c\x24\xe0\xf7\xe3\xc3\xf2\x72\xc8\xb2\x0c\x87\xc3\
\x81\x0f\x4a\xff\x88\x44\xe3\x65\x84\xa9\x75\xd0\x9c\x55\x18\xe8\
\xe6\x51\xdd\x1a\x86\xdf\x5c\x9c\x89\xc1\x91\xc0\x7d\x70\x55\x85\
\xcf\xeb\xe5\x3b\x3b\x3b\xa7\x8d\xc2\x94\xa7\xc0\x66\xb3\xc5\xea\
\x74\xba\x16\xaa\x69\x22\x19\x5f\x2d\x21\x04\x3c\xcf\x23\x31\x31\
\x11\x3d\x77\x3e\xc7\x9b\xaf\x24\xc0\x4a\x6e\x81\xde\x6b\x42\x7f\
\x17\x8f\xca\xd6\x48\xbc\x75\x43\x8f\xb4\x25\xe9\xc8\xcf\xcf\x87\
\xd3\xe9\xc4\xeb\x7b\xf7\x82\x10\x02\x83\xc1\x40\xd2\x33\x32\x12\
\x2f\x5e\xbc\x78\xe7\xa9\x00\x00\x60\xfd\xfa\xf5\x27\x29\xa5\x5b\
\xb4\x71\xf1\x09\x90\xb9\xe1\x14\x07\x77\x7c\x1b\x21\x43\xd7\x40\
\x86\x5d\xe8\xef\xe2\x71\xb6\x31\x0a\xcd\x34\x03\x2f\xbe\xb8\x05\
\x0b\x16\x2c\xb8\x5f\xe3\x39\x0e\xdb\xb7\x6f\xc7\xd5\xaa\x2a\x10\
\x42\x10\x13\x13\xf3\x87\xd6\xb6\xb6\x9f\x3f\x35\x40\x4e\x4e\x4e\
\x82\xa2\x28\x4d\xaa\xaa\xea\x08\x21\xd0\x08\xc1\xc2\x39\x1c\xde\
\x78\x25\x01\x62\xd7\x25\xc8\x23\x83\xe8\xeb\x12\x70\x93\xe4\x20\
\x32\xe3\xc7\x48\x4e\x4e\x06\x00\x0c\x0d\x0d\xc1\xeb\xf5\x22\x32\
\x32\x12\xcd\xcd\xcd\xb0\xe5\xe4\x80\x10\x02\xb3\xd9\xec\xcf\x5e\
\xb6\x2c\xbe\xb2\xb2\xb2\x67\xb2\xce\xb4\xd7\x71\x45\x45\x45\x1b\
\xa3\xf4\xdd\x89\xf0\x67\x25\x89\xf8\xed\x8e\x78\x88\xce\x32\x8c\
\xb9\xdd\xe8\xbe\x2b\x21\xb0\xe4\x77\x58\x57\x70\x18\xa9\xa9\xa9\
\xf0\x78\x3c\xe8\xee\xee\x86\xc1\x60\x40\x6c\x6c\x1c\xdc\x8a\x09\
\x69\x69\x69\x58\xf9\xfc\xf3\x00\x00\x8f\xc7\x63\x6c\x6f\x6f\x7f\
\xf9\x51\x9d\x27\xf6\x03\xa2\x5e\xbf\x9f\x52\x4a\xd7\x65\x06\x61\
\xdf\x4f\x23\xc1\x39\xde\xc3\xa8\x7b\x0c\x3d\x7d\x66\x88\xab\xcf\
\x62\xce\x92\x4d\x18\x1d\x1d\x45\x57\x57\x17\x24\x49\x42\x5c\x5c\
\x1c\xcc\x66\x33\x86\x03\x02\x3e\x6a\x35\xa1\x79\xd0\x84\xc2\xc2\
\xc2\x07\xfe\x06\xfa\xfb\xb7\xe5\xe4\xe4\xcc\x9c\xac\xf1\xc4\xcb\
\xa8\xb2\xb2\xb2\xe9\xc3\xb7\x73\xeb\x56\xa7\x6a\x99\x5a\xdb\xfb\
\x18\xba\xc7\x63\x70\xd4\x8a\xd0\xef\x97\x42\x33\x58\xe1\x72\xb9\
\x10\x16\x16\x06\xab\xd5\x0a\x45\x51\x1e\xdc\xf1\x66\xbd\x8c\xec\
\x78\x23\x2c\x92\x8a\xe8\xd4\x54\x44\x46\x46\x62\x68\x68\x88\x99\
\xcd\xe6\x1b\x8c\x31\x13\x00\xf7\x97\x02\xb8\x2e\xe7\xf1\x11\x51\
\x33\xdf\x5a\xfd\x9d\xd1\x4c\xd2\x7e\x05\x83\x03\x02\xfe\xe9\x32\
\xe1\xa2\x67\x05\xf6\xcd\x4e\x02\x21\x04\x11\x11\x11\x90\x65\xf9\
\x7e\x41\xa2\x14\x8a\xa2\x20\x28\x28\x08\x3c\xcf\xe3\x59\xa3\x0c\
\x45\x51\x70\xfc\xf8\x29\x26\x88\x62\xe5\x73\x59\x59\x7b\xaf\x5e\
\xbd\x5a\xff\xa8\xce\x94\x00\x55\x97\xce\xf3\x59\xd1\xd1\x7f\xa6\
\x23\xf6\x5c\xb5\xe3\x0a\x06\x7a\x75\xa8\xeb\x0c\xc5\xbe\x2b\x16\
\x8c\x78\xab\xf1\x83\x1f\x7e\x8e\xec\xec\x6c\x78\xbd\xde\xfb\xc5\
\x8a\x10\x48\x92\x04\x51\x14\x21\x08\x02\x54\x55\xc5\xd9\xb3\x67\
\x59\x59\x59\xd9\x25\x00\xaf\x75\x76\x76\xd6\x4d\xb7\xd0\x29\x01\
\x12\xe2\x66\x6c\x67\x01\x43\xae\x7a\xb7\x06\x83\x7d\x46\xb4\xb9\
\x67\xdc\x7a\xfd\x8a\x29\xd5\xe3\xf1\x72\x84\x10\x1c\x3e\x74\x08\
\xd9\xd9\xd9\x90\x65\x19\x82\x20\xc0\x60\x30\x40\x92\x24\x28\x8a\
\x82\x53\xa7\x4e\xb1\xb2\xb2\xb2\xcb\x00\x5e\xbb\x76\xed\x5a\xed\
\x13\x76\x18\xc0\x14\x49\xf8\x8f\xdb\xf5\xb1\x86\xd0\xa4\xfd\x30\
\x2e\x80\x3b\x90\x82\x76\x6f\xec\xfb\x7f\xd5\xd6\x2d\xd3\x09\xfa\
\x4b\x13\x27\xa2\xba\xba\x1a\x9f\x7d\xfa\x29\xc2\xc3\xc3\x11\x12\
\x12\x02\x51\x14\xf1\xce\xc9\x93\x6c\xc3\x86\x0d\xd7\x4f\x9f\x3e\
\xbd\xba\xb9\xb9\x39\xaf\xb7\xb7\xf7\x76\x6e\x6e\x2e\x6f\xb7\xdb\
\x9f\x4c\x30\xb9\x41\xac\xae\xbe\xcc\xdd\x75\x39\x3f\xf2\x8c\x8c\
\xb0\xc6\x86\xda\x86\xb7\x5f\xdd\x90\x17\x1a\x1a\x6a\x09\x0e\x0e\
\x0e\x4f\x49\x49\xb1\x45\x44\x44\xd0\x50\x8b\x85\x99\x82\x83\x59\
\x66\x46\x06\xab\xa9\xa9\x61\x87\x0e\x1d\x62\x4b\x97\x2e\xad\x89\
\x8f\x8f\xdf\x2a\x08\x42\xe6\xb8\xa5\x0b\x82\xb0\x48\x14\xc5\x05\
\x06\x83\x61\xae\xc9\x64\x0a\x8b\x8e\x8e\x96\x76\xed\xda\xc5\x3d\
\xda\xb0\x3e\x54\x88\x76\xee\xdc\xf1\x72\xe2\xbc\xd8\x42\x53\x10\
\xb7\xe7\xd7\xc5\x25\xe7\xdd\xee\x11\x41\xd3\x34\x03\x63\xcc\xc0\
\x18\xd3\xcf\x89\x8e\x3e\xd1\xdb\xdb\x9b\x45\x34\x0d\x3a\x9d\x0e\
\x33\x42\x43\x6b\x0d\x92\xf4\x4e\x67\x67\x67\x13\x63\x4c\x37\x1e\
\x51\x1e\x00\xe3\x38\x8e\x70\x1c\xa7\x00\x50\x00\xc8\x3c\xcf\xfb\
\xf4\x7a\xfd\x70\x5c\x5c\x9c\xa7\xa1\xa1\x41\x7d\x2c\x07\x56\xae\
\x5c\xa9\x8f\x08\x37\xab\x3d\x1d\x8d\x0b\xaf\x7c\xd2\xa0\xfa\xfd\
\x7e\x3d\x63\x8c\x07\xc0\x8d\x1b\x44\xbd\xfe\xa8\x24\x49\xcf\x19\
\x83\x82\xfe\x26\x49\xd2\xf1\x8e\x8e\x0e\x3b\x00\x8e\xe3\x38\x9e\
\xe3\x38\x32\xf1\x1f\x00\x06\x80\x02\xd0\xc6\x8d\x01\xe0\x28\xa5\
\xc2\xc0\xc0\x80\x60\xb7\xdb\xd5\xa4\xa4\x24\x00\x53\x94\xe2\xa3\
\x47\x8f\x72\xc7\x8e\x1d\xd3\x75\x75\x75\xe9\x03\x81\x80\x9e\x10\
\x22\x6a\x9a\x26\x32\xc6\xf4\x8c\x31\x21\x21\x21\x21\xa9\xb5\xb5\
\xb5\x6d\x1c\xee\x21\xc0\x49\xe2\x0c\x00\xe5\x79\x5e\x03\xa0\xf2\
\x3c\x1f\x10\x04\xc1\x6f\xb1\x58\xc6\x36\x6e\xdc\x28\x1f\x39\x72\
\xe4\x41\x83\x32\xed\x5d\x50\x56\x56\xc6\x9d\x38\x71\x82\x77\x38\
\x1c\xba\xe1\xe1\x61\x41\x96\x65\x9d\xa6\x69\x02\xa5\x54\x47\x29\
\xe5\x19\x63\x3a\xc6\x18\x87\x87\x13\x99\x71\x1c\x47\x39\x8e\xd3\
\x78\x9e\xd7\x44\x51\x54\x8d\x46\xa3\x6a\xb5\x5a\x49\x6e\x6e\x2e\
\x29\x2e\x2e\x7e\x4c\xec\x3f\x7a\x17\x94\x96\x96\x72\x8d\x8d\x8d\
\x70\xb9\x5c\xdc\xc0\xc0\x00\x37\xd1\x21\x01\xb8\x9f\x13\x33\x66\
\xb0\x59\xb3\x66\x21\x39\x39\x99\x6d\xdb\xb6\xed\xa9\x1c\x7f\x2d\
\x2f\xa3\xaf\x32\xfe\x0d\x1f\x36\x7e\x3d\xa2\xef\xc4\x87\x00\x00\
\x00\x00\x49\x45\x4e\x44\xae\x42\x60\x82\
\x00\x00\x04\xce\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x00\x06\x62\x4b\x47\x44\x00\xff\x00\xff\x00\xff\xa0\xbd\
\xa7\x93\x00\x00\x00\x09\x70\x48\x59\x73\x00\x00\x00\x48\x00\x00\
\x00\x48\x00\x46\xc9\x6b\x3e\x00\x00\x04\x40\x49\x44\x41\x54\x58\
\xc3\xcd\x97\xcf\x6b\x1b\x47\x14\xc7\x3f\x33\xb3\x3f\x24\x4b\xb5\
\x65\xd3\x60\x62\x47\x08\x42\x0e\xc5\x09\x46\xd0\x5b\x13\x88\xce\
\xbd\x05\x72\xe8\x21\xa7\x14\xfc\x4f\xe4\x1f\xe8\x7f\x11\xf7\x8f\
\xc8\x29\x07\x2b\xe9\xa5\x10\x08\x3e\xa4\xc7\x98\xb8\x88\x16\x3b\
\xd6\x56\x56\x22\x7b\x67\x67\x77\xa6\x87\x78\x97\xd5\x8f\xd8\x72\
\x02\x75\x1f\x3c\xe6\xcd\xec\xec\xbc\xef\xfb\xbe\x99\x37\xbb\xc2\
\x39\xc7\x55\x8a\xbc\x52\xef\xff\x07\x00\xde\xd7\x2e\x70\x70\x70\
\xc0\xc1\xc1\x81\x04\xd4\x59\x40\x79\x3b\x69\xcb\x9b\x37\x6f\x1e\
\xd6\xeb\xf5\xb1\x9c\x4f\x01\xd8\xda\xda\x3a\xbc\x71\xe3\xc6\x8a\
\x10\x82\x7a\xbd\x2e\xc2\x30\x44\x7c\x12\x72\x05\x10\x67\x86\x94\
\xb2\x18\x97\x52\x8e\x29\xc0\xd1\xd1\x11\x59\x96\x11\x45\x11\x0f\
\x1f\x3e\x5c\xdf\xdc\xdc\xfc\xeb\x5c\x00\x8b\x8b\x8b\xde\xce\xce\
\x8e\x7a\xf2\xe4\x09\xf7\xee\xdd\x43\x29\x55\x38\xcd\x65\x02\xc8\
\xb9\x0c\x75\xbb\x5d\xb6\x9f\x3e\x65\x71\x69\x89\x33\x46\xce\x4f\
\x41\x18\x86\xee\xd1\xa3\x47\xf8\xbe\x8f\xef\xfb\x45\x24\x17\x49\
\x7e\x9a\xca\xad\x73\x0e\xcf\xf3\xf8\xe1\xee\x5d\xa2\x28\x9a\xe9\
\x6f\x6a\x40\x08\xe1\x96\x97\x97\x49\xd3\xb4\x18\xd3\x5a\xb3\xb7\
\xb7\x77\xa1\x33\xe7\x1c\x41\x10\xd0\x6a\xb5\x8a\x7e\x1c\xc7\x04\
\x41\x80\x94\xd2\xcd\xc5\x00\xe0\x00\xd2\x34\xc5\x5a\x8b\x73\x0e\
\xa5\x14\xb7\x6e\xdd\x1a\x73\x94\x6b\x3e\xa7\xac\x71\x1c\x17\xb6\
\xd6\x9a\x34\x4d\xc9\xb2\x6c\xbe\x14\xe4\x11\xa5\x69\x4a\x1c\xc7\
\x08\x21\xc8\xb2\x8c\xb7\x6f\xdf\x4e\x45\x3b\x8b\x01\x29\x25\xcd\
\x66\x73\x0c\xcc\xa5\x00\x38\xe7\x9c\xb5\x96\x24\x49\xd0\x5a\x23\
\x84\xc0\x39\x57\x2c\x3a\x2b\xe2\x49\xd5\x5a\x17\xf3\x2e\x0d\xc0\
\x5a\xeb\xf2\x17\x72\x06\xac\xb5\xec\xef\xef\xcf\xc5\x80\x10\x82\
\xeb\xd7\xaf\x7f\x2e\x05\x72\x6e\x00\x39\xfa\x9c\x81\xd5\xd5\xd5\
\x73\xa3\x2e\x33\x53\xde\x03\x5f\xc4\x80\x31\x86\x2c\xcb\xd0\x5a\
\x53\xad\x56\xf9\x33\x5e\xc3\x5a\x47\xbb\x09\xd5\x60\x7c\xaf\xcc\
\x3a\x0d\x1f\x3e\x7c\xa0\xd7\xeb\x01\x90\x24\x09\xc6\x18\xac\xb5\
\xf3\x31\x90\x65\x19\x69\x9a\x92\x24\x09\x71\x1c\x63\xad\xa5\x77\
\x0c\x7f\x0f\x24\xab\x95\x7f\x50\xe9\x80\xf3\x44\x4a\x49\xbd\x5e\
\x2f\xfa\xc6\x98\x2f\x4b\x41\x92\x24\x24\x49\x42\x9a\xa6\xac\x7f\
\x13\xa3\xb5\xa2\x5a\x09\x58\xaa\x5e\xe3\x22\xf9\xf8\xf1\xe3\x2c\
\x00\xe2\x52\x00\xb4\xd6\x05\xb5\xcb\xf6\x1d\xdf\xaf\xc0\xf0\x08\
\x86\x25\xca\x27\x53\x51\x4e\xc3\x57\x33\x20\xa5\xe4\xd5\xab\x57\
\x54\xab\xd5\x0b\x8f\xdd\xe7\xf6\x83\xb5\x96\x5e\xaf\xc7\xca\xca\
\x4a\x0e\x40\xcc\x0d\xc0\x39\xc7\xc9\xc9\x09\xa3\xd1\xe8\xdc\xaa\
\x97\x8f\xe7\xcf\xca\xb6\x73\x8e\xa5\xa5\xa5\x32\x03\x17\x03\x28\
\x5f\xa5\xf9\x62\xd6\x5a\xd2\x34\xe5\xf0\xf0\x70\x6c\xce\xa4\x2a\
\xa5\xf0\x3c\xaf\xe8\xe7\xb7\x66\x14\x45\xf3\x03\xf0\x3c\xcf\x55\
\x2a\x95\x31\xe7\xf9\x91\xfc\xed\xe5\x4b\x36\x6e\xdf\xe6\xc1\x83\
\x07\x84\x61\xc8\xf3\xe7\xcf\xf9\xe3\xcd\x1b\x82\x30\x24\x08\x82\
\x31\xf5\x3c\x0f\xa5\x14\x6b\x6b\x6b\x97\x63\x60\x16\x00\x6b\x2d\
\x52\x4a\x86\xc3\x21\x02\xd8\xd8\xd8\xa0\x56\xab\xb1\xb3\xb3\x43\
\x3f\x8a\xa6\x9c\x87\x25\x3b\x5b\x5d\xc5\xf7\xfd\xbc\x0e\x5c\x0c\
\x40\x29\xe5\xc2\x30\x9c\xca\xa9\x10\x82\x38\x8e\x39\x1e\x0e\xd1\
\x5a\xa3\x94\x22\x8a\x22\xa2\x7e\x7f\x0a\x40\x99\x91\x96\x31\xd4\
\x6a\x35\x3c\x6f\xf6\xd7\xdf\x4c\x06\x26\x01\xe4\x35\x1e\x21\x30\
\x49\x52\x94\xd8\xe3\xc1\x80\x28\x8a\xf0\x7d\x7f\xa6\xf3\x20\x08\
\x30\xc6\xd0\x68\x34\xe6\x07\x30\xc9\x40\x0e\x42\x08\x81\x00\x12\
\x63\x80\x4f\x15\xf3\x34\x8e\x31\xc6\x60\x8c\xe1\xf4\xf4\x14\xa5\
\xd4\x14\x1b\x5a\xeb\x82\x81\xd1\x68\x34\x95\x82\xa9\xc2\xd0\x6c\
\x36\x7f\x35\xc6\x10\x86\x21\xe1\x59\x34\xb9\x2d\x84\x20\xd1\x1a\
\xad\x35\x71\x1c\xa3\xb5\x2e\xde\xcb\xbf\x21\x4e\x4e\x4e\x38\x3e\
\x3e\x26\x8a\x22\xfa\xfd\x3e\x5a\x6b\x16\x16\x16\xa8\xd5\x6a\xec\
\xed\xed\xcd\xbe\x8c\x76\x77\x77\x19\x0c\x06\xa2\xdb\xed\x2a\xe0\
\x97\x67\xcf\x9e\x7d\xb7\xb5\xb5\xf5\x93\xef\xfb\x63\xc5\xe6\xdb\
\x6b\xd7\x38\x7c\xff\x9e\x9f\x1f\x3f\x46\x48\x89\xa7\x14\x77\xee\
\xdc\x41\x29\x85\xf2\x3c\xbc\x52\xeb\x79\x1e\xca\xf3\x58\x5f\x5b\
\xa3\x52\xa9\x20\xa5\xe4\xf5\xeb\xd7\xd5\xf5\xf5\x75\xd9\x6e\xb7\
\x6d\xa3\xd1\x00\x40\xdc\xbf\x7f\x5f\xbe\x78\xf1\xc2\x07\x7c\x20\
\x00\x7c\xa5\xd4\x42\xb3\xd9\xfc\xf1\x6c\x4c\x02\xc2\x39\x27\x4b\
\xbb\x58\x38\xe7\x0a\x7b\x22\x28\x57\x52\x0b\x64\x67\x9a\xf6\xfb\
\xfd\xdf\x47\xa3\xd1\xbb\x56\xab\x35\xda\xde\xde\xd6\x9d\x4e\x07\
\x91\x47\xd7\xed\x76\x01\xc4\xee\xee\xae\x1c\x0c\x06\xb9\x33\xd9\
\xed\x76\x8b\x1f\x8b\x1c\x4c\x49\x3f\xeb\xb8\xd3\xe9\xd8\x1c\x40\
\xa3\xd1\xb0\xed\x76\x3b\x03\x5c\xa7\xd3\x19\xbb\x48\xc4\xe4\xc5\
\xf2\x5f\xcb\x95\xff\x1b\x5e\x39\x80\x7f\x01\x0c\xc0\xac\xdd\xf1\
\xb2\xe6\xc4\x00\x00\x00\x22\x7a\x54\x58\x74\x53\x6f\x66\x74\x77\
\x61\x72\x65\x00\x00\x78\xda\x2b\x2f\x2f\xd7\xcb\xcc\xcb\x2e\x4e\
\x4e\x2c\x48\xd5\xcb\x2f\x4a\x07\x00\x36\xd8\x06\x58\x10\x53\xca\
\x5c\x00\x00\x00\x00\x49\x45\x4e\x44\xae\x42\x60\x82\
\x00\x00\x06\xeb\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x00\x06\x62\x4b\x47\x44\x00\xff\x00\xff\x00\xff\xa0\xbd\
\xa7\x93\x00\x00\x00\x09\x70\x48\x59\x73\x00\x00\x00\x48\x00\x00\
\x00\x48\x00\x46\xc9\x6b\x3e\x00\x00\x06\x5d\x49\x44\x41\x54\x58\
\xc3\xb5\x97\x4b\x8c\x1c\x57\x15\x86\xbf\x73\xab\xab\xba\x67\x7a\
\x1e\xf6\x0c\x8e\x09\xc1\x8e\x63\x85\x0d\xca\x62\x44\x78\x6c\x90\
\x02\xc2\x22\x8a\x09\xf2\x10\x45\x11\x0b\x16\x23\x8c\x84\x44\x24\
\x96\x88\x2c\x10\x16\x59\x20\xb2\xf2\x0e\x16\x41\x1e\xef\x22\xb1\
\xf1\x9a\x55\x4b\x08\x89\x84\x10\x66\x04\x62\x11\xe2\xb8\x2d\x5b\
\x8c\x5f\xb1\xc7\xee\x71\x4f\xd7\xbd\xf7\xdc\xc3\xa2\xaa\x7b\x7a\
\x1e\xd8\x2d\x01\x23\x95\xea\x8e\xa6\x6a\xce\x77\xff\xff\x9c\xbf\
\xaa\xc4\xcc\xf8\x7f\xfd\x6c\x6d\x6d\xb9\x7e\xbf\xff\x44\x9e\xe7\
\x47\x45\x64\xfe\xe1\xc3\x87\x73\xe2\xb2\x85\x98\xdc\xd1\x99\xe9\
\xe2\x4f\x0b\x87\x0f\xfd\xa1\xf1\x5f\xfe\xf3\x63\x66\x76\x12\x38\
\x09\x3c\xa5\xaa\x9f\x6e\xb5\x5a\xc7\xfa\xfd\xfe\xd1\xa2\x28\x3e\
\x3b\x3d\x3d\xfd\xc4\xc2\xc2\x62\xb6\x5d\x2a\x41\x41\x9a\x2d\x7a\
\xfd\xc8\x83\x07\xf7\x29\xf2\xec\x1c\xf0\x78\x00\xef\x3d\x1b\x1b\
\x1b\xcf\x17\x45\xf1\xa5\x10\xc2\x73\xcd\x66\xf3\x59\x55\x7d\xa6\
\xdd\x6e\x3f\xb3\xb8\xb8\x98\x8b\x08\xce\xb9\x5d\xf7\xb4\xda\xf3\
\x6c\x0f\x12\xf7\x07\x89\xdb\x3d\x8f\x73\x02\x80\x99\xa1\x09\x0c\
\x50\x55\x00\x1e\x0b\x70\xf9\xf2\xe5\xb3\xc7\x8f\x1f\x7f\xbb\xdd\
\x6e\x3f\x56\x95\xab\x37\x4b\xb6\x4b\x45\x10\x9c\x03\x27\xe0\x1c\
\x98\x81\x50\x9d\x77\x36\x16\x8a\x89\x00\x9a\xcd\xe6\x91\x46\xa3\
\x81\x99\x21\x22\x8f\xbc\x76\xb3\x17\xc9\x32\x41\xc4\x70\x06\x99\
\x13\x0c\x70\x62\xec\x11\x09\xef\xfd\xdc\x44\x00\x40\x4a\xc9\x48\
\x29\x21\x22\x23\x88\xbd\x30\x95\xbc\x09\xc3\xe1\x04\xcc\x09\x66\
\x63\x20\x06\xc8\x50\x05\x21\xaa\xce\x4c\x04\xa0\xaa\x83\x94\x12\
\x29\x19\x22\xb6\x0b\xa2\xa6\xe3\xce\x9d\x3b\xdc\xb8\x71\x83\x5e\
\x3c\xc2\xcc\xcc\x1c\x49\x84\xcc\x0c\x73\x60\x54\x20\x59\xad\x80\
\xa6\xea\x9e\x18\x75\x6a\x22\x80\x94\xd2\x60\xb8\xbb\xaa\x78\x05\
\xe1\x6a\x90\x6b\xd7\xae\x71\xf3\xe6\x4d\x44\x04\x45\x89\x5a\xc9\
\x6d\x40\x56\x17\x1f\x82\x08\x86\xaa\x11\xa3\x11\x24\xb6\x26\x56\
\xc0\x80\xa4\x3b\x16\x88\x18\x56\xaf\x63\xd4\x5a\x1d\x50\x8b\x94\
\x21\x90\x37\x72\x9c\xd5\x16\x38\xc8\xea\xb5\x08\xc4\x64\x04\x4d\
\x64\x4c\xd8\x84\x29\xa5\x6d\x8c\x5d\x3d\x20\xc2\x68\x9d\x52\x22\
\x59\x42\x4c\x88\x1a\xa0\x2c\x31\x83\xbc\x91\x63\x56\x17\x77\x46\
\x66\x20\x0e\x34\x19\x51\x13\xd1\x52\x63\x22\x00\x33\xdb\x56\x55\
\xc4\xb9\x31\x80\x1d\x2b\x54\x15\x4b\x06\x62\xa8\x46\xe2\xa0\x24\
\x19\x98\x09\x8d\x46\x56\x43\x54\x4d\xe9\x0c\xa2\x32\x04\xc8\x26\
\x02\x28\x8a\xa2\xef\xbd\x27\xcf\x8b\x3d\x00\x15\x84\x6a\x22\xa5\
\x04\x08\xaa\x91\xd2\x0f\x46\x00\x4d\x9a\x58\xb6\x03\x91\x04\x54\
\x8d\x10\x8d\x9c\x24\x13\x01\x1c\x3e\x7c\x78\xdb\x7b\x4f\x96\x35\
\x0e\x00\x00\x1d\x9b\x10\x8d\x91\xed\x41\x1a\x01\x80\x60\x45\x93\
\x94\x09\xc9\x20\x73\x10\xd5\x88\x31\xa1\xce\x26\x03\x00\xfa\x31\
\xc6\xb1\x29\xd8\x39\x9c\x08\x49\x75\xd4\x1f\x31\x46\xfa\xdb\x3a\
\x9a\x75\xa3\x9a\xfb\xa2\x68\x62\x59\xb5\x0e\x6a\x84\x98\x88\x2e\
\x31\x69\x0f\x94\xaa\xba\x2f\x07\x44\x04\x13\xa9\x2c\x18\x36\x61\
\x0c\x94\xa5\xaf\x9a\x17\xd9\x51\x01\x21\xcf\x0b\x92\xab\x2d\xd0\
\x44\xa9\xc1\x4d\x04\x20\x22\x29\x99\xd5\x63\xc8\x3e\x15\x46\x16\
\x00\x31\x06\xbc\x2f\xa9\x2f\xac\x37\x50\xd9\x61\x08\x79\xa3\x31\
\xb2\xe0\x61\x39\x98\x2c\x09\x01\xb3\xb4\x3f\x88\xc6\x2d\x18\x4e\
\x41\x88\x4a\x08\x65\xfd\x37\x87\xe0\x80\x0a\xc6\x10\xac\x29\xc4\
\xba\x09\xa3\x0f\x13\x3f\x0b\x30\x1b\x7f\x16\xb0\xdb\x82\x54\x4f\
\x81\x54\x16\x04\x5f\x22\x08\x22\xf5\xd8\x3a\x87\x88\xa3\xd2\x08\
\xa2\x15\x84\x98\xb0\xd2\x3f\x5a\x81\x6e\xb7\x4b\xa7\xd3\x69\x6f\
\x6c\x6c\x1c\x59\xfe\xce\xab\xfb\x92\x70\x64\x81\xa6\x51\x12\x06\
\xef\x47\x16\x0c\x73\x83\x5d\x4d\xeb\x50\x71\x44\x55\x34\xe8\xee\
\x28\xee\x76\xbb\xd2\xe9\x74\xa6\x3a\x9d\x4e\xbb\xd3\xe9\xcc\x5e\
\xbd\x7a\x75\xbe\xdd\x6e\x1f\xfa\xd5\x5b\x6f\xfd\xbc\x28\x5a\xfb\
\xa6\xc0\xd5\x10\xe3\x53\x70\xb8\xed\x10\x99\xe3\xee\x83\x12\xef\
\x07\xf5\xb5\x95\x02\x95\x2a\x82\x34\x72\x54\x13\x31\x59\x15\xc5\
\x97\x2e\x5d\x2a\xce\x9d\x3b\xb7\xb0\xbe\xbe\x3e\x0b\x34\x81\x1c\
\x28\x80\xfc\xf4\xe9\xd3\xaf\xbc\xf8\xe2\xcb\x2f\xc4\xa8\xfb\x9a\
\xcf\x6a\x2b\x34\x25\x92\x19\x62\xc6\xdc\x54\x4e\xee\x22\xdb\x65\
\xc6\x66\xaf\x44\xa4\xea\x81\x9d\xfb\x1c\x99\x8b\xa4\xa4\xa4\x64\
\x55\x14\x9f\x38\x71\x22\xad\xaf\xaf\xcf\x01\xd3\xe3\xc5\xeb\x73\
\xd1\x9e\x2e\x18\x0c\x06\x94\x7e\x7f\x0e\x88\x18\x45\x5e\xd0\x4b\
\xbd\xea\x77\xc0\x30\x42\xf0\xbb\x15\x18\x8b\xf1\x66\xde\xae\x1e\
\xc7\xaa\x19\x80\x5b\x5a\x5a\x8a\x17\x2e\x5c\xb8\x5e\xef\x7e\x1a\
\x98\x1a\x9e\xcd\x2c\x0f\xc1\xe3\x9c\x43\x35\x1d\x70\x18\x4f\x7e\
\xe6\x29\x16\x17\x16\xab\x87\x92\x26\x2c\x25\x42\xf0\x15\x44\x28\
\x09\x7e\x40\xf0\x25\xde\x7b\x82\xf7\xa8\x2a\xaa\x4a\x8c\x2a\x00\
\x0e\x60\x65\x65\xa5\x7f\xe6\xcc\x99\x8d\xf1\xe2\xf5\xb9\x18\xbe\
\xc7\xa5\xba\xdb\x0f\x3a\x9a\xcd\x16\x29\x19\xc9\x12\x66\x46\x88\
\x9e\xe0\xcb\xb1\xc2\x25\x21\x0c\xf0\xbe\x44\x63\x65\x41\x8c\x71\
\x07\x00\xe0\xfc\xf9\xf3\xd7\x80\x6c\x0f\x44\x3e\x0c\x93\xe1\x6b\
\xd9\xfe\xa3\x22\xb4\xe1\xba\x0e\x2d\xef\x4b\xfc\x58\x61\xef\x3d\
\x21\x94\xa8\x06\x52\x52\x42\x88\xbb\xa3\x78\x79\x79\xf9\xcb\xc0\
\xa1\xf1\x1e\x30\xb3\x7c\x98\x03\xaa\x7a\x40\x0f\xd4\x0d\x89\x8c\
\xa2\xda\x30\xcc\x12\xaa\x81\xe0\xdd\xce\x14\xb8\xe1\xd8\x56\x0a\
\xf8\x10\x64\x73\x73\x73\x47\x81\xa5\xa5\xa5\x5b\xf5\xae\xc7\x15\
\x28\xaa\x97\x2b\x23\x46\x25\xc6\xca\xbf\xbd\x2a\x00\x24\xab\x14\
\x18\xaa\x65\xc6\xc8\x0a\xef\x4b\x82\xf7\xc4\xe8\x2d\xc6\x18\x1f\
\xdc\xdf\xdc\x0a\x5b\x37\x7f\x93\xe7\x85\x8c\x14\x58\x5d\x5d\xbd\
\x02\xfc\xf5\xe2\xc5\x8b\x5f\x1f\x57\xe0\x61\xc8\x29\x4b\x4f\x54\
\xad\xa2\xd7\x04\x97\x0c\x57\xef\xa8\xfa\x28\xd9\x51\x00\x03\xd5\
\x48\x35\x10\xa6\xbd\xad\xad\x5e\xda\xea\x3f\x40\x1a\xbd\x44\xd6\
\x6b\x76\xbb\xf7\x5e\x78\xfe\xf8\xcf\xce\xfe\xf4\xdc\x07\xfb\x92\
\x70\x75\x75\xf5\x2f\x6b\x6b\x6b\x9f\x5a\x5f\x5f\xff\xea\x10\xe2\
\x83\xeb\x2d\xb4\x0c\x9c\x6c\x2b\xc9\x09\x2e\x09\xc9\x19\x2e\x09\
\xce\x0d\x0b\x0b\x45\xde\x88\xb7\x6f\xdf\xfa\xf0\xe3\x2b\x97\xff\
\x71\x4f\x67\xbf\xd0\xbd\x7e\xe3\xce\xa0\xf4\x7d\x80\x2c\x6b\x60\
\x31\xc8\xb1\x85\xc6\x3b\x6f\xfe\xe2\x8d\xdf\x3e\xfd\xf4\xf1\x30\
\xac\xb9\x2f\x8a\x3b\x9d\xce\xef\x57\x56\x56\xa6\xd7\xd6\xd6\x9e\
\xed\xf7\xfb\xe9\x8b\x4f\xde\xd5\xbb\xf7\x7a\x59\x59\x2a\x2e\x09\
\xe6\x1c\xd3\xad\x66\x9a\x9b\x99\xfe\x68\x66\xb6\xfd\xfe\xec\xcc\
\xf4\x7b\x53\x53\xad\x77\xff\xf9\xe1\xdf\x2e\xbf\xfe\xfa\x0f\x8f\
\x01\x8d\x57\xbe\xff\x93\xf3\x83\xd2\x97\x00\x49\x55\x06\x9b\xff\
\xba\xfc\xe6\x1b\x3f\xfe\xe5\xb7\x5e\xfa\xe6\x47\x7b\xeb\xc9\x7f\
\xfa\x3a\xde\xdc\xdc\x74\xcb\xcb\xcb\xcb\xa7\x4e\x9d\xfa\xe4\xb5\
\xd7\xbe\xfb\x83\x46\xde\x3a\x34\x3f\x3f\xfb\xde\xdc\x6c\xfb\xdd\
\x46\x23\xfb\xb3\x88\xdc\x3b\x00\xbe\xb9\xbc\xbc\xfc\xb9\xaf\x2d\
\x9f\xfd\xf5\xad\xbb\xbd\x88\x0e\xd2\xb7\xbf\xf1\x95\xb7\x7f\x74\
\xf6\x7b\xef\xcc\xcf\xcf\x1f\x58\x48\xfe\xd7\x9f\xe7\xdd\x6e\xd7\
\xfd\xf1\xfd\xbf\xbf\x7a\xe5\xca\xc7\x47\x5f\x7e\xe9\xd4\xef\x96\
\x9e\xfb\xfc\x8d\x47\x5d\xff\x6f\xd3\x93\x88\x92\xce\x5a\xf7\x9c\
\x00\x00\x00\x22\x7a\x54\x58\x74\x53\x6f\x66\x74\x77\x61\x72\x65\
\x00\x00\x78\xda\x2b\x2f\x2f\xd7\xcb\xcc\xcb\x2e\x4e\x4e\x2c\x48\
\xd5\xcb\x2f\x4a\x07\x00\x36\xd8\x06\x58\x10\x53\xca\x5c\x00\x00\
\x00\x00\x49\x45\x4e\x44\xae\x42\x60\x82\
\x00\x00\x02\x15\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x01\xdc\x49\x44\x41\x54\x58\x85\xed\x96\x4f\x6e\x9b\x40\
\x14\xc6\x1f\x30\x33\xb2\x87\x20\x2f\xca\x3f\x23\x84\x17\x70\x02\
\x96\x91\xac\x6e\xa3\xf4\x14\xae\xe4\x1e\x20\xd7\x69\xaf\xd1\x2a\
\xdb\x6c\x72\x8a\x78\x81\x17\x60\x8f\xd5\x05\x83\x8b\x90\x4b\x3c\
\xdd\x94\x8a\x26\x72\x18\xda\x44\x51\x2b\x9e\x34\xab\xef\xf1\x7d\
\x3f\x8d\x78\x4f\xa3\x08\x21\xe0\x35\x4b\x7d\xd5\xf4\x01\x60\x00\
\x18\x00\xfe\x04\x40\xd7\x75\xf3\x94\x46\x29\xb5\x5e\x14\x80\x10\
\xb2\xa0\x94\xae\x34\x4d\x9b\x3f\xd4\x34\x4d\x9b\xeb\xba\xbe\x22\
\x84\x2c\x7b\x11\x08\x21\xa4\x0e\xc6\x78\x11\x86\xe1\x7d\x1c\xc7\
\xc2\xf7\xfd\x42\x55\xd5\x79\xa3\xa9\xaa\x3a\xf7\x7d\xbf\x88\xe3\
\x58\x44\x51\x74\xc4\x18\x2f\x65\x7d\x15\x99\x55\x4c\x08\x79\x1f\
\x04\xc1\xc7\xc9\x64\xf2\xeb\xc6\x18\x63\xfb\x34\x4d\x2f\x01\x00\
\x3c\xcf\xfb\x62\xdb\xf6\x59\xa3\x71\xce\x45\x92\x24\x1f\x0e\x87\
\xc3\xa7\x2e\xef\x4e\x00\x4a\xe9\x1b\x4a\xe9\x6a\x36\x9b\x4d\x1e\
\x6a\xdb\xed\x76\xaf\x28\x0a\xb4\xc3\x9b\x5a\xaf\xd7\x45\x51\x14\
\x51\x55\x55\xec\xaf\x00\x00\x00\x10\x42\xe7\xae\xeb\x5e\x3b\x8e\
\x63\x74\x36\x03\xc0\x6e\xb7\xfb\x96\xa6\xe9\xbb\xba\xae\x6f\xba\
\x7a\xa5\x7e\xc2\xba\xae\x6f\x37\x9b\xcd\x05\x63\xac\x78\xce\x70\
\x69\x80\x06\x22\xcb\xb2\x27\x21\xfa\x86\xf7\x02\xf8\x59\x58\x51\
\x94\x93\xdf\x08\x21\x8e\x00\xf0\xbd\x8f\xa1\x34\x00\x42\xe8\xad\
\xe7\x79\x9f\x2d\xcb\xd2\x4f\xf5\xd8\xb6\x6d\x4c\xa7\xd3\x6b\x84\
\xd0\xf9\xb3\x02\xc8\x84\xb7\x21\x5c\xd7\x95\x86\xe8\x9c\x82\xd1\
\x68\x64\x1b\x86\x71\x17\x04\xc1\xa3\x09\x60\x8c\xed\x85\x10\xe0\
\x38\xce\xa3\x31\x4c\x92\x24\x2f\xcb\x32\x2c\xcb\xf2\xeb\x53\xfe\
\x9d\x37\x50\x55\x15\xcb\xf3\xfc\x8a\x73\xfe\x1b\x69\xb3\x88\xb2\
\x2c\xbb\x64\x8c\xed\xdb\x5a\x9e\xe7\x47\xce\xf9\x55\x57\x38\x00\
\xf4\x5a\xc5\xcb\x28\x8a\x8e\x5d\xab\x38\x0c\xc3\x7b\x8c\xf1\x42\
\xd6\x57\x1a\xa0\x81\x30\x4d\x93\xb7\xc3\xdb\x10\xa6\x69\xe6\x7d\
\xc2\x7b\x03\x08\x21\x60\x3c\x1e\x5b\xa7\x34\x4a\xa9\xd9\xd7\x4f\
\x6a\x15\xbf\x64\xfd\x7b\x2f\xa2\x01\x60\x00\xf8\xef\x00\x7e\x00\
\xa3\x07\xf4\x19\xd0\x23\x7c\xc0\x00\x00\x00\x00\x49\x45\x4e\x44\
\xae\x42\x60\x82\
\x00\x00\x07\x67\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x07\x2e\x49\x44\x41\x54\x58\x85\xc5\x97\x7b\x6c\x14\xc7\
\x1d\xc7\xbf\x33\xb3\xeb\x7b\xf8\x7c\xf8\xec\x3b\x53\xc0\x76\x4a\
\x1d\x2b\x08\x84\x1d\x62\xaa\x62\x02\x04\x54\xe8\x35\x18\x51\x81\
\x30\xae\x40\x49\x1f\x94\x48\xa9\xa2\x16\x28\x15\xea\x4b\x15\x25\
\xd0\xf4\xf1\x47\x5b\xa9\x49\x1b\xbb\x50\x48\x0b\x35\x7d\xa8\x45\
\x75\x68\x04\xb4\x35\x8a\x1f\x10\xbb\xe6\x8c\x03\x38\x76\xcf\xe7\
\x03\xbb\x3e\xe7\x7c\xe7\xf3\xf9\xee\xf6\x76\x77\x66\xfa\xc7\xd9\
\x8e\xdd\x94\xca\x8f\xaa\xfd\x49\xbf\x1d\xed\xee\xcc\xfe\x3e\xfa\
\xce\xef\x37\x33\x4b\xa4\x94\xf8\x7f\x9a\x32\xcb\x7e\x0c\x80\x03\
\x40\x16\x80\x38\x00\xed\x7f\x01\x40\x0c\xc3\x58\x15\x0c\x06\x9f\
\xee\xe8\xe8\xf8\x48\x20\x10\xb0\xa5\xd3\x69\xb8\xdd\x6e\xac\x59\
\xb3\xa6\xb7\xb4\xb4\xf4\xcf\x2e\x97\xeb\x1a\x80\xf1\x85\x00\x90\
\x7f\x37\x05\xd1\xe8\xe8\xa2\x9e\x9e\x77\x0e\xd5\xd7\xd7\x57\xfa\
\x7c\xbe\xb6\x3d\x7b\xf6\x28\xeb\xd6\xad\xcb\xb1\xd9\xec\x34\x10\
\xe8\x4b\x9c\x3b\x77\x2e\xae\x69\xda\xf2\x9a\x9a\x1a\xc7\xb6\x6d\
\xdb\xbe\xe7\x72\xb9\x6e\xcc\x9b\x40\x4a\x39\xc3\x7b\x7a\x7a\x5c\
\x67\xcf\x9e\x7d\x6d\xc7\x8e\x1d\x3f\x0c\x06\x83\xb5\xa6\xc9\x13\
\xa6\x69\x4a\xc3\x30\xa5\xae\x1b\x52\xd7\x0d\x99\x4e\xeb\x32\x99\
\x4c\x35\x1e\x3d\xfa\xe5\x2f\x1d\x3f\x7e\xfc\x8d\xbe\xbe\xbe\xaa\
\x7f\xfd\xce\x6c\x7d\x86\x02\xdd\xdd\xdd\xb4\xb1\xf1\xfa\x8f\x1a\
\x1b\xff\x1a\x3f\x73\xe6\xcc\x76\x42\x68\xd9\xf4\xf7\x33\x07\x03\
\x52\xca\xf4\xd5\x6b\x57\xbf\xfe\x76\xd7\x6d\x6f\x75\x75\xf5\xe1\
\xd2\xd2\xd2\xae\xb9\x0a\x40\xa7\xdf\x34\x37\x37\x6f\x69\x6d\x6d\
\xc9\xaf\xad\xad\xdb\x2a\x04\xca\xa4\x94\xa0\x94\x80\x52\x0a\x4a\
\x29\x18\xa3\x50\x14\x06\xc6\x18\x08\x01\x38\x17\x96\xcd\x4f\x6d\
\x79\x51\x51\xd4\xd7\xdf\x7c\xb3\xe9\xd8\xf0\xf0\x30\x7d\x58\xa0\
\x59\x01\xdc\xb9\x73\xf7\x99\x03\x07\x3e\x97\x23\x81\xb5\x12\x12\
\x84\x10\x48\x29\x21\xa4\x80\x94\x02\x42\x48\x08\x91\x51\x84\x31\
\x06\x8b\x45\x05\x08\x2c\x07\x0e\x1c\xdc\xdd\xd9\xd9\x89\x9b\x37\
\xdf\x7a\x6c\xde\x00\x27\x4f\xbe\x64\x8b\xc7\xc7\x4a\x56\xaf\x2e\
\xdb\xc2\x4d\x01\x48\x09\x21\x32\x41\xa5\x90\xe0\x42\x42\x4c\xc0\
\x64\x40\x04\xa4\x04\x2c\x59\x2a\x28\xa3\x1b\x96\x2d\x2b\xf4\xdf\
\xbd\x7b\x77\xd3\xbc\x01\x7a\x7b\xdf\xf9\x40\x61\x61\x51\x91\x10\
\xd2\xce\x39\x87\x61\x98\x10\x42\x80\x73\x01\x2e\x04\x04\x9f\xf4\
\xf7\xc0\x26\xdb\x2c\x55\xc5\x13\x15\x15\x1f\x1c\x1e\x1e\xfe\xd0\
\xbc\x01\x02\x81\x00\xa3\x94\xd9\x4d\xce\xc1\x39\x47\x5a\x37\xa0\
\xeb\x19\x08\x31\x01\x31\xc3\x39\x9f\x80\xe3\x10\x42\x22\x3b\xdb\
\xfe\x58\x24\x12\x99\xed\xc2\x36\x65\x53\x03\x1e\x3c\x18\x18\x0e\
\x85\x42\x50\x15\x05\xba\x61\x80\x9b\x1c\x49\x93\x43\x55\x15\xa8\
\x0a\xc3\x64\x2d\x10\x00\xef\x5b\x39\xa4\x44\x28\x14\x2a\x1e\x1b\
\x8b\x5d\x9c\xb7\x02\x3d\x3d\xdd\x63\x3e\x9f\xcf\x17\x0e\x87\xa7\
\xb2\x9d\x10\x40\xd3\x74\xc4\xe3\x49\x68\xa9\x34\x0c\xc3\x84\x69\
\x72\x70\x73\x42\x01\x93\x23\x9d\xd6\x11\x8b\x8d\xeb\x1d\x1d\x1d\
\xc2\xef\xef\x6b\x9a\x37\x00\x00\x04\x83\xfd\xe7\xfe\xd8\xd0\xc0\
\x4d\x83\x83\x0b\x01\x02\x92\x29\x3b\x85\x81\x0b\x01\x4d\x4b\x23\
\x91\xd4\x90\x4c\xa6\x90\x48\x68\x18\x1f\x4f\x41\xd3\x74\x0c\x85\
\xfe\xf1\xee\x8d\xd6\xd6\x81\x68\x34\xd2\xbe\x20\x80\xf1\xf1\xf8\
\xc5\x5f\x5f\xac\x6f\x69\xff\x5b\x07\x0c\xdd\x84\x6e\x98\xd0\x75\
\x03\x86\x61\x82\x9b\x7c\xa2\x22\xa6\x25\xa5\x14\x48\x24\x12\xd1\
\xda\x57\x7f\x2a\xbb\xba\xba\xbe\xe9\xf7\xfb\xcd\xb9\x02\xbc\x6f\
\x2f\xc8\xc9\x71\x96\x3c\xf9\xe4\x86\x37\x9e\x7b\xee\xf9\x92\xd5\
\x65\xe5\x20\x24\x33\xeb\x52\x02\x90\x80\xcc\x5c\x20\x21\x11\x8d\
\x46\x93\xa7\x4f\xbf\xfa\x6e\x7f\xc0\x6f\x84\xc3\xe1\xb5\xed\xed\
\xed\xb1\x05\x03\x00\x80\xd5\x6a\x7d\xb4\xac\xec\xf1\xd3\x5e\xef\
\xf6\x0d\x95\xeb\xd7\x13\x8f\xbb\x00\x36\x9b\x1d\x20\x80\x61\x18\
\x18\x8d\x46\x71\xef\xde\x9d\x68\x43\xc3\xa5\x94\xaa\xb2\x1c\xc6\
\x58\x8e\xae\xeb\x6d\xfd\xfd\xfd\x5e\x9f\xcf\x17\x59\x30\x00\x00\
\x78\x3c\x1e\x2b\xa5\xec\x93\x25\x25\x8f\x3e\x5f\x54\xfc\x48\xb9\
\xd3\xe9\x64\xaa\x9a\x25\xc6\xc6\x62\x22\x32\x12\x8e\x47\x22\xe1\
\xbf\x68\x9a\x76\x6a\xc9\x92\x25\x1f\x0e\x06\x83\x3f\x5e\xbe\x7c\
\x79\x96\xcd\x66\xeb\xf2\xfb\xfd\xde\xf6\xf6\xf6\xc1\x05\x03\x4c\
\x5a\x5d\x5d\x1d\x39\x71\xe2\x44\xa1\xc7\xe3\x29\x61\x8c\xe5\x38\
\x72\x72\xc2\xaa\xaa\xde\xfe\xd3\xe5\xcb\x53\xe7\x00\x8f\xc7\xb3\
\xcb\xe3\xf1\xd4\xaf\x58\xb1\x42\x75\x38\x1c\x7d\xdd\xdd\xdd\x1f\
\xbb\x71\xe3\x46\xef\x7f\x05\x60\xb6\x96\x9f\x9f\xbf\xa7\xa0\xa0\
\xe0\x42\x45\x45\x85\xe2\x70\x38\x86\x6e\xdd\xba\xf5\xf1\xd6\xd6\
\x56\xdf\x9c\x01\xbe\xd3\x26\x1e\x49\x9b\x62\xb5\x42\x69\xca\xae\
\xd2\x96\x43\x6b\x90\x9c\x2d\x84\xdb\xed\xae\xf6\x78\x3c\xe7\x37\
\x6e\xdc\xa8\x38\x9d\xce\xd1\x96\x96\x96\x4f\x34\x35\x35\x5d\x9f\
\x15\xc0\xc9\x56\xdd\xd5\x3b\x18\xfe\x49\xa1\xcd\xac\x2e\xb0\x83\
\x8c\x69\x02\xa1\x14\x1d\x4d\x52\xdb\x57\xeb\x76\x7a\x5e\x99\x2d\
\x44\x7e\x7e\x7e\xcd\xe2\xc5\x8b\x7f\xe1\xf5\x7a\x95\xbc\xbc\xbc\
\x54\x63\x63\xe3\xbe\xab\x57\xaf\xfe\xfe\x61\xfd\x29\x00\xfc\xe0\
\x66\x82\xdd\x09\x0c\xfd\xe1\xa3\xc5\xca\xde\x6f\x3d\x5d\x4c\x5e\
\xd8\x54\x84\x22\x12\xc6\x9a\x22\x67\xae\x53\x8e\xbf\xfc\xec\xef\
\x42\x07\x67\x0b\x30\x32\x32\x52\x1f\x0a\x85\x9e\xbd\x72\xe5\x0a\
\x4f\xa7\xd3\xb6\x9d\x3b\x77\xfe\xa6\xaa\xaa\xea\xd3\xff\x11\xa0\
\xfd\xfe\xf8\xee\x02\x87\xba\xf1\x81\x74\xe3\xf0\xd9\x16\xc4\x92\
\x3a\x86\x22\x71\xdc\xe7\xb9\xc8\x5b\xbc\x0c\x30\xd3\x2f\x1d\xbe\
\x1c\xb5\xcc\x01\xe2\xc2\xe0\xe0\xe0\xa7\x1a\x1a\x1a\x44\x22\x91\
\x60\x35\x35\x35\x3f\xdb\xbb\x77\xef\xd1\x87\x02\x68\xe9\xf4\x56\
\xcd\x56\x80\xbe\x38\x45\xaa\xa8\x12\x44\xb1\x20\x61\x00\x03\xe3\
\xc0\xfd\x64\x16\x88\x35\x37\x6f\x30\x96\x5c\x3b\x5b\x00\x00\x88\
\x44\x22\xbf\x0c\x04\x02\x9f\xb9\x74\xe9\x92\x88\x46\xa3\x74\xff\
\xfe\xfd\xdf\x3d\x78\xf0\xe0\x8b\x6d\x6d\x6d\x33\xfa\x29\x00\x30\
\xce\x55\x5b\x2a\xc1\xa6\x1e\x0a\x09\xf4\xc7\x81\xdf\x76\x03\x84\
\x00\x8f\xbb\xac\xb0\x27\x87\xad\x73\x01\x00\x80\x68\x34\x7a\x2e\
\x37\x37\x57\x69\x68\x68\xa8\xad\xae\xae\xa6\xfb\xf6\xed\xfb\xda\
\xf9\xf3\xe7\x73\x0c\xc3\xf8\x62\x65\x65\xe5\x7b\x00\xf1\x44\xa2\
\xa3\x57\x13\xcf\x38\x2d\x14\x5b\x8b\x00\x4a\x80\x27\x4a\x0a\xf0\
\x76\x14\xf0\x85\x01\xff\x88\xc6\x4b\xa0\xdf\x9e\x2b\x00\x00\x8c\
\x8e\x8e\x9e\xce\xcd\xcd\x55\x9c\x4e\xe7\x2b\x03\x03\x03\xb4\xb7\
\xb7\xf7\xa9\x23\x47\x8e\xd0\x96\x96\x16\x01\x4c\x54\x41\xc5\xf7\
\x6f\xb9\xc4\xa2\xa5\xf7\xc2\xcc\x5d\xf0\xab\x8d\x23\x40\x3a\x01\
\x00\x38\xdf\xd4\x83\x0b\x7c\x13\x8a\xd3\x7f\xff\xf9\xe0\xb7\x37\
\xbf\x20\x84\x20\x52\xca\x49\x9f\x0a\x42\x08\x01\x21\x44\x12\x42\
\x24\xa5\x54\x32\xc6\xa4\xaa\xaa\xc2\x62\xb1\x08\xbb\xdd\x2e\xf3\
\xf2\xf2\x44\x67\x67\xe7\xe7\xf3\xf3\xf3\xbd\x89\x44\xe2\xb3\x03\
\x03\x03\xc3\x53\x63\xcb\xcb\xcb\xd5\xa1\xa1\x21\xd5\xba\xfd\x1b\
\x9b\x8b\xcb\xd7\x9f\x1d\x4b\xa6\xdd\x91\x68\x0c\x8c\x11\x78\x6c\
\x04\x52\xa2\xb9\xab\xf6\x0b\x87\xcc\x50\x6f\x0a\x99\x5f\x34\x86\
\x4c\xee\x4c\xdf\x49\xc5\x84\x73\x00\x9c\x52\xca\x01\x98\x00\x4c\
\x4a\xa9\x49\x08\x31\x29\xa5\x26\x63\xcc\x54\x55\xd5\xc8\xce\xce\
\x36\x97\x2e\x5d\x6a\xee\xda\xb5\x8b\x33\xab\xd5\xaa\xc6\x62\xb1\
\xec\xd8\xad\xd7\xb5\x07\xed\x57\xae\x41\x88\xb8\x4b\x44\x46\x59\
\xec\xfe\x3d\xff\x5b\xd7\x2e\x04\x2f\x7c\xe5\x35\x99\x88\x8a\x89\
\x40\x59\x93\x2e\xa5\xb4\x48\x29\xb3\x00\xa8\x84\x10\x36\x1d\x4e\
\x4a\x49\x32\xc2\x10\x48\x29\x49\x66\x47\x05\x30\x71\x98\x92\x19\
\xf9\xa4\xdd\x6e\xcf\x48\x59\x55\x55\x45\x9b\x9b\x9b\x1d\x9a\xa6\
\x2d\xe2\x9c\x3b\xa4\x94\x96\x09\x69\x0d\x4a\x69\x92\x31\x96\xa4\
\x94\x72\x21\x04\x9d\x16\x60\x86\x11\x42\x04\x00\x49\x29\x15\x84\
\x10\x41\x29\xe5\x8c\x31\xa1\x28\x0a\xb7\x5a\xad\xc2\x6e\xb7\x0b\
\xb7\xdb\x2d\x56\xae\x5c\x29\x76\xef\xde\x2d\xbd\x5e\xaf\x9c\xca\
\x81\x49\x3b\x76\xec\x18\xb9\x7e\xfd\x3a\x8b\xc5\x62\xd4\x62\xb1\
\xc8\xe2\xe2\x62\x71\xea\xd4\x29\xbe\x6a\xd5\xaa\xb9\x65\xde\x1c\
\xec\x9f\x54\x4d\x2a\x10\x24\xb1\xe0\x2a\x00\x00\x00\x00\x49\x45\
\x4e\x44\xae\x42\x60\x82\
\x00\x00\x05\x42\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x05\x09\x49\x44\x41\x54\x58\x85\xc5\x97\x7f\x68\x55\x65\
\x18\xc7\xbf\xcf\x7b\xde\x7b\xee\xbd\xdb\xd9\xbc\x9b\x32\x17\x6d\
\x4e\xa2\x74\xb1\x12\xc9\x0d\x2a\x15\x59\x69\xd1\x0f\x9d\x52\xa3\
\x1f\x64\xb4\xfe\x30\x0d\xb4\x1f\x94\x09\x51\x48\xd4\x40\x21\x10\
\xa4\xa1\x14\x9a\x0e\x73\x48\x6a\x99\x3f\xa0\x09\xfe\x1c\x53\xa6\
\x92\xca\x92\x92\x35\xcd\xf9\x6b\x6e\xbb\x77\x77\x77\xe7\xdc\x73\
\xcf\x79\xdf\xa7\x3f\xdc\xcc\xd5\xa6\xd3\xda\xf6\xc0\x03\x07\xde\
\xe7\x7d\xbf\x9f\xf3\x3c\xcf\xfb\x9e\xf7\x10\x33\x63\x24\x4d\x8c\
\xa8\xfa\x48\x02\xd4\x5c\x7e\x8f\x00\x80\x46\xa2\x04\x4b\xcf\xdd\
\x1f\x4c\xf8\xd7\x1a\x22\x22\x77\xca\x88\x64\x20\xe1\xb7\x2d\x07\
\x90\x57\x79\xdf\x6f\xde\xb0\x03\xbc\x7d\x36\xf2\x5c\xb2\x43\x7d\
\x04\x46\x1d\x30\xcc\x3d\xb0\xe8\xd7\xac\x09\xc9\x98\xaa\x8e\xb7\
\x68\xd2\x3e\x76\x0c\x2b\xc0\xa2\xd3\x39\xa3\x5d\x5b\xed\xbc\x78\
\x54\x65\xa5\xe7\x8a\x84\xa1\xc2\x35\x00\x20\x87\x43\x7c\xf1\x89\
\x82\x90\xab\x9c\xad\xad\xa7\xd4\x03\xa6\x05\xc8\x20\xad\xfd\xea\
\xa1\x2b\x5d\xc0\x30\x64\xe0\x83\xe3\x45\xb2\xcb\xed\xa8\xe9\x38\
\xab\x66\xb8\x71\x46\x66\xbe\x70\x48\x07\xbe\xec\x1d\x1f\xd2\x0c\
\x7c\x71\xea\x79\xd1\x6a\xff\xb9\x3e\x71\x49\x97\x39\x1d\x80\x95\
\x2b\x60\x98\x62\xc5\x37\x25\xed\x97\x87\x1c\xa0\xf2\xf8\x1c\x71\
\x26\x76\xa0\xca\x89\xea\xd7\x9c\x28\x43\x9a\x40\x78\x0c\x9d\x0b\
\x53\xe6\xca\x9b\xe3\x86\x04\xe0\xdd\xfd\x93\xe4\x55\xf7\x8f\x75\
\xa9\x2e\x9e\xef\x76\x32\x8c\x00\x10\x8a\x10\x88\xc5\xfb\x6b\xa7\
\xb5\x38\x43\x0a\x50\xb1\xa7\x20\x68\xab\xf6\x1a\xcf\xe6\xb9\xbe\
\xc3\x10\x12\x10\x12\x30\x82\xb4\x67\xd3\x53\xf1\xed\xff\x8c\xef\
\xd3\x84\x2f\x6f\x1f\x5d\xf0\x5f\xc4\x5f\xfd\x31\x7b\x4c\x22\xd9\
\xbe\xc7\xb3\x79\xae\xf6\x00\x21\x09\x42\x12\x64\x98\x62\x42\x07\
\x16\xf4\x37\xa7\x0f\x40\xca\x4f\x1e\x79\x61\x8b\xb5\xe4\x6e\xc4\
\x5f\xdc\x92\x59\x92\x4c\xb9\xc7\x94\xc7\xa5\x00\x20\x02\x3d\x2e\
\x01\x56\xb4\x64\xf3\x9c\x68\xcb\x6d\x01\xb4\x87\x2b\x5a\xf3\xaa\
\x79\xdf\x59\x6f\x0d\x56\x78\x5f\xf3\x7a\x9a\xb7\xc9\x5a\xa0\xb4\
\x3a\x24\x04\x0a\x7a\x53\x2e\x24\x40\x04\x68\x85\xed\x5b\xcb\x13\
\xd5\x03\xcd\xef\xd3\x03\xda\xc3\x51\xd3\xa2\xc9\x2a\xc5\x6b\xe6\
\x6c\x48\x2f\x0e\x50\x70\xd9\xd6\xd7\x3b\xda\x07\x9a\x3c\x77\xc3\
\xa8\x89\x1a\xde\x6a\x19\xa2\x59\x32\x44\x00\xfd\x3d\xc6\x1a\x48\
\x75\x73\x8b\x29\xc2\xfd\xa6\xbe\xd7\xfa\x7c\x8e\x67\xaf\xb3\xa6\
\x1a\x41\x3e\x1c\xca\x26\xf8\x0e\xc3\x8d\x73\x54\x7b\xb4\x49\x90\
\xd8\x62\x05\x23\xc7\x37\xcf\xbf\x60\xbf\xb2\x31\x3f\xad\xcb\x8d\
\x4e\x65\xd2\x6f\xc8\x20\x5e\x32\x33\xc9\x30\x4c\xea\xbb\x2a\x03\
\x76\x9b\x56\xec\x8b\x27\x7e\x7a\x33\x71\x70\xd0\x00\x00\xf0\xec\
\x9a\xb4\xdd\x99\xf7\xca\x67\x8c\x10\xc0\xac\xe1\x27\x19\x5e\x37\
\xa0\x5c\x86\x56\xb0\xc9\x40\x58\x06\x89\x4c\x0b\x30\x42\x7d\x85\
\x09\x04\x41\x02\x89\x36\x1f\x6e\x0c\x9f\xec\x5e\xd8\xfd\xf9\xad\
\xc4\x81\xfe\xb6\xa1\x96\x0b\xbb\x2e\xe9\x5f\xf2\x27\xe6\x64\x69\
\xc3\x45\x32\x90\x80\x99\xa1\x7b\x47\xd3\x06\x5a\xc8\x14\x21\x84\
\x8d\x0c\xc4\x3a\x3b\x61\x5f\xf3\xf6\x16\x8d\x2d\xa9\xbc\x9d\x38\
\x30\xc0\x8d\xe8\xe9\x55\xe9\xd3\x2c\xcb\xfa\x79\xe2\xe4\xf1\x61\
\x08\x86\xa3\x12\x70\xfc\x4e\x24\x55\x02\x1a\xba\xef\x02\x20\x44\
\xcc\x5c\x58\x32\x1b\x71\x3b\x86\xdf\x4f\x9c\xbf\x20\x29\x34\x65\
\xd7\xe2\xe8\xb5\xbb\x06\x00\x80\x99\x2b\xd3\xcb\x72\x72\x72\xbf\
\x7f\x78\x4a\xa1\x34\xe4\xf5\x54\x33\x33\x3c\x9d\x82\x62\x0f\x20\
\x06\x41\x40\x92\x09\x29\x02\x70\x3d\x1b\xf5\x87\x1b\x5c\xaf\x5b\
\xcf\xa8\xfd\xb0\xfb\xe8\x60\xc4\x6f\x09\x00\x00\x4f\x56\xa6\xcf\
\x1a\x3b\x3a\x6f\xcb\xa3\x53\x4b\x22\x66\xda\xc0\x87\xa6\x66\x8d\
\x03\x75\x7b\xd1\x7a\x21\xb6\x70\xdf\xc7\xf6\xda\xc1\x8a\xdf\x16\
\x00\x00\x4a\x3f\xb3\x1e\x1c\x65\x65\xff\x30\xfd\xf1\xd2\x09\xf7\
\xe4\xe7\xf4\x1b\xd3\x70\xfa\x10\x8e\x1f\x39\xf9\xf5\xc1\xe5\xce\
\x2d\xb7\x5c\x7f\x76\xdb\xfb\xc0\xbe\x4f\x13\x67\x9c\xee\xe4\x23\
\x3b\x76\x6d\xab\xaa\xdb\x7f\x84\xa5\x17\x46\xa6\x39\xe6\x86\xb7\
\xb5\xb6\xe3\x58\xdd\xc9\x23\x19\xc1\xd1\x8b\xef\x54\x1c\xb8\xc3\
\x6b\xf9\xb4\x65\xe9\xa5\xa6\x69\xae\x7a\xac\x64\xfa\xa4\xe9\x25\
\x33\xe1\xf9\x2e\x56\x7f\xbb\xe2\x4a\xd2\x4e\x15\x1f\xac\x8c\x5f\
\xfc\xdf\x00\xaa\xaa\xaa\xa8\xa1\xa1\x81\x9a\x9a\x9a\x44\x67\x67\
\x27\x39\x8e\x43\xae\xeb\x0a\xcf\xf3\x48\x66\xb9\x81\xb1\xa5\xf6\
\xfc\x8c\xac\xb4\x77\x32\xac\x8c\xf1\x2d\x4d\x57\x67\x9f\xdf\x96\
\x5e\x47\x44\x2c\x84\x80\x61\x18\x5a\x4a\xc9\xa6\x69\x72\x28\x14\
\xd2\x91\x48\x84\xf3\xf2\xf2\x74\x61\x61\x21\x97\x97\x97\x73\x51\
\x51\xd1\xbf\x01\x96\x2e\x5d\x2a\x6a\x6b\x6b\x8d\xd6\xd6\x56\xe9\
\x38\x4e\xc0\xf7\x7d\xa9\x94\x0a\x68\xad\x25\x33\x4b\x66\x96\x00\
\x24\x33\x1b\x3d\xcf\x06\x08\x22\x9c\x8d\x34\xa7\x1d\x49\x00\x0c\
\x40\x03\x50\x00\x94\x10\x42\x01\xf0\x89\xc8\x23\x22\x9f\x88\x7c\
\xc3\x30\x3c\x29\xa5\x1f\x0a\x85\xbc\xac\xac\x2c\xbf\xb8\xb8\xd8\
\xdf\xb8\x71\xa3\x96\x8d\x8d\x8d\x68\x6c\x6c\xa4\x58\x2c\x26\x92\
\xc9\xa4\x54\x4a\x19\x4a\x29\xd9\x23\x1e\xe8\x71\xc9\xcc\x01\x5c\
\x3f\xb8\xae\x3b\xc3\xb0\xdb\xd8\x60\x66\xab\x47\x50\x03\xf0\x01\
\xf8\x5a\x6b\x9f\x88\x44\xcf\x0b\x0a\x21\x04\x29\xa5\x00\x00\xae\
\xeb\x72\x3c\x1e\xd7\xcd\xcd\xcd\xa2\xbe\xbe\x5e\xf7\x29\x41\x75\
\x75\x35\xed\xdc\xb9\x93\x9a\x9b\x9b\x45\x34\x1a\x15\xb6\x6d\x1b\
\xa9\x54\x4a\xf8\xbe\x6f\x68\xad\x85\xd6\xda\x00\x40\x5a\x6b\xc2\
\xf5\x06\xee\x3d\x8b\x19\x00\x13\x11\x13\x91\x16\x42\x68\x21\x84\
\x36\x0c\x43\x99\xa6\xa9\x7a\x4a\xa1\xc7\x8d\x1b\xa7\xcb\xca\xca\
\x74\x45\x45\xc5\x0d\xd1\x11\xf9\x37\xbc\xd9\xfe\x02\x6f\xa0\x6b\
\xc9\xe0\xeb\xbc\xa3\x00\x00\x00\x00\x49\x45\x4e\x44\xae\x42\x60\
\x82\
\x00\x00\x08\xb3\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x08\x7a\x49\x44\x41\x54\x58\x85\xc5\x56\x79\x54\x95\x65\
\x1a\xff\x7d\xeb\xbd\x57\x36\x41\x51\x10\x15\x5c\x10\xe4\x5e\xc0\
\x2b\x88\x80\x4a\x2e\x88\x2c\x8a\x4a\x63\x69\x8e\x7a\x4a\x03\x5c\
\xc6\xad\xcc\x66\xea\x94\xd5\xd4\xd0\x34\x33\x7f\x30\x73\x92\x20\
\xad\xf1\x60\x4e\x58\x69\xee\xc1\x94\x7b\x04\x2e\x2c\x81\x3a\xa2\
\x06\x8a\xc6\x72\x59\xbc\x2b\xdf\xf2\x7e\xef\xfc\x11\x12\xa9\x98\
\xc7\x99\x4e\xcf\x39\xcf\xf9\xce\x79\xdf\x73\x9e\xe7\xf7\xfc\x7e\
\xcf\xf7\x3c\x2f\x43\x29\xc5\xaf\x69\xec\xaf\x9a\xfd\xff\x01\x20\
\x79\xc5\x4b\x7c\xfc\x82\x0d\x86\xbb\xcf\xc3\x92\x33\x33\x4c\xa9\
\xd9\x91\x15\x15\x75\xbf\x0c\x80\x8a\x8a\x3a\x84\x25\x67\x2e\xf4\
\xd0\x1b\x2e\xf8\x0e\xf0\xaa\x1a\x9f\xbe\xca\xa7\x57\xf2\x84\x25\
\xf3\x66\x7c\xba\x6e\xd9\xdc\xca\x2d\x79\x3b\xeb\x22\xe7\xac\xf4\
\xe9\x33\x10\xa5\xf4\x91\x3c\x6c\xd6\xb3\xd1\x6f\xfc\x63\x27\x55\
\x09\xa1\x0d\x37\x9b\x69\xc2\xc2\xe7\x4a\x92\x96\x6e\xe6\x29\xa5\
\x98\xb1\x64\xf3\x09\xa7\x4b\xa2\x94\x52\xfa\x4e\xc1\x6e\x1a\x9e\
\x96\x3d\xa8\xaf\x38\x8f\xcc\x80\xc1\xa0\xbf\x68\x77\x76\xc9\x94\
\x52\x0c\xf5\xf7\xc5\xdb\x2f\x2c\x4f\xb4\xdc\xb6\x1f\x4d\x5d\xfe\
\x72\xc9\xd2\xf9\x89\x53\x0c\x7a\x11\x00\xd0\xd6\x61\x55\xcd\x21\
\xa3\x2c\x7d\xc5\x79\x64\x00\x67\xf7\xe4\x3a\xaa\x2e\x5d\x3b\x23\
\xc9\x2a\x14\x45\xc5\x88\x61\x7e\x78\x65\xcd\xe2\xc9\x73\xa6\xc7\
\x4e\x12\x78\x1e\x97\xae\xde\x00\xa5\x14\x9d\x56\xbb\xa3\xe6\x6a\
\xbd\x3b\x00\xfc\x66\xed\xeb\x9c\x29\x35\x2b\x22\x26\x63\x5d\x4f\
\xcf\xf0\x8f\x0a\x20\x77\xdb\x7e\xa6\xd9\xd2\xe9\x68\xb2\x74\xa2\
\x70\xef\xbf\xe1\xe8\x52\x11\x63\x36\x62\x6c\x58\xa8\xc1\x6a\x73\
\x60\x77\x71\x19\x5a\x5a\x0f\xe0\xf1\xe4\xc9\x5e\x8d\xcd\x6d\xa5\
\x09\x0b\x9f\xbb\x6a\x36\x8e\x9e\x32\x7d\xa2\xb9\x7f\xfe\xbf\x0e\
\xe7\x02\x58\x07\x00\x4c\x5f\x73\x20\x65\xf9\xcb\xbc\xa5\xdd\x6a\
\x38\xb3\x27\xd7\x76\xe7\x6c\xf2\xc2\xe7\x4b\x82\x86\x0e\x36\x5a\
\x6d\x8e\x5b\x6d\x1d\x56\x39\x73\x51\x5a\x5c\x69\xc5\x45\x2c\x9a\
\x9f\x8c\x91\xc3\x03\x40\x34\x0d\x2a\x21\x20\x44\x03\x21\x04\x36\
\x87\x13\x05\x3b\xf7\xe2\x77\x8b\xd3\x10\x1e\x12\x04\x96\xfd\x81\
\xf0\x2d\xb9\x85\xd2\x67\x5f\x9c\x0a\xa9\x3e\x98\xd7\xf0\x13\x09\
\x12\x97\xbe\xb8\x7a\xe3\x5b\xf9\xf5\xa6\x94\xac\x60\x81\xe3\xfe\
\xb6\x2c\x23\xd1\x62\x4c\xce\x9c\x0a\x00\xa6\xd4\xac\xc0\xf8\x28\
\x53\xe2\xa6\xac\xa7\xfc\xff\xb4\x39\x2b\x6a\xfb\x3b\x9b\xe3\xea\
\x1a\x6e\x61\xd1\xbc\x59\x08\x1e\x31\xec\xbe\x45\x18\xf4\x7a\xac\
\x58\x34\x17\x7f\x2f\xdc\x0f\x4d\xfb\xa1\x50\xa7\x4b\x42\x7b\xa7\
\x55\x21\x9a\xe6\x75\x8f\x04\x7a\x9d\x38\x32\x67\xd3\xd3\x81\x84\
\x90\xb2\xc8\xd0\x91\x9e\xe1\x21\x41\xdc\xbb\x3b\x0f\x88\xe6\xf4\
\x55\xbe\x9a\x46\x13\xfb\x7b\x79\x40\x52\x54\x10\xc2\xc2\xae\x28\
\x70\x49\x0a\xc6\x06\x8f\x80\x4a\x48\x9f\x52\xe9\x74\x22\xcc\xe1\
\x61\x28\x39\x7d\x1e\x11\xa1\x23\xb1\xf1\xcd\xf7\xea\x2f\xd4\x35\
\xcc\xad\x3d\x9c\x5f\xdd\x23\x81\x31\x25\x33\x6a\xd0\x00\xef\x05\
\x3c\xc7\x45\xc5\x45\x19\x13\x4d\xa1\xc1\x10\x45\x01\xe7\xab\x2f\
\xc2\x66\xb3\xd9\xe3\xcd\x63\xdd\x3e\xfc\xac\x64\xeb\x77\x8d\xcd\
\x8d\x13\xcd\x61\x2f\x3d\x91\x36\xdd\xad\xf1\xfb\x16\x78\x7a\x7a\
\x60\x5a\x7c\xd4\x8f\xb4\xdf\x25\x81\xda\xfd\x75\x76\x49\x38\x52\
\x72\x0c\x4e\x97\xa4\x7c\x59\x5a\x39\xa4\x72\xff\xbb\x3d\x7f\x05\
\x0f\x00\x03\xbd\xbd\x9e\xc9\x79\x31\x7b\xd5\xf1\xb2\x4a\xcc\x9d\
\x35\x15\x7a\xbd\x08\x49\x56\x30\x2a\x70\x28\xf6\x1c\xfa\xd2\xdd\
\xd3\xc3\x4d\xb5\x3b\x5d\xd5\x17\x8e\xe4\xbf\x17\x99\xb6\x72\xe7\
\xcd\x26\xcb\xd1\x69\x71\xe6\x91\xa3\xfa\xa0\xfe\x1e\x16\x04\x01\
\x76\x97\x04\x37\xbd\x4e\x10\x38\xd6\xd5\xfb\x8e\x05\x80\xdb\x36\
\xc7\xab\x2f\xbd\x53\xb0\x4f\x92\x55\x78\x79\xba\xff\x78\xcb\x00\
\x82\x4e\x87\x8f\xf6\x1d\x3d\x35\xdc\x7f\x70\x01\x00\x54\x1d\xdc\
\x7a\xdd\x66\x77\x5c\x10\x45\x11\x9d\x56\xfb\x43\x01\x20\x84\x80\
\x63\x18\xa4\x4c\x8b\x41\x80\x9f\xef\x59\x63\x4a\xe6\xcc\x9f\x00\
\xd0\x89\x82\x03\x80\xd6\x71\xdb\x06\x72\x97\x9e\x4e\xa7\x0b\xeb\
\x9f\xc9\x98\xca\xb0\x28\x01\x80\xb0\xe4\xcc\x45\xa3\x47\x0c\x9b\
\x31\x74\x88\x1f\xce\x56\x5f\x7a\x28\x00\x17\xea\xbe\x43\xe8\xe8\
\x40\x8c\x09\x1a\x8a\xf7\x73\x36\x86\xae\xfe\x6d\x7a\x71\x44\x5a\
\xf6\x96\x1e\x00\x36\xbb\x73\x49\xda\xcc\xc7\xe6\xf9\x78\x7b\xe3\
\xbd\xc2\xbd\x70\x75\x49\xd0\x34\x8a\x53\xe5\x55\x68\x69\x6d\x53\
\x0f\x7c\x55\x5e\x2f\xc9\x8a\xc5\x94\x92\xf5\xe6\x8c\xc9\x31\x1f\
\xa5\x27\x4d\x35\x08\xa2\x88\xa6\x96\x36\xdc\x6a\x6e\x7d\x60\x72\
\x4d\xa3\x38\x56\x7a\x0e\x93\xa2\x4c\x90\x14\x15\x9a\x46\x91\x10\
\x13\x01\x95\x10\x1e\xe8\x6e\xc2\xe8\x79\x6b\xdc\x64\x85\x6c\x50\
\x09\xb9\x3c\x39\xc6\x5c\xa8\x13\x05\xa1\xa9\xb5\x15\x49\x09\xb1\
\x38\x7d\xa6\xea\xe4\x87\x39\xeb\x13\x66\x2c\xde\x2c\x88\xfd\x0c\
\xad\x4f\xce\x4d\xf6\xe2\x39\x16\x1c\xc7\x42\x96\x64\x94\x9c\x28\
\xc5\x0b\xab\x96\xc0\xdb\xd3\xe3\x9e\x26\x94\x15\x15\xdb\x77\xed\
\x43\x74\x78\x30\xea\x6f\x7c\x0f\x9e\xe7\xe1\xeb\xe3\x85\xab\x0d\
\x8d\xd8\xff\xd5\x37\x13\x6a\x0f\xe7\x9f\xfd\xc9\x20\x8a\x7f\x62\
\x83\x2e\x24\x78\x74\x97\x28\xf0\xb7\x4f\x7e\x73\xee\xd9\x99\x09\
\x13\xdf\x6f\x69\x6b\x77\xed\xfa\xeb\x26\xbf\x6e\xfa\x5f\x7e\x7c\
\x4e\xd2\x1b\x1c\xc3\xe0\x66\x53\x33\xac\x36\x87\x1a\x3f\x21\x92\
\x2f\x3b\x57\x8d\xe8\x88\x10\xc4\x45\x47\x60\x80\xb7\x17\xac\x76\
\x27\x2a\x6b\x2e\xe3\x78\xe9\x39\x4c\x8d\x35\xa3\xe6\xf2\x35\xa5\
\xf8\x78\xf9\x7c\x8e\x63\x2f\x6b\x1a\xf5\xa7\xa0\xfa\xda\xc3\xf9\
\xc5\x3d\x0c\xf4\x36\x63\x72\xe6\x5c\x86\x65\xa5\x9a\x43\x79\x47\
\x4c\xa9\x99\xe3\x38\x8e\x6f\xea\xef\xde\xaf\xf3\xf8\xae\xbf\x74\
\x85\xcf\xce\xf6\x55\x15\xb2\x8d\x61\x98\x7d\x3c\xcf\x1e\x03\x20\
\x7b\xba\xb9\x17\x67\x2d\x5b\x10\xd2\xd4\x62\xc1\xd5\xfa\x1b\xb0\
\x3b\x9c\xd0\x89\x02\x46\x0c\x0f\xc0\xb8\xb0\x60\xf4\x33\x88\xd8\
\xb6\x6b\xbf\xd5\xd2\xd9\x3e\xe0\x8b\x6d\x6f\xa9\x77\x4b\x74\xcf\
\x32\xaa\x3d\x92\xff\x79\xcd\xa1\xbc\x23\x00\x50\x73\x28\xbf\x72\
\x98\xdf\xc0\xf5\x53\x63\x23\x9b\x8d\xa9\x59\xa6\x31\x81\x01\x7f\
\x7e\x65\xcd\x53\x73\x00\x74\x54\x1f\xc8\xbb\x52\x7d\x20\xef\xba\
\xc3\xd5\x55\x23\xab\x1a\x06\xf9\x0e\xc4\x94\xd8\x28\x0c\xf0\xee\
\x0f\x49\x51\xd1\xcf\xa0\x07\xa1\x14\x92\xac\x22\x68\x98\xbf\x67\
\xe3\x2d\x8b\xf9\x7e\x3d\xf2\xb3\xcb\x48\x14\x78\xef\x3f\xac\x5c\
\xe8\xd9\x7e\xdb\x76\xc6\x14\x1c\xa8\x1b\x1d\x14\x00\x00\x6c\xc4\
\xec\xec\x40\x96\xe1\x6c\x1e\x1e\x6e\x7e\x77\xa6\x63\xed\xb5\x7a\
\x94\x9c\x28\x5d\x2b\x0a\xe2\xc7\x57\xae\x35\xec\x5e\xfe\x54\x46\
\x82\xef\x00\x2f\x0c\x1f\xea\x0f\x4a\x69\x02\x80\x33\x77\xc7\xef\
\x73\x19\x99\xd3\x57\x27\xfb\xf9\xfa\xa4\x72\x2c\x33\x71\x7c\x78\
\x68\x4c\xcc\xf8\x70\x88\x02\x8f\xf2\x8a\x5a\xb4\xb5\xb5\xa9\xd3\
\x62\x23\xf9\x4f\x8e\x9c\xfc\xf4\xca\xf5\xa6\xd0\xc4\xc7\x26\x19\
\xfb\x7b\x79\xa2\x68\xef\xc1\xb2\xb0\x91\x23\xe2\xb7\xbf\xbd\x56\
\x1b\x97\xbe\xd2\x77\xa0\xb7\x4f\xf9\xa4\x09\x91\x41\xa7\xcf\x56\
\x5e\xfe\xbe\xd9\xb2\xac\xf6\x70\xfe\x37\x0f\x0d\x60\x4e\xe6\x96\
\xcf\x5f\x5d\xff\x74\x7a\xf1\x89\x72\x2c\xca\x48\x06\xcb\x72\x90\
\x64\x05\x92\xac\xe0\x93\xfd\xc5\x98\x16\x63\xd2\x5e\xcb\x2d\xfc\
\x7d\x5b\xa7\xf5\x13\x55\x25\x4f\xea\xf5\xba\xe9\x92\x2c\x3f\x5f\
\x7b\x38\xbf\xea\x4e\x0c\x53\x6a\xd6\x18\x4d\xa3\x13\x02\xfc\x06\
\x7d\x5c\xf2\xe1\x1f\xef\xd1\xff\x81\x00\x22\x67\xaf\x1c\x32\xcc\
\x7f\x50\xe1\x94\x89\xe3\xa6\x2d\x7b\x62\x76\x4f\x72\x49\x56\xf0\
\xe5\xc9\x72\x9c\xab\xac\xa9\x6a\x6a\x6d\x8f\x2e\xd9\x91\x73\xdf\
\xc0\x0f\x6b\x7d\xbe\x88\x78\x81\xb7\x29\x84\x90\x4e\xab\x03\x77\
\x83\x6c\xef\xb4\x62\xe3\x8a\x05\x91\xc3\x03\x06\x9f\x9f\xbf\xf2\
\x35\xee\x17\x01\xa0\xc8\xea\x73\x19\x69\x33\x13\xdd\xdd\xdd\xf0\
\xcf\xdd\x07\xbb\x47\x34\x45\x45\xcd\x7f\xd0\xd4\xdc\x82\x83\x47\
\xcb\x2c\x92\xa2\xca\x7b\xb6\xbe\xda\xf7\x2e\x7e\x08\xeb\x53\x82\
\x88\xb4\xec\x21\x82\x28\x6e\x70\xb9\xba\xae\x4c\x89\x1d\x9f\x27\
\x0a\x3c\x3a\x6f\xdb\x10\x1d\x11\x8a\xaa\x0b\x75\xdf\x6e\x7f\x6b\
\x6d\xc4\xff\x92\xf8\x8e\xf5\xc9\x40\xf5\xc1\xbc\x5b\xe7\xf6\xe4\
\x6e\x12\x44\x7e\x37\xc7\xf1\xd0\x28\x1c\xe7\xbf\xbd\xb8\xba\xb1\
\xc9\xe2\xa4\x94\x8e\x99\xb1\x74\xf3\x23\xbf\x27\x7b\x1b\x43\x29\
\xc5\x07\x1f\x7c\xc0\x14\x15\x15\xb1\xf5\xf5\xf5\x9c\xcd\x66\xe3\
\x5c\x2e\x17\xaf\xaa\x2a\x4f\x08\xe1\xc1\x09\xba\x21\x91\x49\xd9\
\x0c\xb4\x8e\xfa\xb2\x7d\x9f\xfb\x8e\x9d\x3c\x89\x48\x2e\x87\xe5\
\xca\x99\x8b\xdd\x05\x30\xdd\x0e\x00\x14\x80\xc6\x30\x0c\x61\x18\
\x46\x66\x18\x46\x65\x59\x56\xe1\x38\x4e\x12\x45\x51\xf6\xf1\xf1\
\x91\xe2\xe2\xe2\xd4\x1d\x3b\x76\x68\x3d\x00\xc2\xc3\xc3\x85\xe6\
\xe6\x66\xa1\xab\xab\x4b\x24\x84\x88\x84\x10\x81\x52\x2a\x50\x4a\
\x45\x4a\x29\x4f\x29\xe5\xba\x13\x3d\xc8\x19\x4a\x29\x43\x29\x65\
\x00\x80\x61\x18\xca\x30\x0c\x01\x40\x00\xa8\x00\xd4\x6e\x40\x5d\
\x3c\xcf\xdb\xdd\xdd\xdd\xed\xe9\xe9\xe9\xce\x82\x82\x02\xca\x14\
\x16\x16\x32\x45\x45\x45\x6c\x5d\x5d\x1d\x67\xb5\x5a\x39\x49\x92\
\x38\x59\x96\x79\x42\x08\x4f\x29\xe5\x35\x4d\xe3\x00\x70\x94\x52\
\x96\x52\xda\x3b\x29\x7a\x55\xdf\xbb\x91\xb4\x3b\xce\xb2\x2c\xb9\
\xc3\x02\xcf\xf3\xb2\x5e\xaf\x57\x06\x0f\x1e\xac\xa4\xa4\xa4\xa8\
\x39\x39\x39\xb4\x47\x82\xbe\xac\xb6\xb6\x16\x5f\x7f\xfd\x35\xd3\
\xd0\xd0\x80\x9b\x37\x6f\x32\x92\x24\xa1\xa3\xa3\x83\x21\x84\x40\
\xd3\x7a\x58\x04\xcb\xb2\xe0\x38\x0e\xde\xde\xde\xd4\x60\x30\xc0\
\x64\x32\xd1\xa4\xa4\x24\x6a\x34\x1a\x7f\xb6\x07\xfe\x0b\x07\x39\
\x4c\x46\x6d\x6d\x58\x91\x00\x00\x00\x00\x49\x45\x4e\x44\xae\x42\
\x60\x82\
\x00\x00\x08\xe4\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x08\xab\x49\x44\x41\x54\x58\x85\xcd\x96\x7b\x70\x54\xd5\
\x1d\xc7\xbf\xe7\xec\xbd\xbb\x9b\x6c\x36\xbb\x9b\x18\xb2\x59\x0c\
\x49\x30\x42\x9a\x87\x0f\x30\xad\x60\x70\x6c\x32\x32\xcc\xe0\xd4\
\x94\x06\x47\x79\xd5\x19\xc7\x94\x19\xea\x14\xc6\x69\x18\xda\xa1\
\x0f\x19\x8a\x82\xa9\x68\x6b\x0c\xea\x10\x11\xb5\x35\xa4\x01\xca\
\xab\x0c\x64\x8c\x59\x43\x34\x4c\x6d\xc4\xa4\x31\x89\x42\x22\x59\
\x36\xaf\x0d\x79\xee\xde\xc7\x79\xf4\x8f\x84\x10\x04\x89\xb6\x7f\
\xd8\xdf\x9d\xdf\x9c\x7b\xee\x9c\xf3\xfb\x7e\xee\xef\x77\xee\xb9\
\x87\x48\x29\xf1\x5d\x1a\xfd\x4e\xd5\xff\x1f\x00\x94\x6f\x3a\xf0\
\xdd\x8a\x0a\xe7\xf7\xf7\xee\x5d\x1e\x35\x38\xb8\x54\x71\x38\x32\
\x2d\x2e\x97\x47\xda\x6c\x1c\x81\x40\x9f\xa9\xeb\xff\x32\xad\xd6\
\xe3\xb5\xab\x56\xbd\xb7\xa6\xa4\x84\x7d\x1b\x00\x32\xd3\x1a\xf8\
\xc7\xee\xdd\xb7\xdc\xf1\xda\x6b\xbf\x76\xb8\x5c\x3f\x8b\x5e\xba\
\x34\x8a\x16\x14\x80\xa4\xa5\x5d\x33\x46\x76\x75\x41\xd4\xd4\x20\
\x7c\xea\x54\x8f\x71\xfe\xfc\xce\xce\x07\x1f\x2c\xcb\x7d\xe3\x0d\
\xfd\x7f\x06\xa8\x39\x7a\xf4\x47\x77\x95\x94\xec\x73\x16\x16\xba\
\x2d\x0f\x3d\x04\x58\x2c\x37\x0d\x26\x85\x00\xab\xa9\x41\xe7\xfd\
\xf7\x7f\x1a\xb2\x58\x7e\x72\xdf\x7d\xf7\x75\xfc\x57\x00\x2d\x2d\
\x2d\xe8\xe9\xe9\xf9\x65\x6c\x6c\xec\xb3\xb7\x8f\x8d\xd1\x98\xc1\
\x41\x5c\x33\x8a\x10\x80\x52\x80\xf3\xeb\xe6\xea\x0b\x16\xc0\x74\
\xbb\x31\x30\x30\x30\x38\x3c\x3c\xbc\xe2\x9e\x7b\xee\x79\xff\x66\
\x00\x37\x5c\x84\xc1\x60\x70\x9d\xd3\xe9\xdc\xe9\xf5\x7a\xa9\x3e\
\x6f\x1e\xcc\x91\x11\xc8\xde\x5e\x44\x08\xc1\x3f\x75\x1d\x7f\xeb\
\xef\xc7\x5f\x82\x41\x1c\x0f\x87\xf1\xa5\xdb\x0d\x23\x3a\x1a\x3c\
\x18\x84\x1e\x13\x03\xd3\xed\x06\x00\x78\x3c\x9e\x38\x97\xcb\x75\
\xa8\xb1\xb1\x31\xe3\x5b\x65\xe0\xc4\x89\x13\x77\xb9\xdd\xee\x06\
\x8f\xc7\x63\x77\x38\x1c\x20\x84\xc0\x12\x0c\xa2\xe9\xec\x59\x04\
\x14\x05\x59\x39\x39\x5a\x5a\x5a\xda\x88\xd5\x6a\x0d\x87\x42\xa1\
\x48\x43\x43\x83\x36\x34\x34\x74\xeb\xb2\xbc\xbc\x84\xb8\x59\xb3\
\x20\x6d\x36\x10\x42\x00\x00\xbc\xa3\x03\x62\xe3\xc6\xce\x9e\xe4\
\xe4\x3b\xee\xae\xaa\x1a\x9d\x11\xa0\xa2\xa2\x82\xe4\xbf\xf0\xc2\
\x69\xa3\xb0\x30\x9f\xae\x5b\x07\x42\x29\x38\xe7\xa8\xad\x7d\x1f\
\xd9\xd9\x59\x3c\x33\x33\x33\xa8\x28\xea\x97\x16\x8b\x65\x94\x10\
\x30\x21\x84\x2a\x84\xa4\xba\x61\x5c\x78\x7e\xd7\x4e\xb5\xa8\xa8\
\x68\x9d\xcf\xe7\xa3\x44\xd7\xa1\x1e\x3f\x0e\xe5\xcc\x19\x98\x47\
\x8e\xc0\xe0\xfc\xd9\xa4\xd1\xd1\x2d\x33\x96\x20\xa3\xb5\xf5\x87\
\x31\x6d\x6d\xf9\xb7\xbc\xf9\x26\x9c\x2f\xbe\x08\xd1\xd7\x87\x86\
\x86\x06\x64\x67\x67\xb1\xcc\xcc\xac\xe6\xa8\xa8\xa8\x2f\xac\x56\
\xd5\x4a\x29\xf5\x12\x42\x7d\x84\x52\x37\x21\x94\xaa\x8a\xea\xdd\
\xbc\x79\x0b\x0e\x1c\xa8\x2a\x1f\x1e\x1e\x96\xf4\xf0\x61\x90\xca\
\x4a\xf0\xee\x6e\xc8\xb8\x38\x50\x4d\xdb\xf8\xe9\xc3\x0f\xc7\xcd\
\x08\xe0\x3d\x72\x64\x15\x0c\x03\x92\x31\x28\x27\x4f\x82\x6d\xdd\
\x0a\xc6\xb8\xbc\x7d\xde\xfc\x76\x9b\xcd\x36\x0a\x90\x78\x21\x64\
\x94\x94\x52\x91\x52\xda\x00\xe2\xa4\x94\xf8\x14\x45\x49\x03\x21\
\xc9\x6b\xd7\xfe\xb4\xef\x83\x0f\xea\x3b\xf4\x65\xcb\xc0\x74\x1d\
\x2c\x10\x80\x30\x4d\x48\xc6\xec\xee\xda\xda\x15\x37\x05\xd8\xbe\
\x7d\x3b\xa5\x8c\x2d\x17\x00\x78\x28\x04\x1e\x08\xe0\x53\xb7\x07\
\xf3\xe7\xcf\x1f\x55\x2c\x6a\xa7\x10\x42\xe5\x9c\x73\x21\x04\x13\
\x42\x70\xce\xc5\xc4\x25\x04\x93\x52\x10\x45\xb1\x78\x67\xcd\x9a\
\xf5\xe0\x17\x5f\x7c\x7e\x7a\x94\xf3\x6e\x63\xc9\x12\xb0\x40\x00\
\x7c\x60\x00\x02\x00\x74\xfd\xe1\x9b\x02\xdc\x99\x9e\x3e\xc7\x7a\
\xe9\x92\x57\x00\x10\x9a\x06\x11\x0e\xa3\x3b\x35\x15\x09\x09\xb3\
\x7a\x84\x10\xcc\x30\xd8\x38\xe7\x13\xc2\x57\x9d\x73\xce\x05\x67\
\x13\x7d\xa1\x5a\xd5\x58\xaf\x37\x29\xae\xb7\xb7\xb7\x5e\xe4\xe7\
\x43\x84\xc3\x10\x9a\x06\x09\x40\xaa\xea\x82\x96\x96\x96\xaf\x07\
\x88\x1d\x1e\xbe\x15\xe1\x30\x04\x00\x01\x80\x39\x9d\x30\x13\x12\
\xc0\x39\xe7\x8c\x73\xaa\xeb\xc6\x98\xae\x1b\xc3\x6c\xa2\x7f\x45\
\x9c\x4d\xde\x33\xce\x38\x13\x5c\x70\xc6\x78\xd2\xd0\xd0\x50\x23\
\x49\x4f\x87\xa0\x74\x2a\x1e\x22\x91\xa4\x3f\x6c\xda\x74\xdd\x67\
\x3f\xf5\xc0\x3a\x30\xa0\xca\xc9\xc1\x02\x80\xb0\xdb\x11\x17\xe7\
\x41\x5b\x5b\x9b\x5d\xb1\x58\xe2\x09\x21\x31\xba\x61\x68\xe3\xe3\
\x91\x01\x5d\x33\xc6\x18\x63\x26\x63\x8c\x73\xc6\x39\x9b\x70\x73\
\x74\x6c\xbc\xa7\xa3\xa3\x43\x4d\x4a\x4a\x6a\x23\x76\xfb\xd5\x58\
\x00\x04\x21\xe8\xd6\xaf\xdf\x9d\xa7\x00\xb4\xf8\xf8\x61\x3e\x8d\
\x58\x0e\x0d\x21\x3b\x23\x03\xc7\x8e\x1d\xb3\x46\x22\x1a\x93\x52\
\x2a\x94\x52\x07\x21\x24\xda\x64\x4c\x84\x23\xfa\x58\x38\xa2\x8f\
\x44\x34\x7d\x34\x12\xd6\x46\xc2\xe1\xc8\xe8\x27\x9f\x34\x29\x2d\
\x2d\xcd\x35\x1e\x8f\x27\x11\x23\x23\x90\x42\x5c\x05\x50\x14\x96\
\x97\x97\x77\xdd\xb6\x3b\x05\xd0\xed\x70\x74\x98\x89\x89\xe2\x4a\
\x16\xa4\x69\xc2\xd5\xd3\x03\x8f\xc7\xfd\xf9\xfe\xfd\xfb\x13\x22\
\x11\xcd\x34\x0c\xd3\x30\x4d\x66\x72\xc6\x85\x10\x82\x08\x2e\x08\
\x63\x5c\x72\xce\xd1\xd3\xd3\xa3\x97\xbf\xf2\x72\xf4\x67\x9f\xb5\
\xbe\xa8\xaa\x6a\x81\xfc\xf8\xe3\x89\x38\x93\xf1\xb8\x94\xad\xdb\
\xb7\x6f\xff\x7a\x80\x35\x6b\xd6\x8c\x32\x42\x3e\x9c\x3e\x49\x39\
\x76\x0c\xab\x57\xaf\x72\xed\xdb\x57\x51\x59\x5e\xfe\x4a\x72\xa0\
\xfb\x12\x35\x74\xdd\xd0\x74\xc3\xd4\x34\xc3\xd0\x34\xdd\x18\x1f\
\x0f\x9b\x1f\x7d\xd4\x68\xd9\xb6\xed\x77\x19\x31\x31\x0e\xe7\xfa\
\xf5\xeb\x17\x58\xad\xd6\x1f\xb3\xea\xea\xab\xd9\x9c\x68\xfd\xd7\
\xe5\x1f\x5f\x39\x0f\x68\xb7\xdd\x76\x50\xb9\x74\x69\xf1\x14\xdd\
\xbb\xef\xc2\xbe\x76\xed\x5d\x6f\xbf\xbd\xff\xaf\xcb\x96\x2d\x7b\
\xba\xb1\xf1\xc3\xcd\x59\xd9\x39\x29\xa9\xa9\x73\xc7\x6c\x56\x9b\
\x19\x1a\x0c\xa9\x2d\xcd\xe7\xa2\xcf\x9f\xff\xbc\x3e\x35\x35\xe5\
\x4c\x67\x67\xe7\x16\x4a\xe9\xc9\xc0\xc5\x8b\x24\x61\x78\x78\x62\
\xf1\x4d\x1a\x8b\x8e\x3e\x7c\x23\x00\x48\x29\xa7\xbc\xe6\xb9\xe7\
\x12\xda\xa2\xa3\x47\x3a\x00\xd9\x01\xc8\x76\xb7\x5b\xb6\xed\xdd\
\x2b\xdb\xdb\xdb\x59\x53\x53\xd3\xa6\x9c\x9c\x1c\x25\xd1\xeb\xbd\
\x3b\x3d\x3d\xfd\xf1\xef\x65\x66\x3e\x95\x95\x9d\xfd\x68\x6e\x6e\
\x6e\xb2\x94\x12\x27\x4e\x9c\x70\x6c\xd8\xb0\xa1\x75\xe1\xc2\x85\
\xb2\xa8\xa8\x48\xfa\xfd\x7e\x79\xa9\xba\x5a\xb6\xf9\x7c\xb2\x45\
\x55\x9b\x9f\x5c\xb9\x92\x4e\xd7\xba\xe2\xd7\x64\xe0\x64\x28\x34\
\xf0\x68\x76\xf6\x2e\xc7\xd9\xb3\xcf\xa0\xb0\x10\x64\xf3\x66\x90\
\xb8\x38\x50\x4a\x2d\x4e\xa7\xf3\x8f\x07\x0e\x1c\x28\x0c\x87\xc3\
\xbb\x9a\x9a\x9a\x2a\x43\xa1\x50\xa4\xbf\xbf\x9f\x14\x14\x14\xdc\
\xd2\xd6\xd6\x56\x9c\x95\x95\xb5\xe5\x4e\x9f\x2f\xf5\x4f\xaa\x8a\
\x93\x75\x75\xd8\xbd\x7b\x37\x9e\x78\xe2\x09\xe4\xd6\xd5\xc9\x0b\
\x45\x45\x4f\xdf\x9e\x9b\x7b\xc3\x83\xc7\x35\x3f\xa3\x86\x86\x06\
\xbc\x7f\xfa\x74\xf4\xf2\xd9\xb3\x4f\xdb\xf2\xf2\x16\x11\x42\x40\
\x29\x05\x21\x04\xb2\xa1\x01\xf2\xc2\x05\xa8\x45\x45\x90\x76\xbb\
\x29\xa5\xec\x23\x84\x58\x55\x55\x8d\x27\xa3\xa3\xd4\xa8\xac\x44\
\xb8\xb4\x14\x51\x6b\xd6\xe0\xd5\xd8\x58\x9c\x3a\x75\x0a\xe9\xe9\
\xe9\x58\xb8\x70\xe1\xdf\xeb\xeb\xeb\x57\x1c\x3a\x74\xe8\xfa\xc3\
\xc3\x57\xd7\xc0\xa2\x45\x8b\xe0\xf7\xfb\x23\xed\x2e\x57\x51\x26\
\xe7\xf5\x76\xbb\x3d\x95\x52\x0a\x22\x04\xb4\xd7\x5f\x87\x68\x6e\
\x06\x2b\x2f\x87\x25\x27\x47\x25\x5e\xef\x6c\x10\x02\xa3\xab\x0b\
\xac\xb9\x19\xd0\x34\x48\x29\x11\x7e\xe7\x1d\x6c\x3c\x7a\x14\x42\
\x08\xf8\xfd\x7e\x38\x1c\x8e\x87\x18\x63\x4f\x03\xd8\x39\x63\x06\
\xae\x58\x49\x49\x09\x5d\xbc\x78\xb1\x2f\x23\x23\xa3\xca\xe9\x74\
\xfe\x80\x1f\x3e\x0c\x56\x59\x09\x4c\xd6\x0d\x52\x02\x42\x4c\xf5\
\xa7\x9e\x4d\x3a\x4d\x4e\x86\x5a\x5a\xba\x27\xf7\xde\x7b\x8d\x9c\
\x9c\x9c\xa7\xe6\xcd\x9b\x27\x83\xc1\xe0\xf3\x89\x89\x89\x9b\xcb\
\xcb\xcb\xe5\x8c\x00\x00\xb0\x6d\xdb\x36\xe2\x74\x3a\xa3\x0b\xe6\
\xce\xfd\x6d\xd4\xcb\x2f\x6f\xb2\xa7\xa5\x29\x57\x04\xe4\x57\xda\
\x29\x10\xce\xa1\xb5\xb6\x0e\x8f\x74\x75\xfd\xfe\x55\x9f\xef\xcf\
\xce\xfc\x7c\xfe\xd6\x5b\x6f\xbd\x94\x9b\x9b\xbb\x21\x25\x25\x05\
\xfd\xfd\xfd\xe5\xbd\xbd\xbd\x3f\xf7\xfb\xfd\x53\xe5\xf8\xda\x63\
\xf9\xd6\xad\x5b\xe5\xc1\x83\x07\xc3\xbf\xa9\xa8\xd8\x52\xcc\xd8\
\x9e\x5b\x3b\x3b\x7f\x11\x33\x67\xce\x0a\xbb\xcf\x37\xdb\x12\x1b\
\x7b\x8d\xb8\xd9\xd7\x07\xad\xbb\xbb\x7d\xa4\xa5\xe5\xed\x40\x30\
\xf8\xda\x9b\x0f\x3c\xd0\xfb\xc8\x23\x8f\xc8\x95\x2b\x57\x4a\x5d\
\xd7\x9f\xaa\xaa\xaa\xb2\x48\x29\xd7\x7b\xbd\xde\xf5\x8c\xb1\x28\
\x00\x8f\xcf\x98\x81\xe9\x56\x56\x56\x46\x6a\x6a\x6a\x68\x30\x18\
\x24\xcf\xcc\x9d\x9b\xe1\xee\xef\xcf\x88\xca\xca\x4a\x90\x97\x2f\
\x5b\xc6\x2f\x5e\xec\x0c\x09\xf1\xef\x9d\x42\x74\xb9\x5c\x2e\x64\
\x66\x66\xca\x1d\x3b\x76\x4c\xdf\x02\xf0\xd8\x63\x8f\x91\xda\xda\
\xda\x3d\x29\x29\x29\x4f\x52\x4a\x77\x9c\x39\x73\xe6\x57\xdf\x0a\
\x00\x00\x8a\x8b\x8b\xa9\xdf\xef\x57\x2e\x5f\xbe\xac\x44\x22\x11\
\x2b\x63\xcc\x2a\x84\xb0\x0a\x21\xac\x00\x2c\x00\x2c\x52\x4a\x8b\
\x94\x92\x02\x20\x93\x2e\x09\x21\x82\x10\x62\x00\x60\x94\xd2\x3b\
\x6c\x36\xdb\x7b\x1e\x8f\x47\x5b\xba\x74\xa9\xb9\x67\xcf\x1e\x31\
\x23\x40\x5d\x5d\x1d\x8a\x8b\x8b\xad\x7d\x7d\x7d\x36\xc3\x30\xec\
\x9c\x73\x9b\x10\xc2\x26\xa5\xb4\x4a\x29\x95\x2b\xe2\x93\x4e\xa7\
\xb5\x54\x08\x41\x00\x80\x52\x2a\x00\x30\x00\x06\x00\x83\x52\x1a\
\x51\x55\x75\xc8\xe7\xf3\x0d\x7d\xa3\x0c\xec\xda\xb5\x8b\x54\x57\
\x57\x5b\x02\x81\x80\x32\x3e\x3e\xae\x9a\xa6\xa9\x72\xce\x55\x21\
\x84\x3a\xf9\xd6\x53\xa2\xd3\xdf\x7e\x72\xba\x00\xc0\x09\x21\x8c\
\x52\x6a\x2a\x8a\xa2\x45\x45\x45\x19\x73\xe6\xcc\xd1\x4b\x4b\x4b\
\xd9\x37\x2e\xc1\x8d\xac\xac\xac\x8c\x9c\x3b\x77\x8e\x8c\x8c\x8c\
\x60\x70\x70\x90\x08\x21\x20\xc4\x44\xf9\x15\x45\x81\xd3\xe9\x94\
\xf1\xf1\xf1\x58\xb2\x64\x89\x5c\xbd\x7a\xf5\x0d\x85\xfe\x03\xec\
\x8c\xf5\x9a\x58\x44\xca\x61\x00\x00\x00\x00\x49\x45\x4e\x44\xae\
\x42\x60\x82\
\x00\x00\x0a\xcd\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x40\x00\x00\x00\x40\x08\x06\x00\x00\x00\xaa\x69\x71\xde\
\x00\x00\x00\x04\x73\x42\x49\x54\x08\x08\x08\x08\x7c\x08\x64\x88\
\x00\x00\x00\x19\x74\x45\x58\x74\x53\x6f\x66\x74\x77\x61\x72\x65\
\x00\x77\x77\x77\x2e\x69\x6e\x6b\x73\x63\x61\x70\x65\x2e\x6f\x72\
\x67\x9b\xee\x3c\x1a\x00\x00\x0a\x5f\x49\x44\x41\x54\x78\x9c\xed\
\x5b\x7b\x50\x5c\xd5\x1d\xfe\xce\xbd\x77\x5f\xec\x83\xe5\xb1\x64\
\x21\x26\xe1\x11\x1e\x21\x64\x42\x4c\x44\x76\x79\x54\x6d\x3a\x36\
\xb6\xb5\x66\x8c\x64\x6c\x6b\x46\x6d\xa7\xb5\xb5\x71\x9c\xd1\x9a\
\x54\x5b\xfb\x47\xad\xd6\xc9\xd4\x4e\x8d\x8e\x8f\x3a\xb6\xb5\x99\
\x34\xb6\x9d\x5a\xab\x83\x43\x63\x84\x00\xbb\x4b\xe2\xb6\x24\x25\
\x81\x80\x09\x42\x12\x08\x6f\xd8\x5d\xf6\x71\xf7\xde\xbb\xa7\x7f\
\xc0\x12\x60\x17\x58\x16\xf0\x3a\x23\xdf\xcc\x9d\xe1\x9e\xc7\xef\
\xf7\x9d\x8f\xf3\xfc\x9d\xbb\x84\x52\x0a\x39\x40\x08\x39\x38\xfd\
\x9d\x52\xfa\x6b\x59\x78\xc8\x28\xc0\x0c\xc7\x94\x52\x22\x07\x0f\
\x46\x0e\xa7\xd1\x40\x08\xd9\x29\x87\xdf\xcf\x8d\x00\x00\xbe\xf0\
\x02\xc8\x82\x55\x01\xe4\x26\x20\x37\x56\x05\x90\x9b\x80\xdc\x58\
\x15\x40\x6e\x02\x72\x63\x55\x00\xb9\x09\xc8\x8d\x55\x01\xe4\x26\
\x20\x37\x56\x05\x90\x9b\x80\xdc\xe0\x16\x53\xb8\xb4\xb4\x34\x97\
\x51\x32\xdf\x91\x94\x52\x09\xe7\xe7\x0e\xd9\x6c\xb6\x8f\x56\x8a\
\xd8\x67\x85\x98\x04\x28\x2f\x2f\xdf\x2c\xa9\x85\x3f\x53\x3d\xf2\
\x07\x4a\xfa\x18\xbf\xd9\xaf\x4e\x3f\x9e\x51\x5e\xfa\xe5\xd2\xab\
\x9a\x90\xc6\x52\x5b\x5b\x3b\xb6\xd2\x44\x57\x0a\xf3\x0a\x50\x55\
\x55\xc5\x5e\xe9\xbb\x72\x00\x4a\xfa\x54\xf7\xee\x6e\xf5\xa0\xb5\
\x9f\xc1\x64\xdc\x66\xa0\xbc\x4f\x97\x79\x2c\x27\x2b\xd9\x99\xf2\
\x16\x80\x3b\x3f\x03\xae\x53\x28\x2f\x2f\xd7\xb3\x2c\x9b\x2a\x08\
\x42\xd0\x6e\xb7\xf7\x2c\xc5\xd6\x9c\x21\xb1\x8a\x8a\x8a\xf5\xa2\
\x52\x78\xdf\x67\xf6\x65\x5d\x7a\xb0\x43\xc7\xa7\xf0\x11\x65\x18\
\x81\x41\xd1\x33\xc5\x5e\xf5\xa8\xfa\x21\x5b\xbd\xed\xc8\xa2\x1c\
\xcf\x0a\x89\x01\x78\x9e\x52\x7a\x30\x5a\xd9\xca\xca\xca\x74\x91\
\x15\x0f\x50\x42\xef\x23\x02\x31\x50\x8e\x8a\x62\x82\xc8\xb3\x01\
\x56\x41\x09\x3d\xc3\xf9\xb9\x17\x78\x9e\xff\x97\xd3\xe9\x14\x16\
\xc3\x01\x98\x43\x80\x8a\x8a\x8a\xad\x22\x2b\x7e\xd4\x73\xc7\x15\
\xc3\xb5\xaf\xf4\x70\x20\x00\xf5\x00\xf4\x43\x40\xdd\xa6\x41\x60\
\xa3\x1f\x64\x27\x40\x92\x01\x43\x7b\x22\x36\xbe\x91\xff\xe9\xa9\
\xe3\xa7\xb2\x17\xe5\x38\x06\x01\x2a\x2a\x2a\xb2\x45\x85\xf8\x34\
\x40\xef\x19\x28\xeb\x67\x06\x2a\xfb\xd4\x7c\x12\x0f\xca\x4d\x54\
\x25\x22\x41\xf2\xd9\x14\x98\x4f\xac\x75\x6b\x7a\x35\x80\x88\x07\
\x1c\x8d\x8e\x7f\x2c\x49\x00\x8b\xc5\xb2\x13\x6a\xbc\xd3\xb9\xef\
\x13\xed\xc8\xb6\xe1\xa9\x40\xa5\xea\xa9\x04\x6c\x17\x77\x60\xf7\
\xae\xdd\xa8\xa9\xab\x41\x63\x5f\x23\x02\xaf\x8e\x83\x84\x08\x6e\
\xfc\x49\x49\x40\x11\x54\x6c\x6e\x68\x68\xe8\x5c\x2e\x01\x2c\x95\
\x96\xfd\x54\x41\x9f\xeb\xdb\xd9\xab\xec\xbb\xe5\x9a\x42\xd2\x88\
\xf3\xda\xd3\x76\xe9\x50\xf0\x52\xa1\x9f\xe1\xd9\x1f\xdb\x1b\xec\
\x6f\xc6\xca\x63\xc6\x32\x58\x56\x56\x56\x49\x13\xe8\xbb\xed\x3f\
\x6a\xd5\x4d\x6f\x3c\x95\x00\x5c\x20\xd8\xf3\xf5\x3d\xd8\xb7\x6f\
\x1f\xf6\xed\xdd\x07\xa9\x53\x02\xf5\x01\x94\xa1\x18\xd9\x36\x0c\
\x31\x24\xee\x89\xd5\xe9\x7c\xb8\xfd\xf6\xdb\xb5\xa5\x3b\x4b\xff\
\xc9\xa7\x05\x9e\x6b\x79\xea\x8c\xb6\x67\xd7\x95\x05\x1b\x0f\x00\
\xde\xcc\x71\x9c\x7f\xa2\x45\x23\x69\xc5\xc3\xd6\x2f\x59\x1f\x8f\
\xd5\xdf\x54\x0f\xd8\xb1\x63\x47\x02\x67\xe0\x2e\x5e\xfc\x5e\x7b\
\xba\xab\x30\x72\x52\xa7\xe7\x00\xdd\x2b\x89\x08\xb4\x04\xa0\x29\
\xd0\xc0\xf3\xa0\x0b\xa4\x74\xa2\x6e\x62\x9b\x11\x39\x6f\xe6\xb5\
\x9e\x3a\x7e\x6a\x73\xcc\x8e\xa3\xf4\x00\x8b\xc5\xf2\x47\xaa\x91\
\x6a\x86\x6f\x1c\x49\xeb\xda\xdb\xa9\xa6\x5c\x28\x56\x73\x53\x50\
\x8e\xa8\x50\xf8\x42\x91\x8f\x73\x2b\x0e\x3a\xea\x1d\x87\x17\x2a\
\x3f\xd5\x03\x54\x2a\xd5\xfe\x60\x2a\x6f\x8c\xd6\x78\x00\x20\x45\
\x80\xf7\x65\x17\xa4\x7a\x1e\xe3\xaf\x8f\x4d\x35\x1e\x00\xdc\xf9\
\x2e\x30\x12\x93\x5d\x59\x59\x99\xbe\x68\xc6\x93\x48\x49\x49\x31\
\x53\x15\x3d\xdd\x55\xd5\xb5\xee\xd3\x6f\x5f\x8c\xab\xf1\x00\x10\
\x4c\xe6\x71\xe1\x91\xd6\x04\x30\x78\xde\x6a\xb5\x6e\x5c\xa8\xfc\
\x94\x00\x3c\xcf\xbf\xaa\x1a\x52\x75\xe7\xbf\x5a\xe0\x37\x74\x24\
\x02\x8b\xb8\x2f\xa1\x0c\xc5\xd8\x96\x11\x49\x92\xa4\xdd\x71\xb1\
\x06\x90\xb3\x39\xe7\xde\x8b\xdf\x6d\xd7\x0f\xdd\x3c\xb0\xe4\x0b\
\x92\x40\x9a\x1f\x57\xbf\x71\x59\x19\xd2\x84\xde\x26\x84\xcc\x6b\
\x6f\x4a\x00\xa7\xd3\xe9\xa2\x3e\xdc\x6c\x38\x9f\xf4\xd3\xdc\xd7\
\xf2\xbb\x76\x1c\xdc\x1e\xc8\xfc\xfb\x86\x90\xb6\x4b\x17\x93\xd3\
\xa1\x9b\x06\xb5\xa2\x56\xbc\x3f\x5e\xd2\x97\x1e\xea\x50\x8e\x15\
\x8d\xc6\x5b\x3d\x02\x7d\xb7\xf5\xb2\x01\x53\x20\xcf\x52\x6e\x79\
\x64\xbe\x72\xd7\x57\x01\x42\xf2\x01\x94\x03\x28\x01\x70\x77\x9b\
\x4e\xa7\xfd\xcb\x0d\xe6\xae\xba\x35\x46\x13\xaf\x20\x9a\xe1\x9b\
\x86\xd9\x91\xed\x43\x2a\x4f\xb6\x1b\x88\xa2\x29\x23\x30\xd8\xfe\
\x58\x49\x10\x22\x31\xda\xed\x76\xff\x42\x04\x67\xcf\x01\x5c\xf3\
\xc2\x8d\x62\x04\x06\xc6\x96\x64\xa4\x9e\x32\x79\xc7\x8a\x46\x35\
\x83\x65\xfd\x0c\x65\xe6\xee\xaa\xea\x01\x0d\x8a\x9e\xdd\xea\x53\
\x13\xf5\x86\xda\xda\xda\xa1\xa8\x3c\x28\xa5\xd1\x26\x24\x3a\xed\
\x81\x46\xa3\x41\x8a\x29\x85\x98\xd6\x99\xc8\xd0\xd7\x06\xc8\xb5\
\x5d\xd1\x37\x5f\x85\xbf\xd8\x42\xbb\xea\xba\x79\xb7\xdb\x1d\xc2\
\xf5\x41\xa4\x00\xa0\x5c\xb8\x75\xb8\x2e\xec\x1c\x9d\x96\x48\x13\
\x19\x46\xa3\xd1\x67\xde\x68\xe6\xd4\xa9\x6a\xc5\xa5\x87\xdb\x89\
\x6f\xad\x6f\x4e\xb3\x99\x6f\x65\x53\xe1\x5d\xc9\xd3\xdd\xd9\xed\
\x9a\xc6\x89\x02\x00\xa5\x34\x73\xae\xd3\x20\x99\xa4\xc4\x02\x60\
\xfd\x7e\x3f\x7b\xf5\xf2\x55\xe6\xfc\xc7\xe7\x89\xf9\x83\x0c\x28\
\x47\x54\x51\x2b\x8d\x6f\xf6\x10\x43\xa2\x41\x0d\x20\x01\x80\x76\
\xf2\x59\xb8\xf1\x00\x10\x02\x20\x4d\x3e\x62\xf4\x87\x52\x0a\x4a\
\x29\x46\x47\x47\x13\xda\x3e\x6e\x53\xf6\xfc\xb7\x87\xe4\xbd\x50\
\x08\xd5\x1c\x7c\x00\xe0\xda\x1d\xbd\x24\x2d\x23\xcd\xc0\xb2\xec\
\x3a\x00\xeb\x27\x9f\x0d\x93\xcf\xe2\x8e\xc3\xc1\x60\x10\x43\x03\
\x43\x48\x6d\x32\x45\xcd\xf7\xe4\xbb\x61\x30\x1b\x16\x63\x72\x49\
\x18\xe8\x1f\x40\x6f\x47\x2f\xf2\x0e\x15\x82\xf3\x46\x3f\xd6\xf0\
\xa9\x01\xb8\xf3\x5d\x48\x33\xa7\x45\xcd\x9f\xaa\xa5\xd7\xeb\xa1\
\xd7\xeb\x17\x74\x4a\x79\x0a\x53\x43\x1a\x7a\xef\xb8\x1a\x91\xe7\
\xc9\xf1\x20\x5b\x95\x0b\xb3\xd9\x8c\xf0\xe4\xeb\xf5\x7a\xe1\xf1\
\x78\xa6\xca\x10\x86\x80\xb2\x14\x34\x38\x6b\xd4\x69\x10\xb9\xf2\
\x84\x26\xb6\xbb\xa0\x13\xff\xfd\x30\x92\x93\x93\xa1\x54\x4e\x74\
\xac\x90\x18\x02\xdf\xcb\x23\xfb\xf7\xb9\xe8\x78\xb4\x2d\x2a\xe7\
\xbe\x6f\xf6\x20\xb7\x7d\x13\xa8\x34\x61\x23\x6c\x6b\xdb\xb6\x6d\
\x4f\x72\x93\x09\xa4\xb8\xb8\xf8\x16\x00\xb7\x84\x79\x4e\xab\x4f\
\xa6\xdf\xdd\x33\x0c\x43\x58\x9e\x7b\x34\xa1\x37\x41\xeb\xcb\x98\
\x39\xf6\x44\x9d\x00\x49\x27\x06\xd3\xd3\xd3\x8f\x4a\x92\x34\x08\
\x00\xa9\xa9\xa9\x4c\xd8\xa9\xda\xa8\xbe\xd3\x97\xe7\xcd\xfa\xe4\
\xe1\x0b\x9c\xb0\x7d\x26\x49\xce\x3e\xf3\x5d\xe1\x56\xa0\xe8\x97\
\xc5\x62\x68\x34\x74\x9c\xe7\xf9\xce\xd9\x9c\xc2\xbc\x01\x10\x48\
\x60\x12\xba\xb5\x55\x29\xa7\x4d\x89\xc3\x25\x83\x11\x02\x78\xd7\
\x79\x21\x9a\x84\xa0\x79\xdc\x7c\x5c\x10\x84\xfe\x69\xf5\xd7\x4c\
\xf5\x80\x33\x67\xce\xd4\x01\xa8\x8b\xa6\xe0\x6c\x58\x6e\xb3\xe8\
\x8d\x67\x93\x1f\xf6\x65\xf8\xd8\xd9\x79\xee\x02\xb7\x94\xea\x36\
\x35\xdb\x6c\xb6\x17\x67\xd4\xb9\xd5\x72\x38\xb0\xc6\x9f\x71\xf1\
\xa1\x76\x6e\xbe\x99\x1b\x00\x88\xc8\x20\xff\xa5\x42\x1f\x2b\xb2\
\xbf\x3d\xed\x3c\xfd\xb3\x58\x38\x59\xad\xd6\x57\x32\x8f\x65\xdb\
\xdc\x05\x63\x1a\xc1\x10\x79\x28\x1c\xd9\x31\xcc\xa4\x8f\xac\x6d\
\x6d\xae\x6b\x7e\x62\x7a\x7a\x7c\x21\x31\x3f\xfe\x9a\x7a\xda\xe4\
\x8d\x96\x35\xb2\x7d\x48\x23\x6a\xc4\x1f\x4c\x4f\x2b\xbd\xb5\xf4\
\xb9\x60\x62\xf0\x81\x0b\xfb\xcf\x6b\x43\x8a\x85\x77\x78\x59\x47\
\x72\x02\xca\x21\x55\x7d\x53\x5d\xd3\xcf\x63\xa5\x64\xb7\xdb\x9b\
\x09\x25\x2f\x66\xbf\x95\x1b\x75\x49\x70\x15\x8d\x71\x94\xa3\x11\
\xe7\x95\xb8\x04\x68\x6a\x6a\x72\x28\x47\x55\x50\x8e\x45\x4e\xf0\
\xae\xc2\x31\x80\x20\xd3\x6a\xb5\x6e\xa9\xaa\xaa\x62\x4b\x6f\x2b\
\x7d\x56\x48\x14\xf6\xb7\x3e\xde\xa2\x15\xb5\x0b\x1f\x6a\xcc\xb5\
\xe9\x52\x72\x4b\x4a\x8f\x51\x61\xdc\x43\x17\xf9\xfd\x8e\x67\xd0\
\xf3\xb4\xae\x53\xdf\x9f\xf2\xb1\x29\xa2\x9e\x77\xdd\x38\x48\x88\
\x98\xad\x56\xeb\xda\xe9\xe9\x71\x09\x40\x29\x0d\x85\xb8\xd0\x7b\
\x49\xff\x4b\x8e\x70\x44\x19\x8a\x2b\x77\x75\x2b\xa9\x92\x36\x5e\
\x19\xbe\x7c\xd9\xbf\xd6\xf7\x48\xeb\xe3\x2d\x5a\x51\xb7\x70\xac\
\xc2\xd0\x9e\x88\x1b\xde\xdb\xe0\x85\x0f\x3b\x6b\x6a\x6a\xa2\xf6\
\xb0\xf9\x70\xee\xdc\xb9\x20\x17\xe4\xaa\x32\x8f\x65\x07\x14\x6e\
\xc5\xcc\x4c\x02\xb8\x36\x8d\x49\x0c\xc3\xec\x9a\x9e\x1c\x77\x54\
\x58\x11\x50\x1c\x4d\x39\x6d\x72\x47\xcb\x1b\x2c\xeb\xe7\x3a\x7e\
\xd8\x66\xe8\xf8\xfe\x85\x8c\xd6\xc7\x62\x6b\xbc\x6a\x48\x8d\xdc\
\xd7\x0b\xfc\x24\x40\xee\x74\x38\x1c\x5d\xf1\xf2\x6a\x6c\x6c\x74\
\x12\x4a\x5e\xc9\x3e\xb2\x31\x62\x37\x3a\x5a\x3c\xa2\x13\x35\xe2\
\xde\xe9\x69\x71\x0b\x60\x30\x18\x4e\x24\x5c\xd5\x2a\x58\x7f\xc4\
\x3c\x08\x60\xe2\x84\xe8\xce\x77\xc5\x6c\x2f\xef\x77\x05\x02\x11\
\xc8\x01\xbb\xdd\x7e\x32\x5e\x4e\x61\x24\xa9\x93\x9e\xd4\x5f\x4a\
\x1c\x4c\xf9\x4f\xea\x8c\x74\xd7\xa6\x31\x30\x41\xa6\xbc\xa8\xa8\
\x68\x6a\xec\xc6\x2d\x40\x75\x75\x35\x4f\x95\xf4\xe4\x6c\x27\xf1\
\x82\xef\x0d\x76\xc4\x72\x7e\x8f\x05\xd5\xd5\xd5\x3c\x17\xe4\xaa\
\x32\x8f\xe6\xf8\xb9\xf1\xeb\x43\x41\xd4\x09\xe0\x53\x03\xbc\xc1\
\x60\xa8\x08\xa7\x2d\xe9\x62\x84\x19\x67\x5e\x34\x7f\xb8\x76\x7c\
\x29\x36\xc2\x68\x6b\x6e\xab\x5e\x0e\x3b\x61\x34\x34\x34\x9c\x22\
\x94\xbc\x91\x75\x34\x7b\xc6\x50\x18\x2d\x1e\xd6\x42\x81\xaf\x86\
\xdf\x97\x24\x80\xc3\xe1\xa8\x51\x7a\x94\x6e\xfd\xc5\xa5\x6f\x7f\
\x29\xa5\xf1\x45\x40\xe6\x01\x09\x90\x03\x89\x17\x8c\x23\x49\x67\
\x93\xa7\xd2\xbc\x19\x3e\x4e\x54\x89\x5b\xc3\xef\x4b\x12\x80\x52\
\x4a\x09\x4f\x0e\xa5\x9f\xc8\x98\xfb\x38\x26\x23\xec\x76\xbb\x1f\
\x7e\xb2\x37\xeb\xc8\x46\x3f\xe7\x9b\xd8\xf3\xf1\xa9\x01\x30\x94\
\x64\x85\xcb\x2c\xf9\x6e\x30\x18\x08\xfe\xc1\xd0\x6a\x24\x4a\x57\
\x6c\x87\x3e\x00\x73\x9e\x26\x57\x02\x0e\x87\xc3\xc6\x84\x98\x3f\
\x6d\x78\x7b\x62\x28\xf0\x29\x3c\x88\xc0\x4c\x85\xee\x96\x2c\x80\
\xd3\xe9\x74\x51\x96\x1e\x33\x35\xac\x59\x78\x97\x03\x80\x09\xb2\
\x28\x38\x5c\xb8\xe8\x35\x7e\x29\x10\x5c\xc2\x63\x49\x2d\x49\x03\
\x6b\xab\xd7\x45\xac\xc7\xcb\x72\x3b\xcc\x05\xb9\xdf\x98\x4f\xa6\
\x07\xc3\x01\x8b\x39\x41\x81\xdc\x37\xf2\x7d\x0a\xb7\xf2\x9d\xe5\
\xf0\x1b\x2b\x9c\x4e\xa7\x8f\xfa\x70\xb3\xf9\xc3\x8c\x9e\xfc\x97\
\x37\xf9\x28\x43\x47\xc2\x79\xcb\x22\x40\x63\x63\xe3\x79\x84\x48\
\xcb\x9a\xba\xf4\x39\x27\x32\x46\x60\x90\xfb\x5a\x81\x5f\x77\x49\
\xdf\x2c\xba\xc4\x07\x97\xc3\xef\x62\xd0\xd4\xd4\xd4\xaf\x14\x95\
\x25\x09\xdd\xda\xfb\x99\x00\x73\x5f\x38\x7d\x51\xd7\xe3\xf3\x41\
\x25\xaa\xee\xbd\xe1\xfd\xf5\x67\xbd\xeb\xc7\xf5\x9e\xdc\x99\x1b\
\x44\x6e\x5c\x81\x82\xc3\x85\x5e\xd5\x90\xfa\x03\xd1\x25\x7e\xcb\
\xe9\x74\x0a\x0b\x04\x6b\x57\x04\xf5\xf5\xf5\x83\x00\xfe\x36\x3d\
\x6d\xd9\x3e\x90\x38\x79\xf2\xe4\xa7\xac\xc0\xde\x9d\xf7\x7a\x81\
\x2f\xe9\x6c\x32\x48\x88\x80\xe5\x59\x98\x1c\x69\xd8\xf2\xab\xad\
\x5e\xd5\x80\xfa\x70\xd3\x89\xa6\x7b\xe2\xb9\xc0\x5c\x49\x2c\xfb\
\x0f\x26\xac\x56\xeb\x5d\x52\x82\xf4\x0c\x18\x64\x31\x41\x86\xa1\
\x0a\xda\xc8\xfa\xd8\x43\x36\x9b\xed\xdf\x33\x1c\x2f\xe2\x76\x78\
\x25\xb1\x62\xbf\x18\x29\x2b\x2b\x2b\x62\x59\xb6\x7f\xb2\xdb\x45\
\x3a\xfe\x9c\x08\xb0\x6c\x73\xc0\x6c\xd8\x6c\xb6\x73\x2b\x65\x7b\
\x39\xf1\x85\xff\x48\x6a\x55\x00\xb9\x09\xc8\x8d\x55\x01\xe4\x26\
\x20\x37\x56\x05\x90\x9b\x80\xdc\x58\x15\x40\x6e\x02\x72\x43\x4e\
\x01\x9e\x9f\xf6\x77\xbd\x5c\x24\x64\x13\x80\x52\x7a\x90\x52\x1a\
\xbe\x79\xae\x91\x8b\xc7\xff\x01\xc8\xcf\x47\xc6\xbf\xc5\xf4\x6d\
\x00\x00\x00\x00\x49\x45\x4e\x44\xae\x42\x60\x82\
\x00\x00\x07\x55\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x07\x1c\x49\x44\x41\x54\x58\x85\xc5\x97\x6d\x6c\x14\xc7\
\x19\xc7\xff\x33\xb3\xeb\xf3\x1d\xe7\xc3\x67\xdf\x19\xcc\x8b\x5d\
\xea\xb8\x41\x20\xec\x52\x07\xd5\xc1\xd0\x80\x54\x7a\x2d\xa0\x06\
\x2c\xc0\x29\x52\x5a\xa9\x10\xa4\x54\xf9\x40\x28\x15\x52\x5f\x54\
\x21\x02\x4d\x95\x7e\x20\x91\xaa\x94\x40\x21\x38\x2d\xc8\xf4\x0b\
\xaa\x7a\xa5\xae\x9c\xa6\x46\xb1\xcf\xbc\xb8\xc6\xc6\xc1\x38\x76\
\xce\xf6\xc5\xe7\xe2\xe3\xec\xf3\xdb\x9d\xef\x76\x77\xe6\xe9\x07\
\x9f\x1d\x08\xb4\xb2\x4d\xd5\x3e\xd2\x7f\x57\xab\x7d\x66\x9e\xdf\
\x3e\xf3\xcc\xec\x0c\x23\x22\xfc\x3f\x4d\x9b\xa5\x9f\x00\xe0\x04\
\x90\x01\x60\x1c\x40\xf2\x7f\x01\xc0\x4c\xd3\x5c\x1d\x0a\x85\xbe\
\xd5\xd2\xd2\xf2\xd5\xde\xde\x5e\x7b\x2a\x95\x82\xc7\xe3\xc1\xda\
\xb5\x6b\xbb\x8b\x8b\x8b\xff\xe6\x76\xbb\xdf\x07\x30\xf1\x24\x00\
\xec\x71\x43\x10\x8b\x8d\x2c\xec\xea\xfa\xf8\x60\x4d\x4d\xcd\xb3\
\xad\xad\xad\x37\x77\xed\xda\xa5\x95\x97\x97\x67\xd9\xed\x0e\xde\
\xdb\xdb\x13\xaf\xae\xae\x1e\x4f\x26\x93\x2b\xaa\xaa\xaa\x9c\x5b\
\xb6\x6c\x79\xc3\xed\x76\x5f\x9b\x37\x01\x11\x3d\xa4\xae\xae\x2e\
\xf7\xf9\xf3\xe7\xdf\xdb\xbe\x7d\xfb\x9b\xa1\x50\xe8\xb4\x65\xc9\
\xb8\x65\x59\x64\x9a\x16\x19\x86\x49\x86\x61\x52\x2a\x65\x50\x22\
\x31\x59\x7f\xf8\xf0\x8f\x7e\x78\xf4\xe8\xd1\xda\x9e\x9e\x9e\x6d\
\x9f\xef\x67\xb6\x7a\x28\x03\x9d\x9d\x9d\xbc\xbe\xfe\xea\x5b\xf5\
\xf5\x7f\x1f\x3f\x77\xee\xdc\x56\xc6\x78\xc9\x83\xef\x1f\x6e\x0c\
\x10\x51\xaa\xee\xfd\xba\x9f\x7e\xd4\x7e\xdb\xb7\x7b\xf7\xee\x57\
\x8b\x8b\x8b\xdb\xe7\x9a\x00\xfe\xe0\x43\x63\x63\xe3\xe6\xa6\xa6\
\x40\xee\xe9\xd3\x67\xbe\xae\x14\x4a\x88\x08\x9c\x33\x70\xce\xc1\
\x39\x87\x10\x1c\x9a\x26\x20\x84\x00\x63\x80\x94\xca\xb6\xe9\xb9\
\xcd\xaf\x69\x9a\xfe\xe7\x0f\x3f\x6c\x38\x12\x89\x44\xf8\xbf\x0b\
\x34\x2b\x80\x3b\x77\x3a\x5e\xdc\xb7\x6f\x7f\x16\x01\xcf\x10\x08\
\x8c\x31\x10\x11\x14\x29\x10\x29\x28\x45\x50\x6a\x2a\x23\x42\x08\
\xd8\x6c\x3a\xc0\x60\xdb\xb7\xef\xa5\xca\xb6\xb6\x36\x5c\xbf\x7e\
\xe3\xe9\x79\x03\x1c\x3f\xfe\xba\x7d\x7c\x7c\xac\x68\xcd\x9a\x92\
\xcd\xd2\x52\x00\x11\x94\x9a\x0a\x4a\x8a\x20\x15\x41\xa5\x61\xa6\
\x40\x14\x88\x00\x5b\x86\x0e\x2e\xf8\x86\xa5\x4b\x97\x05\x3b\x3a\
\x3a\xbe\x36\x6f\x80\xee\xee\x8f\x17\x2f\x5b\xb6\x7c\xb9\x52\xe4\
\x90\x52\xc2\x34\x2d\x28\xa5\x20\xa5\x82\x54\x0a\x4a\x4e\xeb\x33\
\xb0\xe9\x7b\x86\xae\xe3\x2b\x65\x65\x5f\x88\x44\x22\x5f\x9c\x37\
\x40\x6f\x6f\xaf\xe0\x5c\x38\x2c\x29\x21\xa5\x44\xca\x30\x61\x18\
\x53\x10\x2a\x0d\xf1\x90\xa4\x4c\xc3\x49\x28\x45\x58\xb0\xc0\xf1\
\xf4\xf0\xf0\xf0\x6c\x17\xb6\x19\x9b\x69\xd0\xdf\x1f\x8e\x0c\x0e\
\x0e\x42\xd7\x34\x18\xa6\x09\x69\x49\x24\x2c\x09\x5d\xd7\xa0\x6b\
\x02\xd3\x73\x81\x01\x78\x64\xe5\x20\xc2\xe0\xe0\x60\xc1\xd8\xd8\
\xe8\xa5\x79\x67\xa0\xab\xab\x73\xac\xb5\xb5\xb5\x35\x1a\x8d\xce\
\x54\x3b\x63\x40\x32\x69\x60\x7c\x3c\x81\xe4\x64\x0a\xa6\x69\xc1\
\xb2\x24\xa4\x95\xce\x80\x25\x91\x4a\x19\x18\x1d\x9d\x30\x5a\x5a\
\x5a\x54\x30\xd8\xd3\x30\x6f\x00\x00\x08\x85\xfa\xaa\xff\xe4\xf7\
\x4b\xcb\x94\x90\x4a\x81\x81\x4d\x4d\x3b\x4d\x40\x2a\x85\x64\x32\
\x85\x78\x22\x89\x44\x62\x12\xf1\x78\x12\x13\x13\x93\x48\x26\x0d\
\xdc\x1b\xfc\xe7\xfd\x6b\x4d\x4d\xe1\x58\x6c\xb8\xf9\x89\x00\x26\
\x26\xc6\x2f\xfd\xe1\x52\x4d\xa0\xf9\x1f\x2d\x30\x0d\x0b\x86\x69\
\xc1\x30\x4c\x98\xa6\x05\x69\xc9\xf4\x8c\x78\xa0\x28\x49\x21\x1e\
\x8f\xc7\x4e\xbf\x73\x8a\xda\xdb\xdb\x7f\x1e\x0c\x06\xad\xb9\x02\
\x3c\xf2\x2f\xc8\xca\x72\x15\x55\x54\x6c\xa8\x3d\x70\xe0\xe5\xa2\
\x35\x25\xa5\x60\x6c\x6a\xd4\x89\x00\x10\x40\x53\x17\x10\x08\xb1\
\x58\x2c\x71\xf6\xec\x3b\xf7\xfb\x7a\x83\x66\x34\x1a\x7d\xa6\xb9\
\xb9\x79\xf4\x89\x01\x00\x20\x33\x33\xf3\xa9\x92\x92\x2f\x9f\xf5\
\xf9\xb6\x6e\x78\x76\xfd\x7a\xe6\xf5\xe4\xc1\x6e\x77\x00\x0c\x30\
\x4d\x13\x23\xb1\x18\xee\xde\xbd\x13\xf3\xfb\xff\x38\xa9\xeb\x22\
\x4b\x08\x91\x65\x18\xc6\xcd\xbe\xbe\x3e\x5f\x6b\x6b\xeb\xf0\x13\
\x03\x00\x80\xd7\xeb\xcd\xe4\x5c\xbc\x50\x54\xf4\xd4\xcb\xcb\x0b\
\x0a\x4b\x5d\x2e\x97\xd0\xf5\x0c\x35\x36\x36\xaa\x86\x87\xa2\xe3\
\xc3\xc3\xd1\x0f\x92\xc9\xe4\x89\xfc\xfc\xfc\x75\xa1\x50\xe8\xd7\
\x2b\x56\xac\xc8\xb0\xdb\xed\xed\xc1\x60\xd0\xd7\xdc\xdc\x3c\xf0\
\xc4\x00\xd3\x76\xe6\xcc\x19\x76\xec\xd8\xb1\x65\x5e\xaf\xb7\x48\
\x08\x91\xe5\xcc\xca\x8a\xea\xba\x7e\xfb\x2f\x57\xae\xcc\xec\x03\
\xbc\x5e\xef\x4e\xaf\xd7\x5b\xb3\x72\xe5\x4a\xdd\xe9\x74\xf6\x74\
\x76\x76\x7e\xe3\xda\xb5\x6b\xdd\xff\x15\x80\xd9\x5a\x6e\x6e\xee\
\xae\xbc\xbc\xbc\x8b\x65\x65\x65\x9a\xd3\xe9\xbc\x77\xeb\xd6\xad\
\x6f\x36\x35\x35\xb5\xce\x19\xe0\x97\x37\x55\x61\xca\x52\x6b\x34\
\xce\x27\x1d\x3a\x0f\x1c\x5c\x8b\xc4\x6c\x21\x3c\x1e\xcf\x6e\xaf\
\xd7\x7b\x61\xe3\xc6\x8d\x9a\xcb\xe5\x1a\x09\x04\x02\xcf\x37\x34\
\x34\x5c\x9d\x15\xc0\xf1\x26\xc3\xdd\x3d\x10\xfd\x8d\x57\x37\x76\
\x3b\x35\xc5\x24\x01\x31\x43\x1b\x49\x70\xfb\x8f\xcf\x7c\xdb\xfb\
\xf6\x6c\x21\x72\x73\x73\xab\x16\x2d\x5a\xf4\x3b\x9f\xcf\xa7\xe5\
\xe4\xe4\x4c\xd6\xd7\xd7\xef\xad\xab\xab\xbb\xfc\x1f\x01\x4e\x5e\
\x8f\x8b\x1b\xc1\xa1\x0f\xd6\xe5\x6b\x1b\x5d\x2e\x27\x88\x6b\x00\
\x01\x5c\x1a\xb8\xdd\x1f\x43\x54\x3a\x0e\x54\x57\x2e\x3a\x3d\x07\
\x88\xef\xe4\xe7\xe7\xbf\xb7\x63\xc7\x0e\x91\x97\x97\x27\x6b\x6b\
\x6b\xf7\xfb\xfd\xfe\x77\x1f\xe7\xcb\x01\xa0\xf9\xd3\x89\xca\x3c\
\xa7\xbe\x71\x2c\x73\x31\xfa\x4d\x17\xc2\x29\x07\xc2\x86\x03\x9f\
\xca\x6c\xe4\x2c\x5a\x0a\x58\xa9\xd7\x5f\xbd\x12\xb3\xcd\x16\x60\
\x68\x68\xe8\xe2\xc0\xc0\xc0\xf7\xfc\x7e\xbf\x8a\xc7\xe3\xa2\xaa\
\xaa\xea\xb7\x7b\xf6\xec\x39\xfc\x58\x67\x22\xc2\xae\xdf\xf7\x9d\
\x3a\xd7\x6e\xd1\xfd\x04\x3d\xa2\xda\xbb\x23\xf4\xc2\xc5\x10\xed\
\xb9\xd8\x5f\x31\xd7\xfd\x5e\x76\x76\xf6\x77\xcb\xcb\xcb\xe5\xc9\
\x93\x27\xe9\xf2\xe5\xcb\x6a\xff\xfe\xfd\xaf\xdd\xb8\x71\xe3\x21\
\x1f\x0e\x00\x13\x52\xb7\x27\xa4\x80\xc7\x8e\x47\x14\xbf\xdf\x8f\
\x4f\xc2\x51\x8c\x26\x26\x33\x67\x9b\x81\x69\x8b\xc5\x62\xd5\x1d\
\x1d\x1d\x2f\xf9\xfd\x7e\x15\x89\x44\xd8\xde\xbd\x7b\x7f\x72\xe1\
\xc2\x85\x37\x03\x81\xc0\x8c\x8f\x06\x00\xe3\xf1\x78\xcb\x1b\xcd\
\xea\xc5\xb7\x6a\xfe\x0a\xd3\x34\x21\xd5\x67\x9d\x24\x52\x26\xf4\
\xc5\x5f\x92\xf6\xa4\x71\x7b\xae\x00\x00\x30\x32\x32\x72\x36\x3b\
\x3b\x5b\x73\xb9\x5c\x6f\x87\xc3\x61\xde\xdd\xdd\xfd\xdc\xa1\x43\
\x87\x78\x20\x10\x50\x40\xba\x08\xcb\x7e\x75\xcb\xad\x16\x2e\xb9\
\xfb\x51\xca\x93\x27\x65\x7a\xd1\x07\x00\xc6\xb1\x30\x83\x50\x30\
\xd9\xf9\xee\xc0\x2f\x36\xbd\xa2\x94\x62\x44\x34\xad\x99\x20\x8c\
\x31\x30\xc6\x88\x31\x46\x9c\x73\x12\x42\x90\xae\xeb\xca\x66\xb3\
\x29\x87\xc3\x41\x39\x39\x39\xaa\xad\xad\xed\x07\xb9\xb9\xb9\xbe\
\x78\x3c\xfe\xfd\x70\x38\x1c\x99\x69\x5b\x5a\x5a\xaa\xdf\xbb\x77\
\x4f\xcf\xdc\xfa\xb3\x4d\x05\xa5\xeb\xcf\x8f\x71\x97\x67\x38\x61\
\x41\x70\x86\x3c\x87\x80\x1a\xee\x6d\x6c\x3f\xf5\xca\x41\x6b\xb0\
\x7b\x12\x53\x47\x34\x81\xa9\xe2\x7d\xf0\x4f\xaa\xd2\x92\x00\x24\
\xe7\x5c\x02\xb0\x00\x58\x9c\x73\x8b\x31\x66\x71\xce\x2d\x21\x84\
\xa5\xeb\xba\xb9\x60\xc1\x02\x6b\xc9\x92\x25\xd6\xce\x9d\x3b\x25\
\x2b\x2c\x2c\xb4\x0d\x0d\x0d\x39\x0d\xc3\xc8\xa5\x9c\xc2\x42\xef\
\xba\xe7\x77\xe4\xe4\x64\x17\x4b\x29\x8d\xfe\x4f\x3a\x6f\xc6\xaf\
\xd7\x34\x70\xc6\x12\x98\x3a\x0f\x6a\xd3\x10\x44\x24\x88\x88\xa5\
\xbf\x5c\x4e\x07\x9f\x0e\xcc\x18\xb3\x18\x63\x66\x1a\xc2\x4c\x43\
\x98\x9a\xa6\x59\x19\x19\x19\xa6\xdb\xed\x36\x2b\x2a\x2a\x2c\x46\
\x44\xd8\xb6\x6d\x1b\x6f\x6c\x6c\x74\x26\x93\xc9\x85\x52\x4a\x27\
\x11\xd9\xd2\xa9\x35\x39\xe7\x09\x21\x44\x82\x73\x2e\x95\x52\x1c\
\x00\x27\x22\xf6\xf9\xb1\x66\x8c\x29\x00\xc4\x39\x57\x8c\x31\xc5\
\x39\x97\x42\x08\xa5\x69\x9a\xcc\xcc\xcc\x54\x0e\x87\x43\x79\x3c\
\x1e\xb5\x6a\xd5\x2a\x55\x59\x59\x49\x3e\x9f\x8f\x66\x6a\x60\xda\
\x8e\x1c\x39\xc2\xae\x5e\xbd\x2a\x46\x47\x47\xb9\xcd\x66\xa3\x82\
\x82\x02\x75\xe2\xc4\x09\xb9\x7a\xf5\xea\xf9\xd4\xdf\xac\xec\x5f\
\xb0\x2a\x64\x19\x40\x14\xb1\xc3\x00\x00\x00\x00\x49\x45\x4e\x44\
\xae\x42\x60\x82\
\x00\x00\x04\xaa\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x04\x71\x49\x44\x41\x54\x58\x85\x95\x57\x4f\x68\x5c\x45\
\x18\xff\x7d\xf3\xfe\xec\xdb\xb7\xd9\xa4\x1b\x21\x66\x9b\x34\x6d\
\xd2\x34\x2a\xa6\x4d\x69\xa2\x42\x29\xe8\xa9\x81\xd2\x80\x17\xe9\
\x4d\xd0\x43\xe9\x4d\xf0\x20\x92\x83\x28\x9e\xf4\xe0\xad\x17\xa9\
\x04\x21\x1e\xfc\x07\x6b\x28\x1e\x3c\x54\x30\x15\x3d\xd8\xec\xc1\
\x40\x48\x25\x1b\x34\xd5\x34\x6b\x92\x26\x4d\x77\xd3\xd7\xb7\x6f\
\xc6\x43\x76\x36\xf3\xc6\x79\xaf\xe9\x07\x8f\xf7\xcd\xf7\x66\xe6\
\xf7\xfb\xfe\xcc\x9b\x19\x12\x42\x40\xca\xd4\xd4\x14\xcd\xce\xce\
\x52\x18\x86\x88\xa2\x08\xb5\x5a\x8d\x82\x20\x00\xe7\x1c\x51\x14\
\xa1\xd1\x68\x20\x08\x02\x12\x42\x40\x08\x21\xed\x14\x45\x11\x74\
\x1b\x00\xd5\xd6\x1a\x33\x32\x32\x12\x96\x4a\x25\x2e\x31\x6d\x28\
\x52\x2e\x97\xbb\x26\x27\x27\x6f\xb8\xae\xeb\x36\x4d\x64\x78\x93\
\x24\x4d\x44\x50\xf4\x96\x27\x42\x08\xa1\xb6\x01\x08\x22\x42\x10\
\x04\xd1\xe8\xe8\xe8\x19\x00\x75\x23\x81\x62\xb1\x98\x29\x14\x0a\
\xcf\x39\x8e\xd3\x02\xe1\x9c\xcb\x49\xa1\x46\x4b\xd5\x89\xa8\xa5\
\x33\xc6\x5a\x36\x69\x97\x7a\x10\x04\xe8\xec\xec\x64\x2a\x66\x8c\
\x80\xe3\x38\x96\xe2\x6d\x0c\x44\xb7\xe9\xdf\x24\x98\x10\xa2\x15\
\x19\x95\x98\x6c\x0f\x0d\x0d\xc5\x30\x63\x6c\x2c\xcb\x62\xa6\xc9\
\xd3\xbc\x37\x7d\x4f\x73\xa0\x58\x2c\xc6\x08\xe8\x11\x60\x48\x11\
\x09\x94\xe4\xbd\xec\xa3\xb6\xf5\x7e\x85\x42\x21\x3d\x02\xfa\x64\
\x8f\x23\x63\x22\x66\x8a\x90\x24\x90\xcf\xe7\x53\x09\x58\x89\x88\
\x29\x44\x9e\x44\x72\xb9\x5c\x32\x01\xc6\x18\x4b\x0a\x5f\x92\x3c\
\x69\x7f\xdf\xf7\x93\x6b\x80\x88\x98\xd6\x36\x7a\xa8\x83\xca\x65\
\xa6\x2e\x3b\x53\x7f\x22\x42\x36\x9b\x75\x12\x09\xd8\xb6\x9d\x9a\
\x02\x7d\xa9\xa5\xf5\x49\x12\xc7\x71\x52\x57\x81\x25\x3d\x31\xfd\
\x68\xd4\x62\x3a\x08\xb0\x6a\x93\xba\xeb\xba\xe9\x45\x68\x1a\xa4\
\xea\x6a\xa8\xf5\x76\x5a\x1a\xa4\x4d\x2f\xf4\xff\x15\xe1\x41\xbc\
\x78\x1c\x11\x9d\xb0\xe6\x64\x72\x0a\xd4\x08\xe8\xbf\x53\xa9\x1f\
\xa4\x30\xd3\x8a\x54\x8f\x40\x3c\x1f\xcd\x08\xa8\xe0\x3a\x09\x13\
\x80\x09\x2c\xa9\x9f\x6d\xdb\xc9\x9b\x11\x9a\x1b\x91\x0e\xa8\x4f\
\x96\x16\x01\xd3\x5b\xd3\x93\x23\x00\xad\x26\xe4\x20\xdd\xf3\xa4\
\x55\x62\x22\x63\xa8\x81\xd4\x14\x90\xee\x81\x29\x05\x26\x50\x13\
\xb8\xc9\x0e\x65\xbb\x07\x34\x8f\x39\xe7\xb1\x8f\xa6\x3f\x5c\xd2\
\xf2\x32\xe6\x9e\x80\x86\x78\xa4\x1f\x58\x92\x6b\x60\xef\x24\x45\
\xc6\xb0\x37\xbf\x1b\xbd\xd3\xbd\xfc\x3b\x9c\xc7\xf5\xc5\xab\x58\
\xdf\x5e\x85\x4d\x1e\x6c\xc7\xc2\x91\xae\x41\x5c\x38\xf2\x16\x84\
\x10\xb1\xc1\xfa\x5e\x10\x4b\x81\x1e\xf2\xa4\x25\xa8\xf6\xf9\xed\
\xde\x77\x98\xb9\xf5\x29\xce\x16\x5f\xc3\x40\x37\x87\x00\x87\x23\
\x7c\x6c\x6c\xac\xe1\xe3\x95\x4b\x38\xec\x3d\x9f\x1d\xc7\xf8\x7e\
\x44\xb4\x08\xb4\x96\xa1\x0a\xaa\x83\x24\x3d\xd5\xa0\x82\xaf\x6f\
\x7d\x82\x73\x83\xaf\x22\xdf\x27\xd0\x7f\xbc\x1f\xe3\xc3\xaf\xc3\
\xee\xde\x41\x71\xe0\x10\x46\x0a\xe3\x98\xdb\x9a\xf9\xe0\xbd\xb9\
\x17\x2d\x23\x01\xc6\x98\x6d\x2a\xa2\xa4\xdc\xeb\xf2\xc3\xf2\x35\
\x9c\x3e\x7a\x16\xb9\x9e\x3a\x90\xdd\x81\xcb\x3c\xb4\xd9\x9d\x10\
\x14\x82\xf2\xf7\xf1\xf4\x09\x07\x27\xda\x5f\x38\x55\xd9\x5c\xb8\
\x90\x44\xc0\x22\x53\x31\x1d\xc0\x7b\x22\x42\xb5\x5e\xc1\xe1\xa3\
\x39\x74\x58\xfd\x18\xca\x9c\x47\x5f\xe6\x34\x00\xe0\xa4\x7f\x11\
\xfd\xd1\x79\xd4\xb7\x1a\x18\x7e\x76\x10\x0f\xb7\xc5\x39\x39\xa7\
\x5e\x03\xb6\x0a\xa6\xca\x41\x4e\x3e\x19\x8f\x81\xd9\xbb\xe8\x76\
\x9f\xc1\x31\x6f\xb4\x65\xef\x6b\x1f\x06\x00\xcc\x3f\xf8\x06\x99\
\xb6\x3a\x5c\x96\xc9\x1a\x09\x00\x88\xed\x86\x69\xfb\xbe\x49\x3c\
\xb4\x23\xa4\x55\xfc\xb2\xfb\x21\x7e\xae\x45\x38\x69\x5f\xc6\xa9\
\x8e\x09\x4c\x57\xde\x40\xdd\x5a\x86\xe5\x70\x38\x0d\x86\xfa\xce\
\xc3\x45\x39\x46\xdf\x8e\xf5\x88\xc4\xf4\xb4\x5a\x20\x22\x0c\xe7\
\x27\xb0\xb2\xfe\x27\x3a\xbc\x08\x05\x1f\x00\xf6\x2e\x35\x79\xdf\
\x41\x6f\x57\x1e\xc5\x42\x07\xfe\x58\xba\x5b\xe3\x75\xe7\x5b\x23\
\x01\xbd\x06\x92\x80\x4d\xf9\x17\x02\x78\xa9\x6f\x02\xdb\x4b\x03\
\xf8\x6b\xf3\x5f\xe4\xec\x2c\xfe\x89\xae\xe3\xc6\xfa\x3b\xf0\xbc\
\x1d\x64\x98\x8b\x72\xa5\x82\x5f\x6f\xae\xbc\x3b\x73\x65\xeb\xae\
\x9c\x4b\xf7\xd8\x52\x41\xd2\xce\xff\xaa\x70\xde\x3c\x9a\x73\xe0\
\xf2\x99\xab\x98\x9e\x7b\x1f\xd3\x8b\x5f\xe2\xd8\xf1\x43\xf0\x1c\
\x17\xf7\xef\xed\xe2\xce\xed\x00\xcb\x3f\x36\x3e\xfa\xfd\x73\xff\
\x33\xbc\xbd\x3f\x56\xff\x13\x5a\xf2\x36\xdc\x6c\xb7\xde\xa6\xdb\
\xd1\xde\x03\x08\x01\x44\x3c\x42\x10\x3c\xc2\x6e\xbd\x86\x97\x9f\
\x7a\x13\x23\xb5\x8b\x28\xcf\xfe\x84\xea\xce\x9d\xa0\xb6\x1e\xde\
\x66\xab\x6e\xe9\xe6\xb5\x2f\xbe\xf2\x7d\xdf\xed\xed\xed\x15\x63\
\x63\x63\x61\xa9\x54\x8a\x62\x04\xe6\xe7\xe7\xbf\x5f\x5a\x5a\x7a\
\xa5\x79\xbb\xe5\x9c\x73\x4e\x7b\xb7\x5c\x2e\x84\xe0\xcd\xa4\x72\
\x21\x04\x17\x42\x44\x7c\x5f\xa2\x30\x0c\xf9\xe6\xe6\x26\xaa\xd5\
\x2a\x6d\x6c\x6c\xb0\xb5\xb5\x35\x5a\x58\x58\x68\x84\x61\x48\x44\
\xd4\xb0\x2c\x6b\xd7\xf3\xbc\xd0\xb6\xed\xa8\xad\xad\x8d\xf7\xf4\
\xf4\x08\x00\xf8\x0f\x5b\xcb\x31\x74\xa8\xa6\xc4\xdc\x00\x00\x00\
\x00\x49\x45\x4e\x44\xae\x42\x60\x82\
\x00\x00\x02\xd6\
\x89\
\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d\x49\x48\x44\x52\x00\
\x00\x00\x20\x00\x00\x00\x20\x08\x06\x00\x00\x00\x73\x7a\x7a\xf4\
\x00\x00\x02\x9d\x49\x44\x41\x54\x58\x85\xed\x97\x31\x8f\xd3\x40\
\x10\x85\xdf\xcc\xc4\x71\x22\xe5\x38\x81\xd0\x61\x44\x43\x41\x89\
\x28\xa0\xbc\x82\xf6\x84\x68\x29\x10\x0d\xc5\xfd\x12\x04\xa2\x85\
\x3f\x81\xe8\x69\x80\x06\x89\x82\x82\x86\x82\x02\x1a\x7a\x04\x12\
\x67\x25\x52\xec\x40\xe2\xa1\xb0\xbd\xbb\xde\x5d\x5f\x52\x04\x85\
\xe2\x5c\x44\xb1\x67\x77\xe6\xdb\xb7\x6f\xc7\x09\xa9\x2a\x76\x79\
\xf1\x4e\xab\x9f\x01\xfc\x0f\x00\x03\xf7\xe6\xd1\x93\x67\x57\x55\
\x35\xfb\xb7\x25\xb5\xb8\x92\x5d\xfe\x7c\x7c\x7c\xbf\xea\x00\x3c\
\x7e\xfa\xfc\xde\xad\x9b\xd7\x5f\x5e\xbc\x70\x9e\x2b\xad\x20\x2c\
\x50\x28\xf6\x26\x7b\x5b\x2b\x5d\x2e\x16\xc8\xf3\x1c\x5f\xbe\x7e\
\x7b\x01\xe0\x41\x07\x20\xbb\x74\x70\xb4\xbf\x7f\x8e\xe7\x65\x01\
\xad\x14\xaa\x0a\x16\x46\x32\x4c\xb7\x06\x30\x2f\x0a\x10\x13\xb2\
\xec\xe0\x4e\xfb\xcc\x00\xac\xaa\x4a\xf2\xe9\x02\x65\xb9\x00\xc1\
\xf6\x86\x5f\x27\xc5\xd6\x00\x54\x15\xc9\x30\x45\x51\xfe\x36\xde\
\x73\x00\x18\x45\xb1\xc0\xdd\xa3\x43\xa4\x69\xbb\x6a\x02\xd1\xe9\
\x49\x69\xdd\x00\x0f\xe0\xdd\xfb\x8f\xc8\x73\xeb\x7d\x03\x30\x4c\
\x86\x10\x11\x0c\x92\x04\x55\xd5\x2a\xa0\x2d\x07\xba\x65\xa8\x29\
\x5e\x27\xf5\x90\x4e\x85\x1e\x8f\x46\x10\x49\x42\x00\x16\x01\x33\
\xd7\xfb\x0f\x2f\x29\x91\x07\x60\xc1\x1c\x1c\x1b\x0b\x80\xed\x2a\
\x88\x08\x2c\x36\x6a\x00\x44\x18\xcc\xec\xac\xde\x99\xa7\x2e\x12\
\xf9\xb5\x9a\x98\x33\x2f\x00\x76\xa1\x09\xc2\x12\x03\x18\x40\x44\
\x50\x79\x92\x92\xa2\xab\x87\xad\xe8\x7e\xb5\x0f\xfc\x39\x1e\x49\
\xad\x40\x04\x80\x59\x40\x64\x15\xa0\xe6\x23\xd8\x61\x55\xb4\x2a\
\x84\xb1\x76\xad\x76\x8c\xb7\x5b\x00\x7a\x15\x60\xb0\xb0\x31\x95\
\xc2\x9d\x6c\x65\x37\x51\x3f\xb1\x01\xb2\xfb\x43\x0e\xa1\xb6\xea\
\x10\x41\xa2\x0a\x88\x80\x59\xa2\xae\xd6\xa6\x60\xcc\xdd\xea\x7c\
\xa9\x8f\xa4\x86\x31\x58\x75\x88\x38\xbe\x05\xd2\x9c\x82\x2a\x90\
\xcc\x3e\xa8\xd9\xc8\x8b\xbb\x30\x1a\x89\x51\x07\xa6\x57\x01\x91\
\x41\xdd\xff\x3b\xce\x0e\x2f\xea\x78\x7e\x93\x26\xe4\x99\x9a\x38\
\xee\x01\x16\x06\x31\xd7\xdb\x1b\xc9\x6b\x0d\xe6\x25\x77\x4c\x10\
\x1f\x63\x39\x6b\x63\xf7\x9c\x02\xe1\x46\x81\x48\x06\x42\x1c\xca\
\xad\x66\x54\x8b\xef\x8d\xb9\xb8\x7f\x0b\x18\xc4\xf1\x2a\x3e\x54\
\xdc\x8c\xcd\xd1\x73\x3d\x44\xfe\x18\xf4\x2b\xc0\x22\x10\x11\x98\
\x43\xe0\x37\x10\x37\x91\xbf\x54\x8a\x8c\xf1\xa0\x4d\x1d\xe6\x3e\
\x05\xea\x53\xd0\x39\xf0\x9b\xbf\xe8\x6c\xd1\xe8\x34\xa7\x0d\xf5\
\x9a\x90\x9b\x3e\xe0\x65\x74\xda\x7e\x5f\x76\x63\x90\x60\x6c\x64\
\x4e\xdd\x8a\x23\xaf\x63\xa8\x56\xcc\x82\xf9\xbc\xc0\xa8\xfd\x3d\
\xb0\x81\x02\xd4\x7b\x13\x06\x54\x15\x7f\x96\x2b\x40\x51\x05\x00\
\xb3\xd9\xf4\xed\x68\x34\x7e\xf8\xfd\xc7\x94\x96\xcb\xe5\xfa\xca\
\x1b\x02\xfa\x13\xd2\x34\xc5\x6c\xf6\xf3\x8d\x79\xe2\xb6\xde\x57\
\xaf\x3f\x5c\x23\xa5\x6c\xf3\xe4\xa1\x11\xd7\x8d\x25\xa0\x9c\x4c\
\xc6\x9f\x6e\x1f\xde\x58\x05\x00\xbb\xb8\x76\xfe\xc7\xe4\x0c\xe0\
\x2f\x66\xf1\xe6\xd3\xd5\x05\x2d\x87\x00\x00\x00\x00\x49\x45\x4e\
\x44\xae\x42\x60\x82\
"

qt_resource_name = "\
\x00\x06\
\x07\x03\x7d\xc3\
\x00\x69\
\x00\x6d\x00\x61\x00\x67\x00\x65\x00\x73\
\x00\x08\
\x04\xd2\x59\x47\
\x00\x69\
\x00\x6e\x00\x66\x00\x6f\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x0c\
\x08\x05\xcb\x27\
\x00\x76\
\x00\x69\x00\x65\x00\x77\x00\x6d\x00\x61\x00\x67\x00\x2b\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x0e\
\x0b\xdf\x7d\x87\
\x00\x66\
\x00\x69\x00\x6c\x00\x65\x00\x73\x00\x61\x00\x76\x00\x65\x00\x61\x00\x73\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x0c\
\x05\x68\x0e\x67\
\x00\x66\
\x00\x69\x00\x6c\x00\x65\x00\x73\x00\x61\x00\x76\x00\x65\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x0c\
\x0b\x21\x0f\x87\
\x00\x66\
\x00\x69\x00\x6c\x00\x65\x00\x6f\x00\x70\x00\x65\x00\x6e\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x0d\
\x0c\x95\x22\x87\
\x00\x66\
\x00\x69\x00\x6c\x00\x65\x00\x63\x00\x6c\x00\x6f\x00\x73\x00\x65\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x0c\
\x08\x3f\xcb\x27\
\x00\x76\
\x00\x69\x00\x65\x00\x77\x00\x6d\x00\x61\x00\x67\x00\x31\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x09\
\x07\x3c\x87\xa7\
\x00\x61\
\x00\x70\x00\x70\x00\x6c\x00\x79\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x07\
\x09\xc1\x57\xa7\
\x00\x72\
\x00\x75\x00\x6e\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x0e\
\x02\xa7\xfc\x67\
\x00\x68\
\x00\x65\x00\x6c\x00\x70\x00\x5f\x00\x69\x00\x6e\x00\x64\x00\x65\x00\x78\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x0a\
\x05\x03\x28\x67\
\x00\x6c\
\x00\x79\x00\x62\x00\x6e\x00\x69\x00\x7a\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x0c\
\x08\x03\xcb\x27\
\x00\x76\
\x00\x69\x00\x65\x00\x77\x00\x6d\x00\x61\x00\x67\x00\x2d\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x0b\
\x04\x14\x52\xc7\
\x00\x66\
\x00\x69\x00\x6c\x00\x65\x00\x6e\x00\x65\x00\x77\x00\x2e\x00\x70\x00\x6e\x00\x67\
\x00\x09\
\x0b\xc7\xa4\x87\
\x00\x73\
\x00\x68\x00\x65\x00\x65\x00\x74\x00\x2e\x00\x70\x00\x6e\x00\x67\
"

qt_resource_struct = "\
\x00\x00\x00\x00\x00\x02\x00\x00\x00\x01\x00\x00\x00\x01\
\x00\x00\x00\x00\x00\x02\x00\x00\x00\x0e\x00\x00\x00\x02\
\x00\x00\x01\x0e\x00\x00\x00\x00\x00\x01\x00\x00\x39\x1f\
\x00\x00\x01\x68\x00\x00\x00\x00\x00\x01\x00\x00\x54\x31\
\x00\x00\x00\x12\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\
\x00\x00\x01\x30\x00\x00\x00\x00\x00\x01\x00\x00\x42\x07\
\x00\x00\x00\x68\x00\x00\x00\x00\x00\x01\x00\x00\x15\xdd\
\x00\x00\x00\xe2\x00\x00\x00\x00\x00\x01\x00\x00\x2b\x22\
\x00\x00\x01\x4a\x00\x00\x00\x00\x00\x01\x00\x00\x4c\xd8\
\x00\x00\x00\x28\x00\x00\x00\x00\x00\x01\x00\x00\x06\x3b\
\x00\x00\x00\xc4\x00\x00\x00\x00\x00\x01\x00\x00\x23\xb7\
\x00\x00\x00\xfa\x00\x00\x00\x00\x00\x01\x00\x00\x30\x68\
\x00\x00\x00\x86\x00\x00\x00\x00\x00\x01\x00\x00\x1a\xaf\
\x00\x00\x01\x84\x00\x00\x00\x00\x00\x01\x00\x00\x58\xdf\
\x00\x00\x00\x46\x00\x00\x00\x00\x00\x01\x00\x00\x0d\xae\
\x00\x00\x00\xa4\x00\x00\x00\x00\x00\x01\x00\x00\x21\x9e\
"

def qInitResources():
    QtCore.qRegisterResourceData(0x01, qt_resource_struct, qt_resource_name, qt_resource_data)

def qCleanupResources():
    QtCore.qUnregisterResourceData(0x01, qt_resource_struct, qt_resource_name, qt_resource_data)

qInitResources()


if __name__ == "__main__":
    app = QtGui.QApplication(sys.argv)
    main_win = MainWindow()
    main_win.show()
    sys.exit(app.exec_())
