package latexDraw.generators.svg;

import java.awt.geom.Point2D;
import java.util.Vector;

import latexDraw.figures.Figure;
import latexDraw.figures.LaTeXDrawRectangle;
import latexDraw.figures.Rhombus;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.SVGElement;
import latexDraw.parsers.svg.elements.SVGGElement;
import latexDraw.parsers.svg.elements.SVGPolygonElement;
import latexDraw.parsers.svg.parsers.SVGPointsParser;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines an SVG generator for a polygon.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LRhombusSVGGenerator extends LPolygonSVGGenerator
{

	public LRhombusSVGGenerator(Rhombus f)
	{
		super(f);
	}


	
	public LRhombusSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	/**
	 * Creates a rhombus from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LRhombusSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new Rhombus(false));

		setNumber(elt);
		SVGElement elt2 = getLaTeXDrawElement(elt, null);
		
		if(elt==null || elt2==null || !(elt2 instanceof SVGPolygonElement))
			throw new IllegalArgumentException();
		
		SVGPolygonElement main = (SVGPolygonElement)elt2;
		Rhombus r = (Rhombus)getShape();
		setSVGLatexdrawParameters(elt);
		setSVGParameters(main);
		
		Vector<Point2D> ptsPol = SVGPointsParser.getPoints(elt.getAttribute(
								 elt.getUsablePrefix(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE_URI)+LaTeXDrawNamespace.XML_POINTS));
		
		if(ptsPol==null || ptsPol.size()!=LaTeXDrawRectangle.NB_POINTS_FRAME)
			throw new IllegalArgumentException();
		
		r.getPoint(0).setLocation(ptsPol.elementAt(0));
		r.getPoint(1).setLocation(ptsPol.elementAt(1));
		r.getPoint(2).setLocation(ptsPol.elementAt(2));
		r.getPoint(3).setLocation(ptsPol.elementAt(3));
		
		r.updateShape();
		setSVGShadowParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHADOW));
		setSVGDbleBordersParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_DBLE_BORDERS));
		
		if(withTransformation)
			applyTransformations(elt);
	}
	

	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc==null)
			return null;
		
		Rhombus r = (Rhombus)shape;
		LaTeXDrawPoint2D nw = r.getTheNWPoint(), se = r.getTheSEPoint();
		LaTeXDrawPoint2D gc = r.getGravityCenter();
		LaTeXDrawPoint2D p1 = new LaTeXDrawPoint2D((nw.x+se.x)/2., nw.y);
		LaTeXDrawPoint2D p2 = new LaTeXDrawPoint2D(se.x, (nw.y+se.y)/2.);
		LaTeXDrawPoint2D p3 = new LaTeXDrawPoint2D((nw.x+se.x)/2., se.y);
		SVGElement root = new SVGGElement(doc), elt;
		root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_RHOMBUS);
		root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
	    double gap = getPositionGap()/2.;
	    double cornerGap1 = Figure.getCornerGap(gc, p1, p2, gap);
	    double cornerGap2 = Figure.getCornerGap(gc, p2, p3, gap);
	    
	    if(p2.x<p3.x)
	    	cornerGap2*=-1;
	    
		String points =  (float)p1.x + "," + (float)(p1.y - cornerGap1) + " " + //$NON-NLS-1$ //$NON-NLS-2$
						 (float)(p2.x + cornerGap2) + "," +  (float)p2.y + " " + //$NON-NLS-1$//$NON-NLS-2$
						 (float)p3.x + "," + (float)(p3.y + cornerGap1) + " " + //$NON-NLS-1$//$NON-NLS-2$
						 (float)(nw.x - cornerGap2)  + "," + (float)p2.y;//$NON-NLS-1$
	    String ltdPoints = r.getPoint(0).x + " " + r.getPoint(0).y + " " + r.getPoint(1).x + " " + r.getPoint(1).y +//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
							" " + r.getPoint(2).x + " " + r.getPoint(2).y + " " + r.getPoint(3).x + " " + r.getPoint(3).y;//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$

		if(r.hasShadow())
		{
			SVGElement shad = new SVGPolygonElement(doc);
	    	
	   		shad.setAttribute(SVGAttributes.SVG_POINTS, points);
	   		setSVGShadowAttributes(shad, true);
	   		root.appendChild(shad);
		}
		
        if(shape.hasShadow() && !shape.getLineStyle().equals(PSTricksConstants.LINE_NONE_STYLE))
        {// The background of the borders must be filled is there is a shadow.
    		elt = new SVGPolygonElement(doc);
    		elt.setAttribute(SVGAttributes.SVG_POINTS, points);
    		setSVGBorderBackground(elt, root);
        }
        
		elt = new SVGPolygonElement(doc);
		elt.setAttribute(SVGAttributes.SVG_POINTS, points);
		root.appendChild(elt);
		root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_POINTS, ltdPoints);
		
		if(r.hasDoubleBoundary())
		{
			SVGElement dblBord = new SVGPolygonElement(doc);
			
			dblBord.setAttribute(SVGAttributes.SVG_POINTS, points);
			setSVGDoubleBordersAttributes(dblBord);
			root.appendChild(dblBord);
		}
		
		setSVGAttributes(doc, elt, true);
		setSVGRotationAttribute(root);
		
		return root;
	}
}
