package junit.test.svg;

import java.net.URI;
import java.net.URISyntaxException;

import junit.framework.TestCase;
import latexDraw.parsers.svg.*;
import latexDraw.parsers.svg.elements.SVGElement;
import latexDraw.parsers.svg.elements.SVGSVGElement;

import org.junit.Before;
import org.junit.Test;
import org.w3c.dom.DOMException;
import org.w3c.dom.Node;

/** 
 * This class contains tests for the SVGDocument class.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 03/31/08<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class TestSVGDocument extends TestCase
{
	protected SVGDocument doc1;
	protected SVGDocument doc2;
	
	
	@Override
	@Before
	public void setUp()
	{
		doc1 = new SVGDocument();
		
		try { doc2 = new SVGDocument(new URI("junit/test/svg/test.svg")); }
		catch(MalformedSVGDocument e){ fail(); }
		catch(URISyntaxException e){ fail(); }
	}
	
	
	
	@Test
	public void testSVGDocument()
	{
		try
		{
			new SVGDocument(null);
			fail();
		}
		catch(IllegalArgumentException e){ /* ok */ }
		catch(MalformedSVGDocument e){ fail(); }
		
		try
		{
			new SVGDocument(new URI("dfqsfg"));
			fail();
		}
		catch(MalformedSVGDocument e){ /* ok */ }
		catch(URISyntaxException e){ fail(); }
		
		try
		{
			SVGDocument doc = new SVGDocument(new URI("junit/test/svg/test.svg"));
			assertNotNull(doc.getFirstChild());
			assertNotNull(doc.getLastChild());
		}
		catch(MalformedSVGDocument e){ fail(); }
		catch(URISyntaxException e){ fail(); }
	}
	
	
	@Test
	public void testSVGDocument2()
	{
		SVGDocument doc = new SVGDocument();
		
		assertNotNull(doc.getFirstChild());
		assertNotNull(doc.getLastChild());
		assertNull(doc.getDocumentURI());
	}
	
	
	@Test
	public void testToLaTeXDraw()
	{
//		LShape shape;
//		
//		shape = doc1.toLaTeXDraw(null);
//		shape = doc2.toLaTeXDraw(null);//TODO
	}
	
	
	@Test
	public void testAdoptNode()
	{
		SVGSVGElement elt = new SVGSVGElement(doc2);
		
		try
		{
			doc1.adoptNode(null);
			fail();
		}
		catch(DOMException e){ /* ok */ }
		
		try
		{
			doc1.adoptNode(new SVGComment("", doc2));
			fail();
		}
		catch(DOMException e){ /* ok */ }
		
		doc1.adoptNode(elt);
		assertEquals(doc1, elt.getOwnerDocument());
		assertEquals(doc1.getFirstChild(), elt);
	}
	
	
	@Test
	public void testGetDocumentElement()
	{
		assertNotNull(doc1.getDocumentElement());
	}
	
	
	@Test
	public void testSetDocumentURI()
	{
		doc1.setDocumentURI(null);
		assertNull(doc1.getDocumentURI());
		doc1.setDocumentURI("coucou");
		assertEquals("coucou", doc1.getDocumentURI());
	}
	
	
	@Test
	public void testSetXmlStandalone()
	{
		doc1.setXmlStandalone(false);
		assertFalse(doc1.getXmlStandalone());
		doc1.setXmlStandalone(true);
		assertTrue(doc1.getXmlStandalone());
	}
	
	
	@Test
	public void testSetXmlVersion()
	{
		doc1.setXmlVersion(null);
		assertNull(doc1.getXmlVersion());
		doc1.setXmlVersion("coucou");
		assertEquals("coucou", doc1.getXmlVersion());
	}
	
	
	@Test
	public void testUselessMethods()
	{
		assertNull(doc1.getAttributes());
		assertNull(doc1.getChildNodes());
		assertNull(doc1.getNextSibling());
		assertNull(doc1.getPreviousSibling());
		assertNull(doc2.getAttributes());
		assertNull(doc2.getChildNodes());
		assertNull(doc2.getNextSibling());
		assertNull(doc2.getPreviousSibling());
		assertNull(doc2.getNodeValue());
		assertNull(doc2.getOwnerDocument());
		assertNull(doc2.getParentNode());
		assertNull(doc1.getNodeValue());
		assertNull(doc1.getOwnerDocument());
		assertNull(doc1.getParentNode());
		assertFalse(doc2.hasAttributes());
		assertFalse(doc1.hasAttributes());
		assertNull(doc1.getDoctype());
	}
	
	
	@Test
	public void testGetFirstChild()
	{
		assertNotNull(doc1.getFirstChild());
		assertNotNull(doc2.getFirstChild());
	}
	
	
	@Test
	public void testGetLastChild()
	{
		assertNotNull(doc1.getLastChild());
		assertNotNull(doc2.getLastChild());
		assertEquals(doc1.getLastChild(), doc1.getFirstChild());
		assertEquals(doc2.getLastChild(), doc2.getFirstChild());
	}
	
	
	@Test
	public void testGetNodeName()
	{
		assertEquals("#document", doc1.getNodeName());
		assertEquals("#document", doc2.getNodeName());
	}
	
	
	@Test
	public void testGetNodeType()
	{
		assertEquals(Node.DOCUMENT_NODE, doc1.getNodeType());
		assertEquals(Node.DOCUMENT_NODE, doc2.getNodeType());
	}
	
	
	@Test
	public void testHasChildNode()
	{
		assertTrue(doc1.hasChildNodes());
		assertTrue(doc2.hasChildNodes());
	}
	
	
	@Test
	public void testIsEqualNode()
	{
		SVGDocument doc = new SVGDocument();
		assertTrue(doc1.isEqualNode(doc));
		assertFalse(doc1.isEqualNode(null));
		assertFalse(doc1.isEqualNode(doc2));
		try 
		{ 
			doc = new SVGDocument(new URI("junit/test/svg/test.svg"));
			assertTrue(doc2.isEqualNode(doc));
			assertFalse(doc2.isEqualNode(null));
			assertFalse(doc2.isEqualNode(doc1));
		}
		catch(MalformedSVGDocument e){ fail(); }
		catch(URISyntaxException e){ fail(); }
	}
	
	
	
	@Test
	public void testIsSameNode()
	{
		assertTrue(doc1.isEqualNode(doc1));
		assertTrue(doc2.isEqualNode(doc2));
		assertFalse(doc1.isEqualNode(null));
		assertFalse(doc2.isEqualNode(null));
		assertFalse(doc1.isEqualNode(doc2));
		assertFalse(doc2.isEqualNode(doc1));
	}
	
	
	@Test
	public void testCreateElement()
	{
		try
		{
			doc1.createElement(null);
			fail();
		}
		catch(DOMException e) { /* ok */ }
		
		SVGElement elt = (SVGElement)doc1.createElement("test");
		assertEquals(elt.getNodeName(), "test");
		assertEquals(doc1, elt.getOwnerDocument());
	}
	
	
	@Test
	public void testCreateTextNode()
	{
		try
		{
			doc1.createTextNode(null);
			fail();
		}
		catch(DOMException e) { /* ok */ }
		
		SVGText elt = (SVGText)doc1.createTextNode("test");
		assertEquals(elt.getData(), "test");
		assertEquals(doc1, elt.getOwnerDocument());
	}
	
	
	@Test
	public void testCreateComment()
	{
		try
		{
			doc1.createComment(null);
			fail();
		}
		catch(DOMException e) { /* ok */ }
		
		SVGComment elt = (SVGComment)doc1.createComment("test");
		assertEquals(elt.getData(), "test");
		assertEquals(doc1, elt.getOwnerDocument());
	}
	
	
	@Test
	public void testCreateCDATASection()
	{
		try
		{
			doc1.createCDATASection(null);
			fail();
		}
		catch(DOMException e) { /* ok */ }
		
		SVGCDATASection elt = (SVGCDATASection)doc1.createCDATASection("test");
		assertEquals(elt.getData(), "test");
		assertEquals(doc1, elt.getOwnerDocument());
	}
}
