/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "imarker.h"


#include "icamera.h"
#include "icaption.h"
#include "idata.h"
#include "ierror.h"
#include "ipointglyph.h"
#include "irendertool.h"
#include "ireplicatedactor.h"
#include "iviewmodule.h"
#include "iviewobject.h"
#include "iviewobjectfamily.h"

#include <vtkProperty.h>
#include <vtkProperty2D.h>
#include <vtkTextProperty.h>

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


IVIEWSUBJECT_DEFINE_TYPE(iMarker,Marker,m);

IOBJECT_DEFINE_KEY(iMarker,ColorAutomatic,ca,Bool,1);
IOBJECT_DEFINE_KEY(iMarker,CaptionText,ct,String,1);
IOBJECT_DEFINE_KEY(iMarker,CaptionPosition,cx,Float,2);
IOBJECT_DEFINE_KEY(iMarker,InteractiveMove,im,Bool,1);
IOBJECT_DEFINE_KEY(iMarker,Scaled,sc,Bool,1);
IOBJECT_DEFINE_KEY(iMarker,Type,t,Int,1);
IOBJECT_DEFINE_KEY(iMarker,Transform,tf,Float,6);

IOBJECT_DEFINE_DISTANCE_KEY(iMarker,Size,s);

IOBJECT_DEFINE_KEY(iMarker,TypeAsString,-tas,String,1);

//
//  Inherited keys
//
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COMMON(iMarker);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_POSITION(iMarker);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COLOR(iMarker,1);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_OPACITY(iMarker,1);


iMarker* iMarker::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	iMarker *tmp = new iMarker(vm); IERROR_ASSERT(tmp); // non-inheritable, so no need to use ExtensionFactory
	tmp->Configure();
	return tmp;
}


iMarker::iMarker(iViewModule *vm) : iSolidViewSubject(vm,iDataType::Null(),"Marker",1,ViewSubject::Flag::AlwaysShaded|ViewSubject::Flag::NoReplicating|ViewSubject::Flag::HasPosition|ViewSubject::Flag::NoPalette), mSize(vm,true)
{
	mSubjectId = ViewSubject::Id::Undefined;

	mSize = 0.08;
	mColorAuto = true;

	mCaptionText = "";

	mInteractiveMove = false;

	mTransform[0] = mTransform[1] = mTransform[2] = 0.0;
	mTransform[3] = mTransform[4] = mTransform[5] = 1.0;
	//
	//  Do VTK stuff
	//	
	mActors[0]->SetScaled(true);
	mObject = iPointGlyph::New(this); IERROR_ASSERT(mObject);
	mCaption = iCaption::New(vm->GetRenderTool()); IERROR_ASSERT(mCaption);

	mObject->SetType(PointGlyphType::Sphere);

	mActors[0]->SetInput(mObject->GetOutput());
	
	//
	//  Add observer to keep information about this object
	//
	const iPosition x(this->GetViewModule());
	float xc[2] = { 0.0, 0.0 };

	this->SetColor(0,iColor(255,0,0));
	this->SetPosition(x);
	this->SetCaptionPosition(xc);
	this->SetSize(mSize);

	this->GetViewModule()->AddObject(mCaption);
	mCaption->VisibilityOff();
}


iMarker::~iMarker()
{
	if(mIsInitialized) 
	{
		this->GetViewModule()->RemoveObject(mCaption);
		//
		//  Update attached objects
		//
		for(int j=0; j<mAttachedObjects.Size(); j++) mAttachedObjects[j]->UpdateAfterMarkerDelete(this);
	}
	this->GetViewModule()->RemoveObject(mCaption);
	mCaption->Delete();
	mObject->Delete();
}


void iMarker::FinishInitialization()
{
}


void iMarker::SetType(int m)
{
	mObject->SetType(m);
	this->ClearCache();
}


int iMarker::GetType() const
{
	return mObject->GetType();
}


const iString iMarker::GetTypeAsString() const
{
	return mObject->GetName();
}


void iMarker::SetScaled(bool s)
{
	mActors[0]->SetScaled(s);
	this->ClearCache();
}


bool iMarker::GetScaled() const
{
	return mActors[0]->GetScaled();
}


void iMarker::UpdatePosition(const iPosition &)
{
	this->Move(mPosition);
	if(!mInteractiveMove) this->FinishMoving(); // always finish
	this->ClearCache();
}


void iMarker::Move(const iPosition& p)
{
	mPosition = p;
	mActors[0]->SetPosition(mPosition);
	mCaption->SetAttachmentPoint(mPosition);
	if(mInteractiveMove) this->FinishMoving();
	this->ClearCache();
}


void iMarker::FinishMoving()
{
	//
	//  Update attached objects
	//
	int j;
	for(j=0; j<mAttachedObjects.Size(); j++) mAttachedObjects[j]->UpdateAfterMarkerMove();
}


void iMarker::SetCaptionPosition(const float *x)
{
	this->SetCaptionPosition(x[0],x[1]);
}


void iMarker::SetCaptionPosition(float x, float y)
{
	mCapPos[0] = x;
	mCapPos[1] = y;
	mCaption->SetPosition(x,y);
	this->ClearCache();
}


void iMarker::SetSize(double s)
{
	if(s > 0.0)
	{
		iDistance tmp(this->GetViewModule(),true);
		tmp = s;
		this->SetSize(tmp);
	}
}


void iMarker::SetSize(const iDistance &s)
{
	if(s < 10.01)
	{
		mSize = s;
		mActors[0]->SetBasicScale(mSize);
		float ps = 10.0*pow(s,0.3);
		if(ps < 1.0) ps = 1.0;
		if(ps > 10.0) ps = 10.0;
		mActors[0]->GetProperty()->SetPointSize(ps);
		this->ClearCache();
	}
}


void iMarker::SetCaptionText(const iString &s)
{
	mCaptionText = s;
	mCaption->SetText(s);
	if(s.Length() > 0) mCaption->VisibilityOn(); else mCaption->VisibilityOff();
	this->ClearCache();
}


void iMarker::SetInteractiveMove(bool s)
{
	mInteractiveMove = s;
	this->ClearCache();
}


void iMarker::SetTransform(const float *t)
{
	int i;
	for(i=0; i<6; i++) mTransform[i] = t[i];
	mActors[0]->SetAxisScale(t[3],t[4],t[5]);
	mActors[0]->SetOrientation(t[0],t[1],t[2]);
	this->ClearCache();
}


void iMarker::ShowBody(bool show)
{
	if(show)
	{
		mActors[0]->VisibilityOn();
		if(mCaptionText.Length() > 0) mCaption->VisibilityOn(); else mCaption->VisibilityOff(); 
	} 
	else 
	{
		mActors[0]->VisibilityOff();
		mCaption->VisibilityOff();
	}
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iMarker::SolidViewSubjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyColorAutomatic(),mColorAuto);
	this->PackValue(s,KeyInteractiveMove(),mInteractiveMove);
	this->PackValue(s,KeyScaled(),this->GetScaled());

	this->PackValue(s,KeyType(),this->GetType());

	this->PackValue(s,KeyCaptionText(),mCaptionText);

	this->PackValueDistance(s,KeySize(),mSize);
	this->PackValue(s,KeyCaptionPosition(),mCapPos,2);

	this->PackValue(s,KeyTransform(),mTransform,6);

	this->PackValue(s,KeyTypeAsString(),this->GetTypeAsString());
}


void iMarker::SolidViewSubjectUnPackStateBody(const iString &s)
{
	int i; bool b; float fv[6]; iColor c; iString z;

	if(this->UnPackValue(s,KeyColorAutomatic(),b))
	{
		mColorAuto = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyInteractiveMove(),b)) this->SetInteractiveMove(b);
	if(this->UnPackValue(s,KeyScaled(),b)) this->SetScaled(b);

	if(this->UnPackValue(s,KeyType(),i)) this->SetType(i);

	if(this->UnPackValue(s,KeyCaptionText(),z)) this->SetCaptionText(z);

	if(this->UnPackValueDistance(s,KeySize(),mSize)) this->SetSize(mSize);

	fv[0] = mCapPos[0]; fv[1] = mCapPos[1];
	if(this->UnPackValue(s,KeyCaptionPosition(),fv,2)) this->SetCaptionPosition(fv);

	for(i=0; i<6; i++) fv[i] = mTransform[i];
	if(this->UnPackValue(s,KeyTransform(),fv,6)) this->SetTransform(fv);
}


void iMarker::ShowColorBarsBody(bool)
{
}


void iMarker::ResetPipeline()
{
}


void iMarker::AttachViewSubject(iViewSubject *o)
{
	mAttachedObjects.AddUnique(o);
}


void iMarker::DetachViewSubject(iViewSubject *o)
{
	mAttachedObjects.Remove(o);
}


bool iMarker::CanBeShown() const
{
	return true;
}


void iMarker::ViewSubjectSyncWithData(const iDataSyncRequest &)
{
	//
	//  Attach all vectors
	//
	int j;
	for(j=0; j<=this->GetViewModule()->GetViewObjectFamily(ViewSubject::Id::VectorField)->GetMaxMemberIndex(); j++)
	{
		this->AttachViewSubject(this->GetViewModule()->GetViewObjectFamily(ViewSubject::Id::VectorField)->GetMember(j)->GetViewSubject());
	}
	//
	//  Update attached objects
	//
	for(j=0; j<mAttachedObjects.Size(); j++) mAttachedObjects[j]->UpdateAfterMarkerAdd(this);
}

