/*
 * Copyright (C) 2012 Hermann Meyer, Andreas Degert, Pete Shorthose, Steve Poskitt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * --------------------------------------------------------------------------
 */


////////////////////////////// LOCAL INCLUDES //////////////////////////

#include "gx_common.h"      // faust support and denormal protection (SSE)
#include "gx_mbecho.h"        // define struct PortIndex
#include "gx_pluginlv2.h"   // define struct PluginLV2
#include "mbe.cc"    // dsp class generated by faust -> dsp2cc

////////////////////////////// PLUG-IN CLASS ///////////////////////////

class Gx_mbecho_
{
private:
  // pointer to buffer
  float*      output;
  float*      input;
  // pointer to dsp class
  PluginLV2*  mbecho;
  // private functions
  inline void run_dsp_(uint32_t n_samples);
  inline void connect_(uint32_t port,void* data);
  inline void init_dsp_(uint32_t rate);
  inline void connect_all__ports(uint32_t port, void* data);
  inline void activate_f();
  inline void clean_up();
  inline void deactivate_f();

public:
  // LV2 Descriptor
  static const LV2_Descriptor descriptor;
  // static wrapper to private functions
  static void deactivate(LV2_Handle instance);
  static void cleanup(LV2_Handle instance);
  static void run(LV2_Handle instance, uint32_t n_samples);
  static void activate(LV2_Handle instance);
  static void connect_port(LV2_Handle instance, uint32_t port, void* data);
  static LV2_Handle instantiate(const LV2_Descriptor* descriptor,
                                double rate, const char* bundle_path,
                                const LV2_Feature* const* features);
  Gx_mbecho_();
  ~Gx_mbecho_();
};

// constructor
Gx_mbecho_::Gx_mbecho_() :
  output(NULL),
  input(NULL),
  mbecho(mbe::plugin()) {};

// destructor
Gx_mbecho_::~Gx_mbecho_()
{
  // just to be sure the plug have given free the allocated mem
  // it didn't hurd if the mem is already given free by clean_up()
  if (mbecho->activate_plugin !=0)
    mbecho->activate_plugin(false, mbecho);
  // delete DSP class
  mbecho->delete_instance(mbecho);
};

///////////////////////// PRIVATE CLASS  FUNCTIONS /////////////////////

void Gx_mbecho_::init_dsp_(uint32_t rate)
{
  AVOIDDENORMALS(); // init the SSE denormal protection
  mbecho->set_samplerate(rate, mbecho); // init the DSP class
}

// connect the Ports used by the plug-in class
void Gx_mbecho_::connect_(uint32_t port,void* data)
{
  switch ((PortIndex)port)
    {
    case EFFECTS_OUTPUT:
      output = static_cast<float*>(data);
      break;
    case EFFECTS_INPUT:
      input = static_cast<float*>(data);
      break;
    default:
      break;
    }
}

void Gx_mbecho_::activate_f()
{
  // allocate the internal DSP mem
  if (mbecho->activate_plugin !=0)
    mbecho->activate_plugin(true, mbecho);
}

void Gx_mbecho_::clean_up()
{
  // delete the internal DSP mem
  if (mbecho->activate_plugin !=0)
    mbecho->activate_plugin(false, mbecho);
}

void Gx_mbecho_::deactivate_f()
{
  // delete the internal DSP mem
  if (mbecho->activate_plugin !=0)
    mbecho->activate_plugin(false, mbecho);
}

void Gx_mbecho_::run_dsp_(uint32_t n_samples)
{
  if (n_samples< 1) return;
  mbecho->mono_audio(static_cast<int>(n_samples), input, output, mbecho);
}

void Gx_mbecho_::connect_all__ports(uint32_t port, void* data)
{
  // connect the Ports used by the plug-in class
  connect_(port,data); 
  // connect the Ports used by the DSP class
  mbecho->connect_ports(port,  data, mbecho);
}

////////////////////// STATIC CLASS  FUNCTIONS  ////////////////////////

LV2_Handle 
Gx_mbecho_::instantiate(const LV2_Descriptor* descriptor,
                            double rate, const char* bundle_path,
                            const LV2_Feature* const* features)
{
  // init the plug-in class
  Gx_mbecho_ *self = new Gx_mbecho_();
  if (!self)
    {
      return NULL;
    }

  self->init_dsp_((uint32_t)rate);

  return (LV2_Handle)self;
}

void Gx_mbecho_::connect_port(LV2_Handle instance, 
                                   uint32_t port, void* data)
{
  // connect all ports
  static_cast<Gx_mbecho_*>(instance)->connect_all__ports(port, data);
}

void Gx_mbecho_::activate(LV2_Handle instance)
{
  // allocate needed mem
  static_cast<Gx_mbecho_*>(instance)->activate_f();
}

void Gx_mbecho_::run(LV2_Handle instance, uint32_t n_samples)
{
  // run dsp
  static_cast<Gx_mbecho_*>(instance)->run_dsp_(n_samples);
}

void Gx_mbecho_::deactivate(LV2_Handle instance)
{
  // free allocated mem
  static_cast<Gx_mbecho_*>(instance)->deactivate_f();
}

void Gx_mbecho_::cleanup(LV2_Handle instance)
{
  // well, clean up after us
  Gx_mbecho_* self = static_cast<Gx_mbecho_*>(instance);
  self->clean_up();
  delete self;
}

const LV2_Descriptor Gx_mbecho_::descriptor =
{
  GXPLUGIN_URI "#_mbecho_",
  Gx_mbecho_::instantiate,
  Gx_mbecho_::connect_port,
  Gx_mbecho_::activate,
  Gx_mbecho_::run,
  Gx_mbecho_::deactivate,
  Gx_mbecho_::cleanup,
  NULL
};

////////////////////////// LV2 SYMBOL EXPORT ///////////////////////////

LV2_SYMBOL_EXPORT
const LV2_Descriptor*
lv2_descriptor(uint32_t index)
{
  switch (index)
    {
    case 0:
      return &Gx_mbecho_::descriptor;
    default:
      return NULL;
    }
}

///////////////////////////// FIN //////////////////////////////////////
