/* 
 * Copyright (C) 2004, 2005 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <gnome.h>
#include <glade/glade.h>
#include <eel/eel-alert-dialog.h>
#include <eel/eel.h>
#include <translate.h>
#include "gt-util.h"

typedef struct
{
  gpointer		instance;
  unsigned long		id;
} SignalHandler;

static GtkWidget *gt_error_dialog_real (GtkWindow *parent,
					gboolean blocking,
					const char *primary,
					const char *secondary);
static void gt_g_object_connect_weak_notify_cb (gpointer data, GObject *former_object);

static GtkWidget *
gt_error_dialog_real (GtkWindow *parent,
		      gboolean blocking,
		      const char *primary,
		      const char *secondary)
{
  GtkWidget *dialog;

  dialog = eel_alert_dialog_new(parent,
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				primary,
				secondary,
				NULL);

  if (blocking)
    {
      gtk_dialog_run(GTK_DIALOG(dialog));
      gtk_widget_destroy(dialog);
      return NULL;
    }
  else
    {
      g_signal_connect(dialog, "response", G_CALLBACK(gtk_widget_destroy), NULL);
      gtk_widget_show(dialog);
      return dialog;
    }
}

GtkWidget *
gt_error_dialog (GtkWindow *parent,
		 const char *primary,
		 const char *format,
		 ...)
{
  char *secondary;
  GtkWidget *dialog;

  if (format)
    {
      va_list args;
      
      va_start(args, format);
      secondary = g_strdup_vprintf(format, args);
      va_end(args);
    }
  else
    secondary = NULL;

  dialog = gt_error_dialog_real(parent, FALSE, primary, secondary);
  g_free(secondary);

  return dialog;
}

void
gt_fatal_error_dialog (GtkWindow *parent, const char *format, ...)
{
  va_list args;
  char *secondary;

  g_return_if_fail(format != NULL);

  va_start(args, format);
  secondary = g_strdup_vprintf(format, args);
  va_end(args);

  gt_error_dialog_real(parent, TRUE, _("A fatal error has occurred"), secondary);
  g_free(secondary);

  exit(1);  
}

void
gt_create_interface (const char *name, ...)
{
  char *filename;
  char *pathname;
  GladeXML *xml;
  va_list args;
  const char *widget_name;

  g_return_if_fail(name != NULL);

  filename = g_strconcat(name, ".glade", NULL);
  pathname = g_build_filename(UIDIR, filename, NULL);
  g_free(filename);

  xml = glade_xml_new(pathname, NULL, NULL);
  g_free(pathname);

  g_return_if_fail(xml != NULL);
  glade_xml_signal_autoconnect(xml);

  va_start(args, name);
  while ((widget_name = va_arg(args, const char *)))
    {
      GtkWidget **widget;

      widget = va_arg(args, GtkWidget **);
      g_return_if_fail(widget != NULL);

      *widget = glade_xml_get_widget(xml, widget_name);
      if (! *widget)
	g_critical(_("widget \"%s\" not found in interface \"%s\""), widget_name, name);
    }
  va_end(args);
  
  g_object_unref(xml);
}

void
gt_text_buffer_clear (GtkTextBuffer *buffer)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;

  g_return_if_fail(GTK_IS_TEXT_BUFFER(buffer));

  gtk_text_buffer_get_bounds(buffer, &start_iter, &end_iter);
  gtk_text_buffer_delete(buffer, &start_iter, &end_iter);
}

char *
gt_text_buffer_get_all_text (GtkTextBuffer *buffer)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;

  g_return_val_if_fail(GTK_IS_TEXT_BUFFER(buffer), NULL);

  gtk_text_buffer_get_bounds(buffer, &start_iter, &end_iter);
  return gtk_text_buffer_get_text(buffer, &start_iter, &end_iter, TRUE);
}

GnomeVFSResult
gt_vfs_write_file (const char *uri, const char *contents)
{
  GnomeVFSResult result;
  GnomeVFSHandle *handle;

  result = gnome_vfs_create(&handle, uri, GNOME_VFS_OPEN_WRITE, FALSE,
			    S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
  if (result == GNOME_VFS_OK)
    {
      GnomeVFSFileSize bytes_written;

      result = gnome_vfs_write(handle, contents, strlen(contents), &bytes_written);
      if (result == GNOME_VFS_OK)
	result = gnome_vfs_close(handle);
      else
	gnome_vfs_close(handle); /* discard this result */
    }
  
  return result;
}

gboolean
gt_vfs_uri_exists (const char *text_uri)
{
  GnomeVFSURI *uri;
  gboolean exists;

  uri = gnome_vfs_uri_new(text_uri);
  exists = gnome_vfs_uri_exists(uri);
  gnome_vfs_uri_unref(uri);

  return exists;
}

GSList *
gt_g_utf8_slist_find (GSList *list, const char *str)
{
  g_return_val_if_fail(str != NULL, NULL);

  return g_slist_find_custom(list, str, (GCompareFunc) translate_utf8_strcmp);
}

char *
gt_vfs_read_entire_file (const char *uri, GError **err)
{
  GnomeVFSResult result;
  int size;
  char *contents;
  char *converted = NULL;

  g_return_val_if_fail(uri != NULL, NULL);

  result = gnome_vfs_read_entire_file(uri, &size, &contents);
  if (result == GNOME_VFS_OK)
    {
      if (contents)
	{
	  if (g_utf8_validate(contents, size, NULL))
	    converted = g_strndup(contents, size);
	  else
	    {
	      GError *tmp_err = NULL;

	      converted = g_locale_to_utf8(contents, size, NULL, NULL, &tmp_err);
	      if (! converted)
		{
		  g_set_error(err, 0, 0, _("unable to convert contents to UTF-8: %s"), tmp_err->message);
		  g_error_free(tmp_err);
		}
	    }
	}
      else
	converted = g_strdup("");
    }
  else
    g_set_error(err, 0, 0, "%s", gnome_vfs_result_to_string(result));

  return converted;
}

GSList *
gt_g_slist_copy_deep (const GSList *list,
		      GTCopyFunc copy_func,
		      gpointer user_data)
{
  const GSList *l;
  GSList *copy = NULL;

  g_return_val_if_fail(copy_func != NULL, NULL);

  GT_LIST_FOREACH(l, list)
    copy = g_slist_append(copy, copy_func(l->data, user_data));

  return copy;
}

void
gt_g_object_slist_free (GSList *list)
{
  eel_g_slist_free_deep_custom(list, (GFunc) g_object_unref, NULL);
}

void
gt_thread_create (GtkWindow *parent, GThreadFunc func, gpointer data)
{
  GError *err = NULL;

  g_return_if_fail(func != NULL);
  
  if (! g_thread_create(func, data, FALSE, &err))
    {
      gt_fatal_error_dialog(parent, _("Unable to create a thread: %s."), err->message);
      g_error_free(err);
    }
}

/**
 * gt_g_object_connect:
 * @object: the object to associate the handlers with
 * @instance: the instance to connect to
 * @signal_spec: the spec for the first signal
 * @...: #GCallback for the first signal, followed by data for the
 *       first signal, followed optionally by more signal spec/callback/data
 *       triples, followed by NULL
 *
 * Connects to one or more signals of @instance, associating the
 * handlers with @object. The handlers will be disconnected whenever
 * @object is finalized.
 *
 * Note: this function is not thread-safe. If @object and @instance
 * are finalized concurrently, the behaviour is undefined.
 *
 * The signals specs must be in the same format than those passed to
 * g_object_connect(), except that object-signal,
 * swapped-object-signal, object-signal-after and
 * swapped-object-signal-after are not accepted.
 *
 * Return value: @object
 **/
gpointer
gt_g_object_connect (gpointer object,
		     gpointer instance,
		     const char *signal_spec,
		     ...)
{
  va_list args;

  g_return_val_if_fail(G_IS_OBJECT(object), NULL);
  g_return_val_if_fail(G_IS_OBJECT(instance), NULL);

  va_start(args, signal_spec);
  while (signal_spec)
    {
      GCallback callback = va_arg(args, GCallback);
      gpointer data = va_arg(args, gpointer);
      SignalHandler *handler;

      handler = g_new(SignalHandler, 1);
      handler->instance = instance;

      if (g_str_has_prefix(signal_spec, "signal::"))
	handler->id = g_signal_connect(instance, signal_spec + 8, callback, data);
      else if (g_str_has_prefix(signal_spec, "swapped_signal")
	       || g_str_has_prefix(signal_spec, "swapped-signal"))
	handler->id = g_signal_connect_swapped(instance, signal_spec + 16, callback, data);
      else if (g_str_has_prefix(signal_spec, "signal_after::")
	       || g_str_has_prefix(signal_spec, "signal-after::"))
	handler->id = g_signal_connect_after(instance, signal_spec + 14, callback, data);
      else if (g_str_has_prefix(signal_spec, "swapped_signal_after::")
	       || g_str_has_prefix(signal_spec, "swapped-signal-after::"))
	handler->id = g_signal_connect_data(instance, signal_spec + 22, callback, data, NULL, G_CONNECT_AFTER | G_CONNECT_SWAPPED);
      else
	g_critical(_("invalid signal specification \"%s\""), signal_spec);

      eel_add_weak_pointer(&handler->instance);
      g_object_weak_ref(object, gt_g_object_connect_weak_notify_cb, handler);

      signal_spec = va_arg(args, const char *);
    }
  va_end(args);

  return object;
}

static void
gt_g_object_connect_weak_notify_cb (gpointer data, GObject *former_object)
{
  SignalHandler *handler = data;

  if (handler->instance)
    {
      g_signal_handler_disconnect(handler->instance, handler->id);
      eel_remove_weak_pointer(&handler->instance);
    }
  g_free(handler);
}

double
gt_get_current_time (void)
{
  GTimeVal now;

  g_get_current_time(&now);
  
  return now.tv_sec + (double) now.tv_usec / 1000000;
}

const char *
gt_dot_dir (void)
{
  static char *dot_dir = NULL;

  if (! dot_dir)
    dot_dir = g_build_filename(g_get_home_dir(),
			       GNOME_DOT_GNOME,
			       "gnome-translate",
			       NULL);

  return dot_dir;
}

GList *
gt_window_icon_list_new (const char *icon_name)
{
  GList *list = NULL;
  int sizes[] = { 16, 32, 48, 64, 128 }; /* recommended in gtk_window_set_icon_list() doc */
  int i;
  GtkIconTheme *icon_theme;
  
  g_return_val_if_fail(icon_name != NULL, NULL);
  
  icon_theme = gtk_icon_theme_get_default();

  for (i = 0; i < G_N_ELEMENTS(sizes); i++)
    {
      GtkIconInfo *icon_info;
      
      icon_info = gtk_icon_theme_lookup_icon(icon_theme, icon_name, sizes[i], 0);
      if (icon_info)
	{
	  /*
	   * Since the purpose of a window list icon is to avoid
	   * scaling, we only add icons whose natural size match the
	   * requested size exactly.
	   */
	  if (gtk_icon_info_get_base_size(icon_info) == sizes[i])
	    {
	      GdkPixbuf *pixbuf;
	      GError *err = NULL;

	      pixbuf = gtk_icon_info_load_icon(icon_info, &err);
	      if (pixbuf)
		list = g_list_append(list, pixbuf);
	      else
		{
		  g_warning(_("unable to load theme icon '%s' in size %ix%i: %s"),
			    icon_name, sizes[i], sizes[i], err->message);
		  g_error_free(err);
		}
	    }
	}
      else
	g_warning(_("icon '%s' in size %ix%i not found in theme"),
		  icon_name, sizes[i], sizes[i]);
    }

  return list;
}

GType
gt_toolbar_style_get_type (void)
{
  static GType type = 0;

  if (type == 0)
    {
      static const GEnumValue values[] = {
	{ GT_TOOLBAR_STYLE_DESKTOP_DEFAULT, "GT_TOOLBAR_STYLE_DESKTOP_DEFAULT", "desktop-default" },
	{ GT_TOOLBAR_STYLE_ICONS, "GT_TOOLBAR_STYLE_ICONS", "icons" },
	{ GT_TOOLBAR_STYLE_TEXT, "GT_TOOLBAR_STYLE_TEXT", "text" },
	{ GT_TOOLBAR_STYLE_BOTH, "GT_TOOLBAR_STYLE_BOTH", "both" },
	{ GT_TOOLBAR_STYLE_BOTH_HORIZ, "GT_TOOLBAR_STYLE_BOTH_HORIZ", "both-horiz" },
	{ 0, NULL, NULL }
      };

      type = g_enum_register_static("GTToolbarStyle", values);
    }

  return type;
}

void
gt_source_remove (unsigned int *tag)
{
  g_return_if_fail(tag != NULL);

  if (*tag)
    {
      g_source_remove(*tag);
      *tag = 0;
    }
}

void
gt_display_help (GtkWindow *parent, const char *link_id)
{
  GError *err = NULL;

  if (! gnome_help_display("gnome-translate.xml", link_id, &err))
    {
      gt_error_dialog(parent, _("Unable to display help"), "%s", err->message);
      g_error_free(err);
    }
}
