/* 
 * Copyright (C) 2004, 2005 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <string.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <translate.h>
#include <aspell.h>
#include "gt-language-detection.h"
#include "gt-app.h"
#include "gt-util.h"
#include "gt-conf.h"
#include "gt-shell.h"

typedef struct
{
  int			ref_count;
  char			*tag;
  AspellSpeller		*speller;
  int			score;
} Speller;

typedef enum
{
  REQUEST_DETECT_LANGUAGE,
  REQUEST_DISABLE
} RequestType;

typedef struct
{
  RequestType		type;
  GSList		*spellers;
  GSList		*words;
} Request;

G_LOCK_DEFINE_STATIC(self);

static GSList *spellers = NULL;
static GAsyncQueue *detect_queue = NULL;

static GTLanguageDetectionResultFunc result_func = NULL;
static gpointer result_user_data;

static void gt_language_detection_notify_pairs_h (GObject *object,
						  GParamSpec *pspec,
						  gpointer user_data);

static void gt_language_detection_create_spellers (void);
static gpointer gt_language_detection_create_spellers_thread (gpointer data);

static void gt_language_detection_autodetect_language_notify_cb (GConfClient *client,
								 unsigned int cnxn_id,
								 GConfEntry *entry,
								 gpointer user_data);
static void gt_language_detection_enable (void);
static void gt_language_detection_disable (void);

static Speller *gt_language_detection_speller_new (const char *tag, GError **err);
static Speller *gt_language_detection_speller_ref (Speller *speller);
static void gt_language_detection_speller_unref (Speller *speller);

static Request *gt_language_detection_request_new (RequestType type);
static void gt_language_detection_request_free (Request *request);

static gpointer gt_language_detection_detect_thread (gpointer data);
static gboolean gt_language_detection_use_word (const char *word);
static int gt_language_detection_speller_compare_tag (const Speller *a, const char *tag);
static int gt_language_detection_speller_compare_score (const Speller *a, const Speller *b);

/*** implementation **********************************************************/

void
gt_language_detection_init (GTLanguageDetectionResultFunc func, gpointer user_data)
{
  g_return_if_fail(result_func == NULL);
  g_return_if_fail(func != NULL);

  result_func = func;
  result_user_data = user_data;

  if (eel_gconf_get_boolean(GT_CONF_AUTODETECT_LANGUAGE))
    gt_language_detection_enable();

  eel_gconf_notification_add(GT_CONF_AUTODETECT_LANGUAGE, gt_language_detection_autodetect_language_notify_cb, NULL);
  g_signal_connect(gt_shell_get_translate_session(gt_shell), "notify::pairs", G_CALLBACK(gt_language_detection_notify_pairs_h), NULL);
}

static void
gt_language_detection_notify_pairs_h (GObject *object,
				      GParamSpec *pspec,
				      gpointer user_data)
{
  if (eel_gconf_get_boolean(GT_CONF_AUTODETECT_LANGUAGE))
    gt_language_detection_create_spellers();
}

static void
gt_language_detection_create_spellers (void)
{
  TranslateSession *session;

  session = gt_shell_get_translate_session(gt_shell);
  gt_thread_create(gt_app_window, gt_language_detection_create_spellers_thread, g_object_ref(session));
}

static gpointer
gt_language_detection_create_spellers_thread (gpointer data)
{
  TranslateSession *session = data;
  AspellConfig *config;
  AspellDictInfoList *dict_list;
  AspellDictInfoEnumeration *dict_enum;
  GSList *pairs;
  const AspellDictInfo *dict_info;
  GSList *new_spellers = NULL;

  config = new_aspell_config();
  dict_list = get_aspell_dict_info_list(config);
  delete_aspell_config(config);

  dict_enum = aspell_dict_info_list_elements(dict_list);
  pairs = translate_session_get_pairs(session);

  while ((dict_info = aspell_dict_info_enumeration_next(dict_enum)))
    {
      TranslatePair *pair = translate_pairs_find(pairs, dict_info->code, NULL);

      if (pair
	  && translate_pair_get_flags(pair) & TRANSLATE_PAIR_TEXT
	  && ! g_slist_find_custom(new_spellers,
				   dict_info->code,
				   (GCompareFunc) gt_language_detection_speller_compare_tag))
	{
	  Speller *speller;
	  GError *err = NULL;
	  
	  speller = gt_language_detection_speller_new(dict_info->code, &err);
	  if (speller)
	    new_spellers = g_slist_append(new_spellers, speller);
	  else
	    {
	      char *primary;
	      
	      primary = g_strdup_printf(_("Unable to use %s dictionary"), translate_get_language_name(dict_info->code));
	      
	      GDK_THREADS_ENTER();
	      gt_error_dialog(gt_app_window, primary, "%s", err->message);
	      gdk_flush();
	      GDK_THREADS_LEAVE();
	      
	      g_free(primary);
	      g_error_free(err);
	    }
	}
    }

  delete_aspell_dict_info_enumeration(dict_enum);
  gt_g_object_slist_free(pairs);

  G_LOCK(self);

  eel_g_slist_free_deep_custom(spellers, (GFunc) gt_language_detection_speller_unref, NULL);
  spellers = new_spellers;

  if (g_slist_length(spellers) >= 2)
    gt_language_detection_enable();
  else
    {
      GDK_THREADS_ENTER();
      gt_error_dialog(gt_app_window,
		      _("Language detection not available"),
		      _("Less than two languages or Aspell dictionaries are available."));
      gdk_flush();
      GDK_THREADS_LEAVE();

      gt_language_detection_disable();
    }

  G_UNLOCK(self);

  g_object_unref(session);
  return NULL;
}

static void
gt_language_detection_autodetect_language_notify_cb (GConfClient *client,
						     unsigned int cnxn_id,
						     GConfEntry *entry,
						     gpointer user_data)
{
  const GConfValue *value = gconf_entry_get_value(entry);

  G_LOCK(self);

  if (gconf_value_get_bool(value))
    gt_language_detection_enable();
  else
    gt_language_detection_disable();

  G_UNLOCK(self);
}

static void
gt_language_detection_enable (void)
{
  if (! detect_queue)
    {
      detect_queue = g_async_queue_new();
      gt_language_detection_create_spellers();
      gt_thread_create(gt_app_window, gt_language_detection_detect_thread, detect_queue);
    }
}

static void
gt_language_detection_disable (void)
{
  if (detect_queue)
    {
      g_async_queue_push(detect_queue, gt_language_detection_request_new(REQUEST_DISABLE));
      detect_queue = NULL;	/* unreffed by thread */
      eel_g_slist_free_deep_custom(spellers, (GFunc) gt_language_detection_speller_unref, NULL);
      spellers = NULL;
    }
}

static Speller *
gt_language_detection_speller_new (const char *tag, GError **err)
{
  AspellConfig *config;
  Speller *speller = NULL;

  config = new_aspell_config();
  
  if (aspell_config_replace(config, "encoding", "utf-8")
      && aspell_config_replace(config, "master", tag))
    {
      AspellCanHaveError *possible_err;

      possible_err = new_aspell_speller(config);
      if (aspell_error_number(possible_err) == 0)
	{
	  speller = g_new0(Speller, 1);
	  speller->ref_count = 1;
	  speller->tag = g_strdup(tag);
	  speller->speller = to_aspell_speller(possible_err);
	}
      else
	g_set_error(err, 0, 0, "%s", aspell_error_message(possible_err));
    }
  else
    g_set_error(err, 0, 0, "%s", aspell_config_error_message(config));
  
  delete_aspell_config(config);
  
  return speller;
}

static Speller *
gt_language_detection_speller_ref (Speller *speller)
{
  g_return_val_if_fail(speller != NULL, NULL);

  g_atomic_int_inc(&speller->ref_count);

  return speller;
}

static void
gt_language_detection_speller_unref (Speller *speller)
{
  g_return_if_fail(speller != NULL);

  if (g_atomic_int_dec_and_test(&speller->ref_count))
    {
      g_free(speller->tag);
      delete_aspell_speller(speller->speller);
      g_free(speller);
    }
}

static Request *
gt_language_detection_request_new (RequestType type)
{
  Request *request;

  request = g_new0(Request, 1);
  request->type = type;

  return request;
}

static void
gt_language_detection_request_free (Request *request)
{
  g_return_if_fail(request != NULL);

  eel_g_slist_free_deep_custom(request->spellers, (GFunc) gt_language_detection_speller_unref, NULL);
  eel_g_slist_free_deep(request->words);
  g_free(request);
}

void
gt_language_detection_detect_language (const char *text)
{
  g_return_if_fail(text != NULL);

  G_LOCK(self);

  if (detect_queue)
    {
      char **words;
      GSList *used_words = NULL;
      int i;
      
      words = g_strsplit(text, " ", 0);

      for (i = 0; words[i]; i++)
	{
	  g_strstrip(words[i]);
	  if (gt_language_detection_use_word(words[i]))
	    used_words = g_slist_append(used_words, words[i]);
	  else
	    g_free(words[i]);
	}

      g_free(words);

      if (used_words)
	{
	  Request *request;

	  request = gt_language_detection_request_new(REQUEST_DETECT_LANGUAGE);
	  request->spellers = gt_g_slist_copy_deep(spellers, (GTCopyFunc) gt_language_detection_speller_ref, NULL);
	  request->words = used_words;
	  
	  g_async_queue_push(detect_queue, request);
	}
    }

  G_UNLOCK(self);
}

static gpointer
gt_language_detection_detect_thread (gpointer data)
{
  GAsyncQueue *queue = data;
  Request *request;
  gboolean disable = FALSE;

  do
    {
      request = g_async_queue_pop(queue);

    start:
      if (request->type == REQUEST_DETECT_LANGUAGE)
	{
	  GSList *l;
	  Speller *speller1;
	  Speller *speller2;
	  
	  GT_LIST_FOREACH(l, request->spellers)
	    {
	      Speller *speller = l->data;
	      speller->score = 0;
	    }
	  
	  GT_LIST_FOREACH(l, request->words)
	    {
	      const char *word = l->data;
	      GSList *m;

	      GT_LIST_FOREACH(m, request->spellers)
	        {
		  Speller *speller = m->data;
		  Request *new_request;
	      
		  if ((new_request = g_async_queue_try_pop(detect_queue)))
		    {
		      gt_language_detection_request_free(request);
		      request = new_request;
		      goto start;
		    }

		  if (aspell_speller_check(speller->speller, word, -1) == 1)
		    speller->score++;
		}
      
	      request->spellers = g_slist_reverse(g_slist_sort(request->spellers, (GCompareFunc) gt_language_detection_speller_compare_score));
	
	      speller1 = g_slist_nth_data(request->spellers, 0);
	      speller2 = g_slist_nth_data(request->spellers, 1);
	      
	      /*
	       * If the first language leads by 10 points or more, we
	       * assume it is going to win, and we stop here for
	       * performance.
	       */
	      if (speller1->score - speller2->score >= 10)
		break;
	    }
      
	  /* we only give an opinion if there is no tie */
	  if (speller1->score != speller2->score)
	    result_func(speller1->tag, result_user_data);
	}

      disable = request->type == REQUEST_DISABLE;
      gt_language_detection_request_free(request);
    }
  while (! disable);
  
  g_async_queue_unref(queue);

  return NULL;
}

static gboolean
gt_language_detection_use_word (const char *word)
{
  g_return_val_if_fail(word != NULL, FALSE);
  
  for (; *word; word = g_utf8_next_char(word))
    if (g_unichar_isalpha(g_utf8_get_char(word)))
      return TRUE;
  
  return FALSE;
}

static int
gt_language_detection_speller_compare_tag (const Speller *a, const char *tag)
{
  return g_ascii_strcasecmp(a->tag, tag);
}

static int
gt_language_detection_speller_compare_score (const Speller *a, const Speller *b)
{
  return a->score - b->score;
}
