from Config import Config
import configvalues as conf
import cmdlineparser
import values
import desktop

from TimeWatcher import TimeWatcher

from GlassWindow import GlassWindow
from CalWindow import CalWindow
from Cal import Cal
from Navigation import Navigation
from SkinBrowser import SkinBrowser
from Menu import Menu

from planner.Planner import Planner

import gtk
import os
import sys



class CalMediator:

    #
    # Constructor.
    #
    def __init__(self):

        # initialize and load config
        userhomedir = os.environ.get(values.DESKCALUSERHOME)
        homedir = os.environ.get(values.DESKCALHOME)
        configfile = os.path.join(userhomedir, values.CONFIGFILE)
        formatfile = os.path.join(homedir, values.CONFIGFORMAT)
        self.__config = Config(configfile, formatfile)
        self.__config.load()

        # read command line arguments
        try:
            cmdlineparser.parse(sys.argv[1:], self.__config)
        except StandardError:
            cmdlineparser.usage()


        # time watcher for reporting regular events
        self.__timewatcher = TimeWatcher()

        # background watcher for reporting changes to the background image
        #self.__bgwatcher = BGWatcher()


        # components
        self.__window = CalWindow(self.__config)

        self.__cal = Cal()
        self.__window.connect("button-press-event", self.__cal.on_button_press)
        self.__window.connect("motion-notify-event", self.__cal.on_day_enter)
        self.__window.connect("leave-notify-event", self.__cal.on_day_leave)
        self.__cal.show()
        self.__window.put(self.__cal, 0, 0)

        self.__navigation = Navigation()
        self.__navigation.show()
        self.__window.put(self.__navigation, 0, 0)

        self.__skin_browser = SkinBrowser()


        # day planner
        self.__planner = Planner()
        self.__cal.set_planner(self.__planner)


        self.__popup_menu = Menu([
            [_("_Current Month"), self.__cmd_navigate, [0]],
            [_("_Previous Month"), self.__cmd_navigate, [-1]],
            [_("_Next Month"), self.__cmd_navigate, [1]],
            [],
            [_("_Open Evolution Calendar"), self.__cmd_evolution, []],
            [],
            [_("_Move"), self.__window.move_window, []],
            [],
            [_("Change _Skin..."), self.__cmd_show_skin_browser, []],
            [],
            [_("_Quit"), self.__cmd_quit, []]])

        # register observers
        self.__timewatcher.add_observer(self.__on_timewatcher)
        #self.__bgwatcher.add_observer(self.__on_bg_change)
        desktop.watch_bg(self.__on_bg_change)
        self.__window.add_observer(self.__on_window)
        self.__cal.add_observer(self.__on_cal)
        self.__navigation.add_observer(self.__on_navigation)
        self.__skin_browser.add_observer(self.__on_skin_browser)
        self.__planner.add_observer(self.__on_planner)

        # get screen resolution if the user did not tell
        dpi = self.__config.get([conf.TEMPORARY, conf.DPI])
        if (not dpi):
            dpi = str(self.__guess_dpi())
            self.__config.set([conf.TEMPORARY, conf.DPI], dpi)


        # configure
        self.set_config(self.__config)




    #
    # Tries to guess the screen resolution in dots per inch.
    #
    def __guess_dpi(self):

        height = gtk.gdk.screen_height()
        height_inch = gtk.gdk.screen_height_mm() / 25.4  # 1 inch is 25.4 mm
        dpi = height / height_inch

        return dpi



    #
    # Reacts on time watcher events.
    #
    def __on_timewatcher(self, src, cmd, *args):

        if (cmd == src.OBS_NEWDAY):
            self.__cal.set_month()
            self.__cmd_set_month_bg()

        elif (cmd == src.OBS_SAVE_CONFIG):
            self.__config.save()



    #
    # Reacts on changes to the background image.
    #
    def __on_bg_change(self, src, cmd):

        self.__window.update_bg()

    


    #
    # Reacts on window events.
    #
    def __on_window(self, src, cmd, *args):

        if (cmd == src.OBS_WHEEL):
            step = args[0]
            self.__cmd_navigate(step)

        elif (cmd == src.OBS_CLICK):
            x, y = args
            navx, navy, navw, navh = self.__navigation.get_allocation()
            if (navx <= x <= navx + navw and navy <= y <= navy + navh):
                self.__navigation.send_click(x - navx, y - navy)
            
        elif (cmd == src.OBS_CLOSE):
            self.__cmd_quit()

        elif (cmd == src.OBS_MOVE):
            x, y = args[0]
            self.__config.set([conf.GEOMETRY, conf.WINDOW, conf.X], str(x))
            self.__config.set([conf.GEOMETRY, conf.WINDOW, conf.Y], str(y))

        elif (cmd == src.OBS_MENU):
            button = args[0]
            etime = args[1]
            self.__popup_menu.popup(button, etime)



    #
    # Reacts on calendar events.
    #
    def __on_cal(self, src, cmd, *args):

        if (cmd == src.OBS_ENTER_DAY):
            year, month, day = args
            self.__planner.popup(year, month, day, self.__window)

        elif (cmd == src.OBS_LEAVE_DAY):
            self.__planner.close_popup()

        elif (cmd == src.OBS_OPEN_DAY):
            year, month, day = args
            self.__planner.edit(year, month, day)
            # TODO: implement



    #
    # Reacts on navigation events.
    #
    def __on_navigation(self, src, step):

        self.__cmd_navigate(step)



    #
    # Reacts on skin browser events.
    #
    def __on_skin_browser(self, src, cmd, skin):

        self.__config.set_offset([conf.SKIN])
        self.__config.clear()
        self.__config.load(os.path.join(values.SKINDIR, skin, values.SKINFILE),
                           [os.environ.get(values.SKINPATH, "")])
        self.__config.set_offset([])
        self.set_config(self.__config)

        month, year = self.__cal.get_month()
        gtk.timeout_add(250, self.__cal.set_month, month, year)
        self.__cmd_set_month_bg()



    #
    # Reacts on planner events.
    #
    def __on_planner(self, src, cmd, *args):

        if (cmd == src.OBS_CHANGE_CALENDAR):
            month, year = self.__cal.get_month()
            self.__cal.set_month(month, year)



    #
    # Configures the application.
    #
    def set_config(self, config):

        self.__window.set_config(config)
        self.__cal.set_config(config)
        self.__navigation.set_config(config)
        self.__planner.set_config(config)

        navx = config.get([conf.SKIN, conf.GEOMETRY, conf.NAVIGATION, conf.X])
        navy = config.get([conf.SKIN, conf.GEOMETRY, conf.NAVIGATION, conf.Y])

        self.__window.put(self.__navigation, navx, navy)



    #
    # Sets the background image for the current month.
    #
    def __cmd_set_month_bg(self):

        month, year = self.__cal.get_month()

        path = self.__config.get([conf.SKIN, conf.PATH])
        images = self.__config.get([conf.SKIN, conf.GRAPHICS,
                                    conf.IMAGES]).split(";")

        imagefile = images[month - 1].strip()
        if (imagefile):
            filename = os.path.join(path, imagefile)
            alpha = 255
        else:
            filename = ""
            alpha = 0
            
        gtk.idle_add(self.__window.set_bg_image, filename, alpha)





    #
    # Quits the application.
    #
    def __cmd_quit(self):

        self.__config.save()
        gtk.mainquit()



    #
    # Navigates through the months.
    #
    def __cmd_navigate(self, step):

        if (step == 0):
            self.__cal.set_month()
            month, year = self.__cal.get_month()

        else:
            month, year = self.__cal.get_month()
            month += step
            
            if (month == 0):
                month = 12
                year -= 1
            elif (month == 13):
                month = 1
                year += 1

            self.__cal.set_month(month, year)

        self.__cmd_set_month_bg()



    #
    # Displays the skin browser.
    #
    def __cmd_show_skin_browser(self):

        skin = os.path.basename(self.__config.get([conf.SKIN, conf.PATH]))
        self.__skin_browser.show()
        self.__skin_browser.set_current_skin(skin)



    #
    # Opens the Evolution groupware application.
    #
    def __cmd_evolution(self):

        os.system("evolution evolution:/local/Calendar &")
