import java.io.File;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.TreeEvent;
import org.eclipse.swt.events.TreeListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.program.Program;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

public class Main {
	public static void main(String[] args) {
		Display display = new Display();
		Shell shell = new Shell(display);
		shell.setLayout(new FillLayout());

		// Create a Tree with some data
		Tree tree = new Tree(shell, SWT.V_SCROLL | SWT.H_SCROLL);
		File[] roots = File.listRoots();
		for (int i = 0; i < roots.length; ++i) {
			TreeItem newItem = new TreeItem(tree, SWT.NONE);
			newItem.setText(roots[i].getAbsolutePath());
			newItem.setData(roots[i]);
			if (roots[i].isDirectory()) new TreeItem(newItem, SWT.NONE);
		}
		tree.addTreeListener(new TreeListener() {
			public void treeCollapsed(TreeEvent e) {
				TreeItem item = (TreeItem) e.item;
				TreeItem[] items = item.getItems();
				for (int i = 0; i < items.length; ++i) items[i].dispose();
				new TreeItem(item, SWT.NONE);
			}
			public void treeExpanded(TreeEvent e) {
				TreeItem item = (TreeItem) e.item;
				TreeItem[] items = item.getItems();
				for (int i = 0; i < items.length; ++i) items[i].dispose();
				File dir = (File) item.getData();
				File[] files = dir.listFiles();
				for (int i = 0; i < files.length; ++i) {
					TreeItem newItem = new TreeItem(item, SWT.NONE);
					newItem.setText(files[i].getName());
					newItem.setData(files[i]);
					if (files[i].isDirectory()) new TreeItem(newItem, SWT.NONE);
				}
			}
		});
		
		// Add listeners to determine the set of currently visible tree items
		// whenever changes occur to the tree widget.
		// Pessimistic -- does not cache any tree structure information.
		Listener treeItemsExposedListener = new Listener() {
			TreeItem[] oldItems;
			public void handleEvent(Event e) {
				System.out.println("Check");
				// Compare using object identity (fixme?)
				Tree tree = (Tree) e.widget;
				TreeItem[] newItems = getVisibleTreeItems(tree);
				if (oldItems != null && oldItems.length == newItems.length) {
					int i;
					for (i = 0; i < oldItems.length; ++i) {
						if (oldItems[i] != newItems[i]) break;
					}
					if (i == oldItems.length) return;
				}
				oldItems = newItems;
				treeItemsExposed(tree, newItems);
			}
		};
		tree.addListener(SWT.Paint, treeItemsExposedListener);
//		tree.addListener(SWT.Resize, treeItemsExposedListener);
//		tree.addListener(SWT.KeyDown, treeItemsExposedListener);
//		tree.addListener(SWT.KeyUp, treeItemsExposedListener);
//		tree.addListener(SWT.Expand, treeItemsExposedListener);
//		tree.addListener(SWT.Collapse, treeItemsExposedListener);
		
		// Event loop
		shell.setSize(300, 300);
		shell.open();
		while (! shell.isDisposed()) {
			if (! display.readAndDispatch()) display.sleep();
		}
		display.dispose();
	}

	/**
	 * Called when a new set of tree items has been made visible
	 */
	static void treeItemsExposed(Tree tree, TreeItem[] items) {
		/* do some work */
		for (int i = 0; i < items.length; ++i) {
			if (items[i].getText().charAt(0) != '>') {
				items[i].setText(">" + items[i].getText());
				File file = (File) items[i].getData();
				String fileName = file.getName();
				String fileExt = fileName.substring(fileName.lastIndexOf('.') + 1);
				Program program = Program.findProgram(fileExt);
				if (program != null) {
					ImageData imageData = program.getImageData();
					if (imageData != null) {
						Image image = new Image(tree.getDisplay(), imageData);
						items[i].setImage(image); // LEAK!
					}
				}
			}
		}
		/* test */
		System.out.println("count: " + items.length);
		for (int i = 0; i < items.length; ++i) {
			System.out.println("  " + items[i].getText());
		}
	}
	
	/**
	 * Returns an array of all visible TreeItems in a Tree widget
	 * based on the current vertical scroll position of the Tree widget.
	 * 
	 * @param tree the Tree widget
	 * @return the array of visible TreeItems
	 */
	static TreeItem[] getVisibleTreeItems(Tree tree) {
		Rectangle bounds = tree.getClientArea();
		TreeItem firstItem = getFirstVisibleTreeItem(tree);
		if (firstItem == null) return new TreeItem[0];

		// compute the maximum number of items that could be visible at this time
		// note: we assume that all items are exactly itemHeight pixels tall
		int itemHeight = tree.getItemHeight();
		int itemYOffset = firstItem.getBounds().y - bounds.y;
		int maxItems = (bounds.height - itemYOffset + itemHeight - 1) / itemHeight;
		if (itemYOffset != 0) ++maxItems;

		// iterate over all of the visible items
		TreeItem[] items = new TreeItem[maxItems];
		items[0] = firstItem;
		for (int i = 1; i < maxItems; ++i) {
			items[i] = getNextExpandedTreeItem(items[i - 1]);
			if (items[i] == null) {
				// compress the array
				TreeItem[] visibleItems = new TreeItem[i];
				System.arraycopy(items, 0, visibleItems, 0, i);
				return visibleItems;
			}
		}
		return items;
	}
	
	/**
	 * Determines the first TreeItem visible at the top of a Tree widget
	 * based on the current vertical scroll position of the Tree widget.
	 * 
	 * @param tree the Tree widget
	 * @return the first visible TreeItem, or null if no such TreeItem exists
	 */
	static TreeItem getFirstVisibleTreeItem(Tree tree) {
		// Binary search for first visible item
		int kUpper = tree.getClientArea().y;
		int kLower = kUpper - tree.getItemHeight() + 1;
		TreeItem[] items = tree.getItems();
		for (;;) {
			int low = 0, high = items.length;
			// search for item with y pos between kUpper and kLower
			while (low < high) {
				int mid = (high + low) / 2;
				int y = items[mid].getBounds().y;
				if (y < kLower) low = mid + 1;
				else if (y > kUpper) high = mid;
				else return items[mid]; // found it!
			}
			if (low == 0) return null;
			items = items[low - 1].getItems(); // low converged to parent + 1
		}
		
		/* Works, but could fail if the top-most item isn't 
		Rectangle bounds = tree.getClientArea();
		Point point = new Point(0, bounds.y);
		for (point.x = bounds.x; point.x < bounds.x + bounds.width; ++point.x) {
			TreeItem item = tree.getItem(point);
			if (item != null) return item;
		}
		return null;
		*/
	}
	
	/**
	 * Determines the TreeItem that would be displayed beneath the specified
	 * TreeItem if both were visible on-screen.
	 * 
	 * @param item the TreeItem to consider
	 * @return the next displayed TreeItem, or null if no such TreeItem exists
	 */
	static TreeItem getNextExpandedTreeItem(TreeItem item) {
		// examine children
		if (item.getExpanded()) {
			TreeItem[] items = item.getItems();
			if (items.length != 0) return items[0];
		}
		do {
			// examine siblings
			TreeItem parent = item.getParentItem();
			TreeItem[] peers = (parent != null) ? parent.getItems() : item.getParent().getItems();
			for (int i = 0; i < peers.length - 1; i++) {
				if (peers[i] == item) return peers[i + 1];
			}
			// examine aunts and uncles
			item = parent;
		} while (item != null);
		return null;
	}
}
