package client

import (
	"io"
	"net/http"
	"net/url"

	"github.com/docker/engine-api/types"
)

// ImagePull request the docker host to pull an image from a remote registry.
// It executes the privileged function if the operation is unauthorized
// and it tries one more time.
// It's up to the caller to handle the io.ReadCloser and close it properly.
func (cli *Client) ImagePull(options types.ImagePullOptions, privilegeFunc RequestPrivilegeFunc) (io.ReadCloser, error) {
	query := url.Values{}
	query.Set("fromImage", options.ImageID)
	if options.Tag != "" {
		query.Set("tag", options.Tag)
	}

	resp, err := cli.tryImageCreate(query, options.RegistryAuth)
	if resp.statusCode == http.StatusUnauthorized {
		newAuthHeader, privilegeErr := privilegeFunc()
		if privilegeErr != nil {
			return nil, privilegeErr
		}
		resp, err = cli.tryImageCreate(query, newAuthHeader)
	}
	if err != nil {
		return nil, err
	}
	return resp.body, nil
}
