## An Idle extension, converted for Boa

# Extension to format a paragraph

# Does basic, standard text formatting, and also understands Python
# comment blocks.  Thus, for editing Python source code, this
# extension is really only suitable for reformatting these comment
# blocks or triple-quoted strings.

# Known problems with comment reformatting:
# * If there is a selection marked, and the first line of the
#   selection is not complete, the block will probably not be detected
#   as comments, and will have the normal "text formatting" rules
#   applied.
# * If a comment block has leading whitespace that mixes tabs and
#   spaces, they will not be considered part of the same block.
# * Fancy comments, like this bulleted list, arent handled :-)

import re

from Utils import _

class FormatParagraph:
    def __init__(self, stc):
        self.stc = stc

    def close(self):
        self.stc = None

    def format_paragraph(self):
        text = self.stc
        first, last = text.GetSelection()
        if first and last:
            data = self.stc.GetText()[first:last]
            comment_header = ''
        else:
            p = text.GetCurrentPos()
            col = text.GetColumn(p)
            ln = text.LineFromPosition(p)
            first, last, comment_header, data = find_paragraph(text, ln, col)

        if comment_header:
            # Reformat the comment lines - convert to text sans header.
            lines = data.split(text.eol)
            lines = map(lambda st, l=len(comment_header): st[l:], lines)
            data = text.eol.join(lines)
            # Reformat to 70 chars or a 20 char width, whichever is greater.
            format_width = max(70-len(comment_header), 20)
            newdata = reformat_paragraph(data, text.eol, format_width)
            # re-split and re-insert the comment header.
            newdata = newdata.split(text.eol)
            # If the block ends in a \n, we dont want the comment
            # prefix inserted after it. (Im not sure it makes sense to
            # reformat a comment block that isnt made of complete
            # lines, but whatever!)  Can't think of a clean soltution,
            # so we hack away
            block_suffix = ""
            if not newdata[-1]:
                block_suffix = text.eol
                newdata = newdata[:-1]
            builder = lambda item, prefix=comment_header: prefix+item
            newdata = text.eol.join(map(builder, newdata)) + block_suffix
        else:
            # Just a normal text format
            newdata = reformat_paragraph(data, text.eol)
        #text.tag_remove("sel", "1.0", "end")
        if newdata != data:
            text.BeginUndoAction()
            text.ReplaceSelection(newdata)
            text.EndUndoAction()
            text.SetCurrentPos(first)
        else:
            text.SetCurrentPos(last)
        #text.see("insert")

def find_paragraph(text, lineno, col):
    line = text.GetLine(lineno)
    while lineno < text.GetLineCount() and is_all_white(line):
        lineno = lineno + 1
        line = text.GetLine(lineno)
    first_lineno = lineno
    comment_header = get_comment_header(line)
    comment_header_len = len(comment_header)
    while get_comment_header(line)==comment_header and \
              not is_all_white(line[comment_header_len:]):
        lineno = lineno + 1
        line = text.GetLine(lineno)
    last = lineno
    # Search back to beginning of paragraph
    lineno = first_lineno - 1
    line = text.GetLine(lineno)
    while lineno > 0 and \
              get_comment_header(line)==comment_header and \
              not is_all_white(line[comment_header_len:]):
        lineno = lineno - 1
        line = text.GetLine(lineno)
    first = lineno+1
    return first, last, comment_header, text.GetText()[first:last]

def reformat_paragraph(data, eol, limit=70):
    lines = data.split(eol)
    i = 0
    n = len(lines)
    while i < n and is_all_white(lines[i]):
        i = i+1
    if i >= n:
        return data
    indent1 = get_indent(lines[i])
    if i+1 < n and not is_all_white(lines[i+1]):
        indent2 = get_indent(lines[i+1])
    else:
        indent2 = indent1
    new = lines[:i]
    partial = indent1
    while i < n and not is_all_white(lines[i]):
        # XXX Should take double space after period (etc.) into account
        words = re.split("(\s+)", lines[i])
        for j in range(0, len(words), 2):
            word = words[j]
            if not word:
                continue # Can happen when line ends in whitespace
            if len((partial + word).expandtabs()) > limit and \
               partial != indent1:
                new.append(partial.rstrip())
                partial = indent2
            partial = partial + word + " "
            if j+1 < len(words) and words[j+1] != " ":
                partial = partial + " "
        i = i+1
    new.append(partial.rstrip())
    # XXX Should reformat remaining paragraphs as well
    new.extend(lines[i:])
    return eol.join(new)

def is_all_white(line):
    return re.match(r"^\s*$", line) is not None

def get_indent(line):
    return re.match(r"^(\s*)", line).group()

def get_comment_header(line):
    m = re.match(r"^(\s*#*)", line)
    if m is None: return ""
    return m.group(1)

#-------------------------------------------------------------------------------

class FormatParagraphViewPlugin:
    def __init__(self, model, view, actions):
        self.model = model
        self.view = view
        self.formatter = FormatParagraph(view)
        actions.extend( (
              ('-', None, '-', ''),
              (_('Format paragraph'), self.OnFormatParagraph, '-', '') ) )

    def OnFormatParagraph(self, event):
        self.formatter.format_paragraph()
    
from Views import PySourceView
PySourceView.PythonSourceView.plugins += (FormatParagraphViewPlugin,)

