// Note: Some of the longer tests are commented-out.
var count = (function sqrt(BigNumber) {
    var start = +new Date(),
        log,
        error,
        passed = 0,
        total = 0;

    if (typeof window === 'undefined') {
        log = console.log;
        error = console.error;
    } else {
        log = function (str) { document.body.innerHTML += str.replace('\n', '<br>') };
        error = function (str) { document.body.innerHTML += '<div style="color: red">' +
          str.replace('\n', '<br>') + '</div>' };
    }

    if (!BigNumber && typeof require === 'function') BigNumber = require('../bignumber');

    function assert(expected, actual) {
        total++;
        if (expected !== actual) {
           error('\n Test number: ' + total + ' failed');
           error(' Expected: ' + expected);
           error(' Actual:   ' + actual);
           //process.exit();
        }
        else {
            passed++;
            //log('\n Expected and actual: ' + actual);
        }
    }

    function T(root, value, dp, rm) {
        BigNumber.config(dp, rm);
        assert(root, new BigNumber(value).sqrt().toString())
    }

    function isMinusZero(n) {
        return n.toString() === '0' && n.s == -1;
    }

    log('\n Testing sqrt...');

    BigNumber.config({
        DECIMAL_PLACES : 20,
        ROUNDING_MODE : 4,
        ERRORS : true,
        RANGE : 1E9,
        EXPONENTIAL_AT : 1E9
    });

    // Test the special cases specified by ES 15.8.2.17
    T('NaN', NaN);
    T('2', 4);
    T('0.1', 0.01);

    assert(false, isMinusZero(new BigNumber(0).sqrt()));
    assert(false, isMinusZero(new BigNumber('0').sqrt()));
    assert(true, isMinusZero(new BigNumber(-0).sqrt()));
    assert(true, isMinusZero(new BigNumber('-0').sqrt()));

    T('Infinity', Infinity);
    T('NaN', -Infinity);

    BigNumber.config({ERRORS : false});

    T('NaN', undefined);
    T('NaN', null);
    T('NaN', []);

    BigNumber.config({ERRORS : true});

    T('NaN', -1);
    T('NaN', -35.999);
    T('NaN', '-0.00000000000001');

    // Test against Math.sqrt of squared integers.
    for (var i = 0; i < 1e4; i++) {
        var j = Math.floor(Math.random() * Math.pow(2, Math.floor(Math.random() * 26) + 1));
        j = Math.pow(j, 2);
        T(Math.sqrt(j).toString(), j.toString());
    }

    // Initial rounding tests.
    T('11', '101', 0, 0);             // 10.04...
    T('11', '111', 0, 0);             // 10.53...
    T('1000', 999000.25, 0, 0);       // 999.5
    T('1', 0.25, 0, 0);               // 0.5

    T('10', '101', 0, 1);
    T('10', '111', 0, 1);
    T('999', 999000.25, 0, 1);
    T('0', 0.25, 0, 1);

    T('11', '101', 0, 2);
    T('11', '111', 0, 2);
    T('1000', 999000.25, 0, 2);
    T('1', 0.25, 0, 2);

    T('10', '101', 0, 3);
    T('10', '111', 0, 3);
    T('999', 999000.25, 0, 3);
    T('0', 0.25, 0, 3);

    T('10', '101', 0, 4);
    T('11', '111', 0, 4);
    T('1000', 999000.25, 0, 4);
    T('1', 0.25, 0, 4);

    T('10', '101', 0, 5);
    T('11', '111', 0, 5);
    T('999', 999000.25, 0, 5);
    T('0', 0.25, 0, 5);

    T('10', '101', 0, 6);
    T('11', '111', 0, 6);
    T('1000', 999000.25, 0, 6);
    T('0', 0.25, 0, 6);

    T('10', '101', 0, 7);
    T('11', '111', 0, 7);
    T('1000', 999000.25, 0, 7);
    T('1', 0.25, 0, 7);

    T('10', '101', 0, 8);
    T('11', '111', 0, 8);
    T('999', 999000.25, 0, 8);
    T('0', 0.25, 0, 8);

    T('1850249442642802149156602', '3423.423e45', 0, 8);
    T('0.01', '0.0000001', 2, 0);
    T('0', '0.0000001', 2, 1);
    T('2.6', '7', 1, 4);
    T('30490054086086012186828338', '929643398172450327852909723850478224408368639961857.22035621170', 1, 4);
    T('10.6', '112.439121106', 1, 4);
    T('1007.1', '1014216.88354115504848', 1, 4);
    T('14496268662.7', '210141805141682355705.090', 1, 4);
    T('220983929785.7', '48833897223529771949253.378778868849049559624562513592321569', 1, 4);
    T('1180052939140.7', '1392524939174533146207410.12619352287', 1, 4);
    T('120468836204612.6', '14512740496493785900839846763.82328768609126547', 1, 4);
    T('1.0998820819', '1.20974059415009437172900', 10, 4);
    T('4372247436270791523.3453396636', '19116547643976509183347719022510178407.6659485696744517151189447', 10, 4);
    T('645.6559215353', '416871.569013637', 10, 4);
    T('9.3832989934', '88.0463', 10, 4);
    T('20195394139.0634492424', '407853944432118316238.060', 10, 4);
    T('203574.8431903851', '41442716779.9898655080652537763202594320363763900771333411614469322118841', 10, 4);
    T('434835511792648.9460364767', '189081922315974940615057855618.468666216438821105981678672603', 10, 4);
    T('1870631871126475697594310.8568506983', '3499263597274139582669747936294437345027253159338.2893957584', 10, 4);
    T('76418328144.477890944', '5839760876397101738682.29836547353', 10, 4);

    T('346.00153309920463414559', '119717.060907', 20, 4);
    T('263359760985014.62241224437474495709', '69358363706084030080212405554.3002793975414740876419285502', 20, 4);
    T('4.14645859760929707104', '17.193118901688058566035426', 20, 4);
    T('492974271191746.46231483998877250102', '243023632057033585989741137703.800100153395469328387', 20, 4);
    T('164.78781194238374484766', '27155.022964758430758944789724435', 20, 4);
    T('7366.78400597487165781255', '54269506.59068717789735301787194313247', 20, 4);
    T('23762326490698.34975320239967323205', '564648160250544549780100380.5514166040523', 20, 4);
    T('213634053277525452.46978666514569063831', '45639508719784583520579147409415270.488869', 20, 4);
    T('5.291502622129181181', '28', 20, 4);
    T('11.47156968462642430546', '131.59691102924', 20, 4);
    T('316358493873927157.1740897995245240711', '100082696646179606634487051294887359.5035240594546', 20, 4);
    T('500048686.48939798981543930512', '250048688859772239.7160140065116884182948058142834537', 20, 4);
    T('6022.37550162532363706717', '36269006.68257686850553015065182301017562876', 20, 4);
    T('6.8460154244205197188', '46.86792719140366873824501484885278', 20, 4);
    T('45189.48536114270306146252', '2042089587.204930656134881238462876156607', 20, 4);
    T('856160326.88774167756915629603', '733010505336524684.93087677', 20, 4);
    T('522130.17609423801622137924', '272619920788.2', 20, 4);
    T('52292561.72399606203863014237', '2734512011657938', 20, 4);

    T('26887623635.3707912464629567820776000467035000476700672842803068855485151227900078069', '722944304757350004189.2345', 73, 4);
    T('196.9765983083095309326276839454825311907994129311587833723243476358522465777', '38799.78028111312916322430363229763959902004100996', 73, 4);
    T('120016703223.6151811849913819241621570830640571538995217287681670128389432142005932891', '14404009052665322623144.3467403', 73, 4);
    T('15.1179067962466285341140019013436699388546686629569704817549228235938583883', '228.5511059', 73, 4);
    T('88.8125725884151110485104354919995585100283886785637267479492612921303631122', '7887.6730497725031780', 73, 4);
    T('19107253881509743.3779260316306131671181869224843248420235662435238492667857670998535738552', '365087150892469154433642273521263.83276841', 73, 4);
    T('400.3068500117233817006410985991134968372763336206911625632292309118077689013', '160245.5741663084', 73, 4);

    T('1.7320508075688772935274463415058723669428052538103806280558069794519330169088000370811461867572485757', 3, 100, 4);
    T('1.0488088481701515469914535136799375984752718576815039848757557635800059255011006914193852889331944178', 1.1, 100, 4);
    T('0.3162277660168379331998893544432718533719555139325216826857504852792594438639238221344248108379300295', 0.1, 100, 4);
    T('0.9486832980505137995996680633298155601158665417975650480572514558377783315917714664032744325137900886', 0.9, 100, 4);
    T('0.99999999999999999999999999999999999999994999999999999999999999999999999999999999875', '0.9999999999999999999999999999999999999999', 100, 4);
    T('11111.1110609999998870049994903925471271162312358157188236047401971668110785559489100264545730662845742184','123456789.00987654321', 100, 4);

    T('44.40502294864850084', '1971.80606307', 17, 4);
    T('14192376.643123461322847366137186892667087869800484670508360531721790645551341594437545247745705539930735', '201423554780276.3686384009255372', 97, 1);
    T('2553252336.0331', '6519097491458426345.3554782053764871036657698339691144086', 4, 4);
    T('313415140.8429479671745256541153927986977746351343070479080617690487091', '98229050509604911', 61, 5);
    T('1.537808174192351787259717', '2.3648539806128145775', 24, 0);
    T('62.30949119560079364913', '3882.4726930546528212066', 20, 4);
    T('20416269776260474006.927083982432268806569265768750039002235065141713481228428261', '416824071577046905366185925400975939799.8', 60, 1);
    T('272.7796965259930339886508185007545884578607374328807903817278380417881107290650631082406652753', '74408.76283681285609575234724616', 92, 3);
    T('2576.47265', '6638211.30041', 5, 0);
    T('163256493.009563679477426538597267934928790668772138046453329418', '26652682509781714.54341292267818515442604', 54, 6);
    T('6413898755.503103237879757203310804711689478588298288050385897319721', '41138097245844256487.4', 57, 6);
    T('651.83586891179899229526795359574620017366743', '424890', 41, 4);
    T('9.840172762711028743800750259911398446280416492345', '96.829', 48, 3);
    T('936008582670169403.24586736956499382709796268807769178807', '876112066832219350113028056940121643.4989897905396802', 38, 4);
    T('3.1691816831920187703897673787206897581382652839389771357354751032132', '10.043712541079797228859738302727066114683041689841812835450076520', 67, 1);
    T('201959414.0905242545521797507742344926679143', '40787604939787846.812484750440288725205988', 34, 2);
    T('3.235364585328831221179338900994013430389718885202447735097146763445925265374153439191445', '10.4675840', 87, 2);
    T('1391.926691686756208298398279775140528179063627807536777764086771937670002756456', '1937459.9150300380745526266059714001651697913188367053087', 75, 2);
    T('869.8354526359820716456765000567449871995222312', '756613.7146624438099121', 45, 2);
    T('15459965429.4176806583122774869069153370447277862964242', '239010531078789811116.914', 43, 2);
    T('5248173.56787', '27543325798538.753780506094441280733789361778933753426582596791104', 5, 4);
    T('6.05855328019481326674235', '36.7060678489593315124567', 23, 6);
    T('907424493602245756936.066129375330541537462629768', '823419211589292150660922033473258691103594.23268549807171826865266573694', 27, 0);
    T('11439280.9959241551974895751378888566402750717305622394543095198836564669553210222', '130857149703711.532', 73, 2);
    T('164207.20075782859923311531097410558094817321169039986325465336651402772907533144002814794168144323', '26964004780.7218252942882108', 93, 6);
    T('67866.91300857901339', '4605917881.314031310494203432328648', 14, 5);
    T('2.723397260313830449769700482256736114679660269490922765', '7.4168926374848775741132294', 54, 6);
    T('145626737347.699863159799647089039014085200730659988131836086866937945539247414906654868', '21207146630535962175511.4592', 75, 6);
    T('3198723.293675674761595756274163847124886137068077187912', '10231830709503.357046273671271695065871403474525', 48, 2);
    T('19699.655043109373052433709', '388076408.8175045546569', 23, 0);
    T('15625591.477694858909123645800863577029584072458683168542082897370551849161553', '244159109027810.20442572274363387', 69, 1);
    T('85762713.1327157666976061', '7355242963884497.437336977648000579022368327415840055129120741942567239', 16, 1);
    T('41.0148460976104358708576256643898030638770464836414', '1682.21760041067', 49, 0);
    T('5.880136052847757025272423749873232616784', '34.576', 39, 5);
    T('74542661576900.1739714442102128431140459355738187797819043083309413315', '5556608394968269531202589612.9', 55, 2);
    T('7719200214.3705019090215357285448226659983404326837819', '59586051949537602626.95016591737228440200846895047666959869445081113', 43, 5);
    T('74629.64', '5569583803.0907402', 2, 1);
    T('44.05501263636205450407476438041804283847324427155286698836103844', '1940.8441383900203', 62, 6);
    T('1119.912102354466031064620923', '1254203.117', 24, 1);
    T('92093628036.6480124503328403126750210810151658188780763586760022178450462311', '8481236324952480852423.718960437545027', 64, 1);
    T('7.55149177315316950216695542333911', '57.025028', 32, 0);
    T('1947605507655512.1474872179815308192647511525219417', '3793167213450085186132700081381', 34, 4);
    T('897186472920280057757.758955997894245720068203481844645803566957566661460082190511', '804943567191132421390195361422932115503582.03422', 60, 2);
    T('8775397.48306750827108348789044450160735393428', '77007600985827.5591133', 38, 0);
    T('3.6766253682232922013', '13.517574098263058968167681315043395710343763189956497798543', 19, 1);
    T('364.437601989709154496727744417233984148798720533510776338306467381487', '132814.76574400966188457585354574350011547665650395112601058', 66, 3);
    T('29784617918.0127174734394784458305404701929975', '887123464522004224898.5545193466489', 34, 5);
    T('374863740368.0156985152529892838083189109926642299794', '140522823842699082383715.919571886197001394994582', 40, 4);
    T('3321962129060763825931501.6171393711588977731624802456157919679857160686287774881', '11035432386913922898118885283020330437836330418535.765864', 55, 5);
    T('3039991938095044282.834999396730197828406224765246402870200714765906547758763203772', '9241550983682863551151813316288949279.9625949439074666', 63, 1);
    T('5.5727910421977962406401022785010286010813965565340383771288278905072628945', '31.056', 73, 5);
    T('1809373906.386837382140423787863676125946315451309506008234344142479881132815536036260566', '3273833933113563766.3701096193056864', 79, 2);
    T('948722.35086770259132293410301310241', '900074099035.9401838348750892', 29, 4);
    T('23603002885797536.790280728725364655699852314300322649', '557101745226966849549415425061219.969394346783798760227', 36, 1);
    T('1.3760699520736582089086174843894761768111119574016193232032195760251607', '1.893568513', 71, 0);
    T('14865553.721249212888439807434', '220984687439346.321003197181614764009809253917', 21, 0);
    T('1.4602844538376761066216387158903803538206186160126991802', '2.13243068612', 55, 0);
    T('52497683.2535666615551547765238857555777973908683364642662996875860780953333112481757317373', '2756006746991813.49967765680', 82, 0);
    T('229.108483256294986105584052287445152310628239217404673001762961439710445106676980785785353195978212', '52490.6971', 96, 2);
    T('247788926631.668855265587728589', '61399352161274570866824.398764295216033428917127', 18, 1);
    T('21424.038976941299431640698751966107735138702410669278306', '458989446.0855', 51, 3);

    /*
    T('56956.133334288074055649096270448425887257859236300516446822612657765367909481348792658918577056212964517487314659149646507878360034362047738124408378251380663712880296', '3244001124.393201124203315112854063456225', 162, 4);
    T('8.654478609367522146986967465937732834299195635917473015566911783193688697212875063970818876014535054471159717340710683348537384638848599775961897310476836944774203', '74.9', 162, 4);
    T('2.090157855342270554221573582468214310613536016537697179653818592200887519700347392246743415543531332923171565100303846336657331022146874248351314711045809920119035', '4.368759860249', 162, 4);
    T('4.898979485566356196394568149411782783931894961313340256865385134501920754914630053079718866209280469637189202453228378249717730919675514683251567902474557105657825', '24', 162, 4);
    T('5.744562646538028659850611468218929318220264457982792367699877470565900721457404627027125365596788122419557476945650837869934822615203403401618532019562307772727978', '33', 162, 4);
    T('2761776864220467592773690125.261636709953648799225108854949552005927418667927585431550152627344259326832390700476831904159140150409161169310381661652182261480548883652702710905192306201153399', '7627411447743439090016907911169126086985950772934507394.445820', 162, 4);
    T('1016542.132684851351405985352421550427715149810639767584056678930969543043928600815438993175050555365846506655226675294372370499294878158021265695180065201424027976920607', '1033357907523.46593019166446894535290441918452466583813147208529350643', 162, 4);
    T('67416893557691730594377.689393229996593018164745728497408716814685783382213046224003027235822262763354330687209958508471208061724779548394876757652983131645878622964693470107017790602867', '4545037536969136767952040548173187304914752143.582107466151', 162, 4);
    T('931291993003973764424957570.943051255559335880882830118695138743017644168272132602760327188450198488698421979593757322045111416715065737343620373909805293078217072698779281254073818550714084', '867304776233313518982078261171691244956483307888374339.8', 162, 4);
    T('7949156432.018787509478522996179666181753678431098013792417245447999791336046537264938248517945929921988256639266549619633490785427215813297039897879841661734306307344111428', '63189087980705660327.62527705239541530402411702633104087462882', 162, 4);

    T('1809865.4646006170616739546078520411523015790932463877848464678333257743018277689819262142781464857679567195196165283927863805894754890115198595353876516167856225104865664069164914054910658837497837052107088079809127635832553363804364513985303062345471350909037103500826046492068332088151173009073561164990623048692985267110502955089488933796659145847257595817786679297018780916', '3275612999954.007450386437800884', 370, 4);
    T('15.467269449647536285076994289048952792417346402263779331615370527324047804918952006389255250887444861529354673321542742996531338786891880304667201838005095317222412003012101386939275591127834757079082739161518869565123311068788189396146419765711123314785783579057845157770289130253714499563733678728735780353326475011925555877160399757945810318742227262034456726810253364', '239.236424228', 370, 4);
    T('21515312700788.8445283321238112651368817617351069674058374989844530588669193691975020463789803512804134095628531146877959931040403076115365864125573069051677734320206655238129119512589782592335983752466615626386174785231997318261202374171285170556291668296113625567950158566625308039499284336455793814655198232231567816284588561787228106085162970376394887992755493290799749825112551493', '462908680612725763398121582.1988499361251936376406501063453105637', 370, 4);
    T('74521299.538521976899190350443219558946674884841824374949238074926728684231529951608824829578859140723045085427094885285460459390853176343907992141008355659931966765663861028960423841901422569042236880047925960465288786047115089356108425087999854467423897118283417884004208867563212081515310360937085763989389534476061158207010096784634714697440008094764047817723635160212611201', '5553424084910115.80715723353', 370, 4);
    T('1195846909.8136451949628200680969918871468342637166442528552430459926616316029985160802054349155727577926247240133710269402224145037039410361654704934859568454043181682255773043960693809830780634720080576036391829106533152077104572023692346636965236517235893617548662539881565621455421628698536701895553668525060605712336259944095451585719721016583710402332219225707779232668654387', '1430049831710844464.5', 370, 4);
    T('63.363458803814609930869562561468877004673338900304322840644175341400833525250548003632643676779722860896524366210225631392441735539731411621657232048800481726194808203828100942514894307898059274946673100310375760980610689938987663753721124115742622475359805871962619958182552829213760795254063144063662267166941269648405943630007682837690708592188638914255746685886657562', '4014.9279115827111984', 370, 4);
    T('1373.2346174982060050121030325620899957248769145184187327677463246552753848965766950735510496743337066151954054529870656708088991835100431884436734596808270643667093234708089558732282842724926267168039341016233235612898755425310217129820000361621133178074632555226320411936053443770992225078255443638854008065800708779022149941454885905152491670161465306374888415427437669047', '1885773.314695444154208', 370, 4);
    T('99039087799528681.3905138833923030877513778499442375472507806052812263260160068742079767622207118750338964200879423680059646588427359069818605754805574948873861241916211290058134240391945041506590998012465268591014107092271256564136984223735037018803536800185626368799784882325385180065015284472329021977308861285931031378317690220446799508322901392419207606647824465676961912238391025575', '9808740912162750909706883310050182.630527570227310645154', 370, 4);
    T('3.7549966711037175032676936665539744167725993728876755783348100730830605055312952529891445353338440126820518274797006625702148474699248805782797465540022583593955664081518990293569512283879169129175449735381912776937984799578622377264197905942888539479665214673739023726144209253217799902885022479674447157064859576103449224248517314930440882218595889166920009969275046849', '14.1', 370, 4);
    T('206071.9099345662880392539413524097442128237359131885717624301620021107001424182134616945091475733036367971785943678559801220356724867730180036789665163630501875994159091513263234064759811698332280709521477933310954189936526370887672959806770806072945194934626710227301991226774310750612896243496085684922345254659244357871503291966086952146902972859397840015475313905910082084', '42465632064.08', 370, 4);

    T('89817010695289.72578371683807498008284714528555535180774576243832060526469276583706655412471844057215592302486816618503186850287965389458117650040638501953754812900903677140548024632204429469967898609400125991154018500677065159619195677725360191990304667964228249993084865619207470251438905759438499669605919588917712329966593460182177256512515083462907857531175748343129201000221666928295926394404306298592439213726996025312323143449177688154398992378748586802380541165612583584316700420134596203072442651698983252', '8067095410237788990654508945.4938889620226978171380538868', 500, 4);
    T('186.91379248627563760392813675540099210378050749661641999000996750119358911115159296630333873386370549699551207601328525759106230341322609045770969551608159859115697029049776748016235872091886303045181663791045868196184184442196533834203793388185938722188507323334759581974443773297252169512928459122350262968374432650606625865642232545046019710829844705222543050921091232570671006274142816262680418047451611346801167563231483515250455068357839726457106935322253398592250219377292043183684947137995080635', '34936.765821602511000', 500, 4);
    T('292.92251535175646173232431728499234372421154418946156867421786152176227096294653007831356181142938132889371346675529071648577515728469391396601893486617422900836680345530254508647264899909100464612083986261265346005164216654675922351728655350260050928208855804233241090095534108831210571537049578731309192576632259878664884788879135261573200114727909644940386409994830780435119468243089463108989041078103645175912033463261798424149302644555980788660713206241347966250504060797821189239505007200036079705', '85803.6', 500, 4);
    T('87.99026650715407396765197446961570650402634143914903620761528887814661859660279310794320854730578987027187472233291266405007186843204281271710739248189201489451091841274017441508638469723085782460443376671758758899784113134866643875281588211922668056326695185017497400934539461041472568010022163207379422457239669042993592416973790381056817762649457565899290348718808685202005765019872456068078605894916581511896996069884199563861272506806598271531781334519497271503568279500098078388331561457291680397', '7742.287', 500, 4);

    T('9.964115615547623702713334497987282609345101391172570364849116418075816800113640413848859226514077074748449697076758383273301113543709109909215564170600206464958142454266490605912308358603735813671079804922222224887150495344623713603270259563840115497313009052230649783355536482654846683965583855936794259084015703511152134740957173368048269478700424790749310707209710341986660107034655477210204159808351721182045528640956247532335115997324595227811783575481533603875179445069446441375208735891932273626153407869944580993315321965826502787138270543', '99.2836', 546, 4);
    T('1.307100287867682515567807786801412047161737137612032679611924993481917045781734946943323223902612735068918688582083522849332454081551075797942682446349695592044164001986554222994267637605078971511529704949687080403427315218684552520061137105759437079989266294002711708038550421771209354048361232627247521144464653931131639432007805143883006541670565652328321108764304216848523001514451272005371748881596518362518185934727350138270219612720116871348545672028299730589386444001708243439232746226873799802838450712779197269114313367331740420673483739', '1.7085111625437785', 546, 4);
    T('2179715624641.104045451948581834441551779587152634672576510540355644708264438654277476265980691695066448533110871133836814682961495657626427294887178053496311230025672990968595927778033477830265241856037103266547607039668793835170188641774636594856220333273217160195552886973287073504687530341285661755863078114639819058730667005812528346789548066256665974549595181496892972941783341809790771728591496793467248475138025398858570306396853160195272474728879369794382569618121601470222156680464345972678848920184128051561184284016452602903346229119622194482045757', '4751160204304558385373451.3272332038165539477606810406144617', 546, 4);
    T('39700.822577241580199678797939419163898886687420779218247258787086313113076859180773033077810860147036816212863235098843384590784069398316497506974294775336733386252013568969235999557522683195367974539653911477425701468208578424416631114715708576208486734127431256111817135113173806314284300643116495170122929525074255758154469988956319381763329406201302760736651128326823314686997919541126017908099065125010684846684750979030402294608072194301572266690808347512619218310009474588358252587183377754733236309560561175738877889452782714224060764008968002', '1576155313.3096147862201867397559036886508', 546, 4);
    T('1.25794522328842464889802954979839811250322044212082313174176990936228267186851635174161106107130430165025307524401690618283690880517619898891441753062149884262103288048373107482275796232219249109268011616848844350466917383546494632887628802000805894961164041424123794218062370303230048277719411087505573740519315205969736920463416188404939520455876144966180409180303192875736712339448801403919111160506886246944037198526719945050982972666268955585211897123758272243907184437446082156997384618356833205776554862542099528216674650656658010050062231', '1.58242618479416454763664575065542265574614542', 546, 4);
    T('1.624807680927192072091976713653208069700840817345062565530315118906583360568134538999831520589587375487595902303174139599422604662135717284255618657549630378160122248467747111071314750502291444441102163105431957570130226305138621706071808286505297736384595648509872968772996129393038122816853427310850020078925936656695761188275301963596778663712697756996198695154835116221911512411787284450370861055864406526637410758400234987941681870570312002436208851096849359131928628121324230998619957793873321252762427349655447578019631964607840051797535418', '2.64', 546, 4);
    T('18717636231873067799.216510948844550745146581264323539180043356582323609699016487098997305478787718252655554063999956808135384833559712709262538242436026132212417100611618928564814091772415407036455749502462185745609885577739497207736237382361239184683282552834573776377544551919252603573173778169991084849083487588143333148865799493939248463005741865268503183685175205052842601336209845292039862576097287476259672443140291459083371484939336391166547726508221768343629133487223197760680035244638718987232096565446767519310698558767017657627306105991008804977618723991', '350349906108727416303231044881030663104.72849777316356826833515716205', 546, 4);
    T('1406647840009650.118863996881143742849154676375102411920435563067539895576725213355050743434710662527010561053318071130411761077372296190615719469716323240116449116038768804389193494725211826193028915331814451122261159238787939478074609835120057229806915907338375241900963530017435537812109185293173117802944350037251407192941405132508530625153810048779676085394020972072535926554304846973136440349018882339256154370074386916139960439011412622403829257294020556809081837690809972797912280848596346835691547349767211626827238693605471315108662202888476421988937787', '1978658145803814237711662058049.2544850135714954117875755770241265', 546, 4);
    T('16224378173956061607697862.633217842546427767370100855088171518705366007862250561754707243455931128283904259226113595005700826232879759488413300163002142315099475508032176198042904834129778691907469882290201099517369886984799723525714860875048918708175179481025780980785512861833918274288833606546216800173441803028496920438426672142753847581458627155975653306204761742838982103620921662180680218361989958267237075686446982867517247246816010479297470319222022775438943899994549052934785516690930630661752088290413832045273976791735833819617572894548667319311757989955998909', '263230447131541828089867036367977560519472832588035', 546, 4);

    T('3.86684626019706141613723825656699191004871243725435371952851102513193897300170783147869803597806359975114434638267696014418502590333002587206926390370265920607601489345602853935200922900139573459186086120455990493891388555877391847755178584266457035830275159909609535167181434973137025675857847818998237738460415581254272837772822208258242097068355115922762128443987625212248757154527289970249951218201075262114449980122023773692545156022247353209795532545041431858554008616906673871136025390102854614065356679474908404595273532216911231835866021312311702806567650395727590121314352963554602357180371548166632447396668297987703182598', '14.9525', 632, 4);
    T('6136762.26717905744611296480849548677883172137201795887253092440892121660504456966433584211515954953762017772580407928362904974338214424385785116489885777383448138309647350406214299077359122447734595685831178247253060071791025543499094904828078657524955810034216582881357125704944563248190767043385107964306199859119157271835974466792774248278303577397200213978694185732540975537354775071603950395990745619635709671790827074205226736000604510720564275623625311465188763715973066573171788415579221561790250264794581356922531739753608438715435448872139277604185884704559289000914307078242295952209310265904281185435952775847430570132829431484', '37659851123872.6452468949180815564659044706870951617310978456174564869', 632, 4);
    T('55584781983.24490696445791805408404378469879379761888397244664120330644772739351913528775184610233179267381874891789954760338007940124374026003041155925024622975618046085564412007615044940190982877876574993585348546104467074110944013210078722838691327953750473674565370730974387207982723611177965946904757209441712191363169109505165807521321899672463583311642877639945439852157230367246653362314539530980304123767170725630541393623958472950450526635784974484049786565210736734671468997713205366127380246276456634498806056301931312061784211915142498400899530080598267700949248388709352054923014180086178893193927288901542298741856815720408810426', '3089667988124867612740.083389049922345', 632, 4);
    T('1805064995.270685679369819389640432247172036506554563452359428083087328761361845652635553731357289990951745475326934025307848780337675451754210392546185393830016044551272507825821935983758954014040138643670810605163035656676595367710188204749719310806964242566460755804623103957354726598643871684791346043973374359934258681039001336439023622338991932038561462612292035112751670372206620090017313856898980333410467539643215942489844251168601645977120694640664003642300053879055949999123327998888486028967684255707894113041249429918747443321228571191289316571869868803697397454910198466063233810801875443876639885268362424418465737511676419844', '3258259637151560514.0297800164403204384', 632, 4);
    T('125608085.77042703662921929440576851258415350669039591021196510765322333997277721479170585932128266456989640874478239906029432251376275054410584089545239517498359890440064005754383089916535639818817512288635165379914355624246231855413853842721091980292116104830063838444932363335270696391961823972078627647993680508983028621966220040659701503608235895398573773969030627362781943814154637432716276122770088096908725956005011070226435272443863093204149560949439672185351571599604932488380558361146700216684520038630554153765869107472500236645669202075005111635063065232469777930428976428832531849390576765836025904341256201817451204276318678703', '15777391210910955', 632, 4);
    T('400136410988877553453582.37101163421699743482259029089762021744122736955079765363264351177743971980656087133988131526954003823716158204437583618123747540964307177407397030906491059469918005937208664667811006296696880134895202857597704648341649853175892159884095827375237964722988879265135211319838716432321482642548617255100570921567626849693162463365611013775481030077269042675179237005903008278699962449590513074693187300493182718210865710393790428022783178966907764188174129568850369221656184631847400442208130426552208194145273689796897187359838715920990228135785300048009159741600170382195085666098719445640520314294446983533576482734809632684663212616', '160109147399059929314877918876109427110612023604.056303', 632, 4);
    T('64425.39708677991402075085097430222723408726402575346205033002867968020486619151730572708995063516358815252500577128885851092173171595206567522039205583159322846821419081246805971111195118357652973497110646963488156724407496612917479150812759668015947988612070186616711130012713409715318722886985358698149495882008505150340412263757850467789892597196854389378140820673566768146692006440247742979249196409269612046734062701334694240456008673406876828634709710253057778438021730083759189755501262144599075931168939668888175040990882526418711846468669816130425272415774060047616063598593294412963338255219828974342389552617191145782209745534', '4150631789.7892698323562331016634257014646926299287710080', 632, 4);
    T('2348735572822814115892.26408027183370740113506236192664551809826630380561100793615743475343501311869598689198309313317112398706024097995873626836363321037439405890327574217792370459096267386514893897401545965801783674703449797383205640533682179139228566108876766321414465683099263110272385159768745386268309348310852289538873334789073960542826136816561584906159230939562245879068384254598940764221550586208646296806811463864282431442235899226384157724906001227498116520670108267462634019231455786976410894467414188400388964899956986375533986475289820788610178735962555060048758466647342357027384051113897165096975205112706486602166905905775844738974189124', '5516558791043312750956805399216206701389822', 632, 4);
    T('19103802792.01564371961775338869391426446448029286247977641509368293241906066424287682005366039677603554915595198800053308245870218297432073530945933295939680395210865727953275117534560571377076828346246298768896265547871244418699953520469341265486692862736845896173671301154158095469340988225077329614333663129688855004957099781613220058661676296497307424000643112303165084920380937145312220630345213927929149505102622408340154771991875263250438238123325276964888634956357905768104652580983969312174465484750475804611861727043684792253730740484442336788006549604123329617407383758260482554791627340811587637419676122565011048819721900614226661', '364955281116224704333.02204944536', 632, 4);
    T('786.66034601980542438553763403049983609539851928329220147388661995860830168191247746764643770838112443216591486730544324590358255064211791028661132434467570268525552578636466675861637897309153324177405101531421741979722830892607461959298447068614984201744211147154166256831083543381185024040849431152799003153201087012609166537695511250190586392519199228602942589459773862505985356810151934653735972412239324596989225867188473011661629585794534658478043644553227272995034111771271753677915199703704761728445313825636390237745870611713634838750078939991244597310336345791337461711284642126303822583081171392986526905442664045593489255837', '618834.5', 632, 4);
    T('484123309.97874763869513775027582083760255026422629367680383349228461771885785050876393607959363763562763934512375018110967693444089318521616498831171036375527664923123605952647435477649669320835658085436446422080237311136711591389890017124515694523108269386574671466685756523731039177122366440515798582799809939132135784826352950211725429348489639356434994276274903713471535999323126607571342955569317639976321722080574385327905662904910228854922313780512285515691527026937933195761382827883549497100022950886879913145746098441721130133524826420689100661999466875481778061645781459258894902002955465824742017940875767981672057269065469536219', '234375379264778573', 632, 4);
    T('60481.17814179171662585426325768194943112296504919133512658120011357361699298422464207199076657876167621169341810823582081006427726259877303866169887391022736822586491147183342565113223464521871784706899966618623012338886757864618596318826063555227727089240080604262744311440712982849273211826938920661172972338303097711545177717262755702859402796454231559971416344991871436949784360654196445387214315983072589147867091579864546816048090705979976439302638898419268726958146631934559321370862789950979122360237770183583617309004258400281803258965980586023033093791875491145215922928234063783157425179949277559990637828701834500469899317547', '3657972909.4191441244525931', 632, 4);

    T('2597183854328.5679091905600850947966546165753359857320959797758353429783579407079182939565903056335948426093650795717332704791479071528177519723191124947077728435798063428371161778782597345429371741476675515164182960273091825852969638261871905188816333787724804370693852910065686936635346834748197083291357776431587087848800111892221988002406355426392615087567342248786343354283235703582188835390882988811316410094207611114574124785705884308466258369161492987882300555227716609880303616554951040594534188187971605746361153720086319724295669667359878510806390096282081082192285001657224319276565070534065340503324866753603713794833032825245475870135528342060466443304262980556284100001276572690814262354674742134248012676070023441506227778586365371105', '6745363973184995853492478.5954', 736, 4);
    T('6774655005740.164037907899356400277125449228462111588353929941351964991999943642694024106002045654190469894640190593584143456920968883709637122260941740850619226234998910249288941064461839175935885418202698405604592790954509193961851585542921730413008144762195532966076043993178536807834225605865732009222042249139153172302609878922537826024024765317187183190793601555449704914268951048131236504332674153119659391120390475083522595748414072828021281006933882134638700653092554277006524993846324403138880161051477517078245698926520722382184525492883300322699107657083780701388937788565713142369080803278939033122119335669992754410484005485791854368115176247813286850021744435931737469007811125083823541974831435625884357393099194117719985023995309226', '45895950446800262033415363.0107589586980934', 736, 4);
    T('9.8994949366116653416118210694678865499877031276386365122367581659351273492347492719527127402934910091450969236163860791744738523909551604885088504796995508198925586614925391492905996535080694744240002907669198012417919217421780341030996219304862243902668663046133901610002501558783382253959667598559278107737758243226618074843443068084667653832202520143197321088490265423627229990794806222238594919301227082997948930337052138081945322151797347614794505584303510640492417811766809910911093299829106606470195648540287464840079528339429999802421027773045400050984119089599487743576041585398129111011611345309324458600582014599091265650208792257610596486142602179717725277099578410704164010928022305362395364280948034487315508458169196178987', '98', 736, 4);
    T('87181.6234142115525219620524691647002880373681637662664358089648865241053931751056720336743475573704667754469828869574971770860976622090504059965985957160672759992061409022462794221981547073047121333832741614179766660169320255090772060280986675121181149188392777041782943372931223028492877312925787086980314820336807572760207027632594701362292053256504201751684547275915772868234710319808033440202969911746099931970130960400927941767359496945988333067403190717036520674748830967881080827391448011943102976676893692584203783154822448864894042034636583043861494867962494678297073555456340277748807852218397069734889863567431741539523977203341337841540435143868785323742066976229168363835362478774511412078654288725506862167769261569623256490877', '7600635461.13740', 736, 4);
    T('4.5825756949558400065880471937280084889844565767679719026072421239068684255477708866043615594934450326776009053975857408733118991707556019635738633623447553702817228244277981919405969049143449200907589407063891768364965598654748231477641506627774586518102302361748154008139424109257393659219939175422411846714980804218526078649141055769578125547677613025186352569361961122418978100629104930136394500794769783448686170875778719949754100133598746930349723271067563482654740784196320965532158989549437375426406190866502980945808909769566178106179115952200007681424851342306651436695554446605650313382719148849087683899267628798139136482473790088818917079648337517673741700029719068290493626817167652340083439423393091048762655243512703702846', '21', 736, 4);

    T('379533758647.33202998025355894771830486671460954826351562848093352359089756978384860789725657951059073867429383079637341943322428876224868137020890589117272041215996572802315226952651591422629979697849613357409838019360130112971147443970457701882865210579938015503729060796093840510498092974918957475137648847670773105840478476180774814337983644367224768670934474545884964455308420763849854108455051079039927823221492636701262715567129335694288707429643301669699497029554944442286919275175462196928260780423046910979090241979424076642169272255969335731465485030674055509703481930565243394712596916609989415808863418419835631677700360441697210083653827363223966242987581200002526377610727629936341627933125420316464144353438091881740224503276222975587368345923672238451009927592206329287360888148821167396976641547561007365328906795045416630208044281045956444564', '144045873952971280443387.35507896', 848, 4);
    T('29280910699687273104284697299144.07732565567508749484602637961092769263535159046426239841768965764616197975075043919815786449076328689684216110222824295072121081997345891375218925808047569533170997772123590787790622766592641361527147653442819434809931915905219637858026995741191495013690327145578287454531481294854702286205399948005290251392013711014077536227775356988930049595580907927661449806384364243569981109141658673685934994029666133875734164693622860720260710554408636555912010467046231604717399110220057952965339477795911201244340638839241074392352601833736408661548059322837128092912263901889749977388312010067566997456646543486536129645809758335017030111183330787782491931288652088515560703734327816427347518899939257817316435042124357010083771290898500690085513377138936923205163016022860153854624965235876950834393909963448127891696108571214858469890656907205330389064', '857371731403060633386241816096841559023596513404769917799492267', 848, 4);
    T('168053.04565201524955446843348960783836388810418113245166824850488690908142778357772728901345436007400982876927959300542146773837124920813399432364711787980877197182718826668919533357860217297872518296792465703667850144785269168793146280553383246166503410178151205910689122332208239404835749221852981903152552178180222439382939101404418973062745842861363866266832597644550779142636812629347441140406958185996959809516768124071903548716934343215775054973484772291998064110162394410530239517557259046875734667137531966463173455499642893757346141380481602264381153514663075350700048417551424923591583912256618460494552096382446465611749335306415219871948883932584670263698301372130930285595038191486637767642010763257747895310053515488933059515998388551181750336022799922855572438292341639496128201699781316142673719517556328928078948111316144339893593051947', '28241826152.9183215732505128601928949420068589166172608979274924914576295', 848, 4);
    T('1.53638009381417855214702667564342109510652564612190733749841594063585238492794950136881162227066019813049456848975091045178675462617786165883956111632075830748654795458628807912771681561582305470619962729037215106906417797478639043759073671526233150625502278985764285240647638298730753064584097255132097117503260610789389208952460787617822842317892961282136931991642143510463540230197185978192801359575147064297679386685692745377174344652745131673928629270650815492384956061132688378461069498104923826104838586490333978324236045359018410309925372259501184209171535596493166826553453125808213238306178803667493845876133602921479334100929846229655991832177474753899009595435568508601133899514000384344943496815277376139253215613276576713699227714772914534633123209445267405754874955789973243763421775941905041695031197207082735610030092448816414167544', '2.36046379266846408899539510197326890965893678862224859555792838223', 848, 4);
    T('714.61241243068259484311371189785768955751306153276228425541721886979377905758882133446197375671353721565433127023956674702853562936178184807139172505791453268944641741010891530662590385073593152193455805953240947347607646479943534513966413454182559305296106776895316113963327870900366698599660151420070929801570405751345702353410644051682554211976059908915814146604831899204551744694140994594366464823764389570912629592676719346123171637726003544266618213687267325838123794874975527306940385282149798065123421458819840361280851912003779857428730583255104381030891533142101378117306978641247346815845252044894839537420273047299982385917944987149055257937383992606691257056568881397021059046488147159677518654506427704867625959014109859319284280932956098825423372240493017931941597922444987378795761902915931399027545696337420679858404395579037783661674', '510670.9', 848, 4);
    T('773.59472663687912451159535229389853939842605059162703128538789346863286440830494881075568422219621192333688675281822184873387170452981843472591508432370385971110199119193528015500760543414524536242455544083490104908083974792351201506935765119161643924169037264164099952814746396926161440416117565279304931119669840952024989118295794278477494347926021963207180850887861862534960178708980838985387368839444500154547698864030192542133112686175071539548217199903944950274902992241082394258689568848798217431438259861325777938469444568799288405922158163958958223163833937391698968594686798212843588496878387116515526389464647971296312524748306815742536620947543414457504426763294470951946122437335497099494910632585762600740409680286609238779306759977660554002182801075912581235512217070538480899809469699327979364989178344899103109616009308707414801940872', '598448.80108038774004895', 848, 4);
    T('19008878036934380567.84396002161007307494234445138537376816800048343616085674206626936199749636474720092004342883045275441459628576839920971330702661877884977464784187807450210290565316696580997145384562781682128000995898273298139743308228182662902555052057691717197189289590092259236658987055552383019917577612145454574579719020972478133587710937217336044281243302964649388576337469548420171603649754413233412982255987047582379037245718253216484000312226525699478281161585924999301376693452172203272239337852180725955560197800243914977419579018956725334811006235730357909615062778332245395221460477491786215707869006840640344479485584853728869293288919876710562517254695116835080137005037450411798398558498522188066156712112948975309706291905861236209197663377395628993573005290776392002172440861220587153770661110412322002802191380189193046537735515344097800652278112', '361337444223046269803581586105656008038', 848, 4);
    T('46167547682276.69659839142168613300984418968797481977026434015151042113254784992976400662854539508545353029755222531760805168040819029032876000018647210639969596871042153723516022413007597033768299999552605409537870966847134283383166186477065225427587170669800959221329223140225378864660505062217758873720294200147677451805234547441880387069084927910307749688653503693156999242010683739605876996616940672435261250941033196533134836202233289888667726447834984240822254637787801181963951894038647953211128853346780059900073416876543354577813540692889219448228728777909901391731624805683971528990295277436840161294579496431876765112237134229826779443182941938910283177882124027553167373791516105803517046194645899736931109009785536544701916275081569242392295709503599284148920065992874897610987646342569478342044371048498987276792566426432612227674148724386753297859', '2131442458995292379923442891.39619052405363168659553832', 848, 4);
    T('357.42144219667737476940723134311437237901070981997683604955545480813878989092658709890042422907764330003065939163827123844540310894482334918652641803484529491350116195509236941247526441637135860986047310858645732381561361417002489911900180310969665452837050841076667945444753611474726298122460938050651055487683172415032060683745296386129046642462594028333292429079902699978203730134661132884803238529838974477445191759967198471721764551979289888303016458787572583353512731118148130889821591351857798041593003937898405797601412782467072190053744174243225780844049835772183482949546683341934096818885756007124382254659796426711715080646604861045965994466209388552589568508290975087418855481674502829846363820364296679009102756062772768505880054488205022390142319706776092235958701849286052672927602214663014895377779429627657631859807708535799000791569', '127750.0873419527858363938899699529356795979', 848, 4);

    T('7962389669.802619226612420082472137241175608666661735561414607900997891208674399029037690473846801874885757356935098345580426591903055562834097030473393461243383271678522047266370481726950449902079177413029320835311617540338949723112537110203362427858192330773112242226738687601815379027664862463277214286567895961535965292898558369885348154321255856088316638660454457806259636801735717647353100345104124424309537274722443977557763052770155305495894443403910135805323387075986979884756306279940582254155475425470071419924272589017637263436788853746084045571609754415271337660969860121360062716460814804823355709978626612344057738355406928320053347312100043083244150469170748731787389950929048830984414804237666515908525165972258804621273737876286813438283880399279278335982619019392983109710369604460059859022468994373480045434248040713896862456845920018164581498873236394584793215632879036361684025776086529569', '63399649253779463637.88320468645646644051', 900, 4);
    T('36985.537435306622236467270937252825254026014721851309188254305389713697015230264185263378799026382334714397309571564625416241215217867149723572425854181806966554642360423829060014881901773199386994460664634062101904797359294551321318937235734678512961188607801986220680153014134029493832615546969623974916827167078654634358574333678389616366410278915565703581941672170730893983545040190300249274221426700113848290541507663318853535927775893362185694382544176373482231304831101837902849800951258220726268990677590958157914974175471404517996778205179950329995483136931650616286373118261630043285103533033040476431824360766396365422877272153761572905701234187642833656097211644081252236980059582418494584853215946576617549259994456993750322107749520964217795499374138097379418654677424839627787482713659455961703181646841698674573783883478313655188817893687086751450042662911182317161526490882315503431203481', '1367929979.37846755563562136', 900, 4);
    T('138.653825283302247079520109639909554830495824478735118824960659298570304260365235201452949658158554711469364399658798667165750129626981635907513515727752104322402093210187651015212649867280391248459370998127185412494794237194604573645718383791562171900088214128456904501530127980235949384533924069927959103110766006405141214543337550010111094563431666164054892460728841085337947430066655077091396441886965426495467949136348081716343859215231704926659844712690080814553677280196114278268613368774166884194655930307728924841441917187821902525242579401284440544702546289065296084983476636972703960454064127793719473325530264873714830294999070781526696237151909302447919905808564848939241449791664413964130684658760296224947984073972311661335447223880763498816716066907256956648196113869920459515551692886783038790708833001987620466496959254546687493058038109403699935644806603442158642646086655023039259769', '19224.883265692505457601247932291125354259868778652089681716374870935', 900, 4);
    T('50950.440174742357175572071690890164127102751485515727366550390683608820436132956072025681803295779371667169337127406084175358445300796083953408843705969018873810510032889756321223393470759559987454753284749000495070038043049225296587093286942372337838245436787768377078265062338033176995223379224008076742040545070485158584805287828654557110412928438688654162185555625503306254982210625622689237909265654674405693567429263013187828668914075564173639899430670900405284455694427199638591118702956228653269921902023423130819746958603054104765246190233159809281895358531923339811813637075650854380095230369039753422001934387105096606677337450258221260413504059391491686285381140987958601361542484696758005412715512502019501470056625769917175363496998761143393423591945956802212685363531078784887861348308753434147834816578247850672086197881232957501680862454026927989538997998521459117388396046360043657658205', '2595947354.0', 900, 4);
    T('10046454623254826961875382317790.784871452244144975743064060498244021304220015969466939024480223499960901641559066665153037197032593546502649980935727654208439388853861419936425552530271530035494928115227726482675679211892969869213891378152728772302193716001319517533416123919336946085430093922543201768660204280287527989092149328624358683559115176981914090302709863561780525393374380243154381915249441910602391360612168161991933174297686912644986231916957310635449019093633359598148150548731899729849141307548127201093974033939034042561257126753456557941211256135898280856479300825592342274773764495952079469431439741906942077077162075390603294007186657196505030105092603376243537890291750177091215762681318887536194545745370727643575228977262734270257261824137515623111996150864948935760969962110266201746190014377183448005770929873280926905656965694617633851807904103533078781099132125392433972079961527137877338300842675298415396', '100931250497118287147460895653422009190477535915527140997436145.5', 900, 4);
    T('50312147.895978789945318850814716769716556529351439472968676011531432412685095617354326688006158268280144448102851948493093105193248333706042845584818818717006618051899381392239777663697210274767993016068817609233147463673391458914578371028438213381481915193432915606333241411830951892952556850603476911011964242221281045401447599028993820263011805445545067496904941492500519891679173682065830498079066798970609902316454882840175774548136572667012712115571877757306654257623667462477526405934096044262931622005593806759836229355795092155596007885177190326989920383365286112949649812553854003818096593686636208662967380298108969324741181800273045253898252610330097550632621999520072653397155741940824966908033268285600350252107768305144976490584881544838157210518233799995832553321591285254897707500335337126009624774906497706411815347441932376790317896998001657977732263675649024815291900604673316858390505885', '2531312225906842.98', 900, 4);
    T('5.75006173879898440869241532260800496489142124887111833188883261579283008763735295296179041343250976501937101036585472900854336327489380123282821037481044202916203441849004689471000360958945402609580784572353735408452226943576599178952237206876788896067842553596176879865752033951176088455975165207593727159176177085095190710656229140725205998687018442446680477310130738465615556577329602100580746699396762667032579047007129320587668015965987773825003138735379716404415157863124832466280028377682906939092007140554677511356063954505216670858009340392141492467369216371170692156583092566735501630865322582512619490429963282482788624823392158158604152325793299574006823928013422664018092614581197981204160781788183807939164255353777192327520682879326632498396112963534952155920134048996062606067862129834413685924986614062985203382638465654985330058303916447123647081510424589324268892533850475930056739', '33.06321', 900, 4);
    T('2.028430427695266062116855907049682671555242569377976031112753398044812309192685964100965099265530959360284365757870326033053601551523629075378378856745863785169732838994844919825336134776071475304944281214520910583421287945003559393430216592924156944793703503429916620090936360805054797721801359569417127688057382996969292921041738603630480342967692998233877581650620914181744888433057925936957891749666196052068806893889255339075884495080504767342571830767185504079538214007613797919720564247610464573496500053065413926974012933910366107634249863611197215314558918759158057008769626766485169836568960326396402057910902853473776179915408801623228568357317462913587164892064775658027833999904990053550155044417555290139236050616243301785245316354941879867361509091783046268854502291693337916935351747382119202947309008531765489919678841207661025090634628632028473270852667224552478918030875120109633343', '4.11453', 900, 4);
    T('14733978519855.986227038024852091926922255130476070894431360177088165996112823755229640976682373714559133174204321582457357379198592714332912375824678967479352499320917276441508273646332050526613540155298794906758075083423887078200491328803728379571737163423282193094119893971969158039986280336661223358044916730250644983211811708805361569086764053956417999765164201297583468149451593886541587337271459070510498401319438130775438554793347084598691604611470750783139289554584374151809770130348507437456446993653835198138094762369827003035065371494357530678660768629856933317557149699487339377901600994041273525198654016009799518759093883882149332746270829795417341182432885267251862546055608965944281933745525115737980560603226764663919494161818265452113376337284183674111018389097369572010360312518568654815367830169145059770161568817087559064704425726854477168276312695713542835478594810618373570380952296080331906', '217090123023577598725208942.7547', 900, 4);
    */

    T('3.7', '13.69', 1, 6);
    T('14.8', '219.04', 1, 6);
    T('3.578708384438441', '12.80715370085', 15, 6);
    T('49.56954417823', '2457.139710037', 11, 6);
    T('4.9808503290100979', '24.80887', 16, 6);
    T('54108953.7650348268963935650889893674872', '2927778877546676.5690784769302', 31, 6);
    T('2076.600289732761739123272421368', '4312268.76331819', 27, 6);
    T('7133463.90984542863204814019379', '50886307353067.22955041604721776192', 23, 6);
    T('698.204626309508207056912760818928760494769304', '487489.7002', 42, 6);
    T('767720659.69699525124684551424800638106577299', '589395011325589588.7', 35, 1);
    T('176542124.06107607520987374389174', '31167121567996376.09', 23, 4);
    T('57915787478641.83665522345250623634912714466', '3354238439271206551123435622.197210160416', 29, 1);
    T('744842151657434987604.2791022189701', '554789830885677382051879929528491741366735.693239816410110026056705884', 13, 3);
    T('95147005221980315847.29763029742483', '9052952602711549492924074356460479036559.21', 14, 1);
    T('57999657.955796300947757717200828', '3363960322989365.1472240139389186879250416930', 24, 1);
    T('7.2644370552737657226851043538350751488055', '52.7720457300345807456', 40, 3);
    T('41913846342.70464397859492535599', '1756770515239855459853.783', 20, 5);
    T('105264062737.871347006', '11080522904082514982875.06226597726166', 9, 5);
    T('1543838165699.532476511077553323', '2383436281870497095278880.8298819175197', 18, 6);
    T('1.0000000000001', '1.000000000000000001402', 13, 2);
    T('18.000001', '324.0000000000000000000008467', 6, 2);
    T('11.00000000000000001', '121.000000000000000000000000001944', 17, 0);
    T('2.00000000000001', '4.0000000000000000000000000000000005485', 14, 0);
    T('7.1', '49.00000000000000000000007338', 1, 0);
    T('19.00000000000001', '361.0000000000000000000000005645', 14, 2);
    T('1.0000001', '1.00000000000000000000000000000006986', 7, 0);
    T('30771277.0274560007853901874282412', '946871489900441.411712749999999999999999999999999999999999999999', 25, 6);
    T('668.093253221434649746', '446348.594999999999999999999999999999', 18, 5);
    T('0.00000000001', '1E-22', 11, 3);
    T('0.0000000000000001', '1E-32', 16, 3);
    T('0.9999999999999999999999999999999999999999', '0.9999999999999999999999999999999999999998', 40, 5);
    T('0.9999999999999999999997187097975391240474677563824', '0.9999999999999999999994374195950782480949355918890297218761543222489261', 49, 3);
    T('0.0000000000000298840312673111495324166498096924391126062701', '8.9305532478563043E-28', 58, 5);
    T('0.000000000000000000002', '4E-42', 21, 3);
    T('0.00000000001', '1E-22', 11, 1);
    T('0.000000000000000856007781721035575', '7.32749322366968086676820306213E-31', 33, 1);
    T('0.99999999995656155671', '0.99999999991312311342744139242664416779926531', 20, 6);
    T('0.999989535626887626186838139976861683253478341268', '0.999979071363278357008648285297622562859535708874691230', 48, 4);
    T('0.99999999995596524660275731456220677327214', '0.99999999991193049320745368863116952277069525', 41, 6);
    T('0.99999994006685191867', '0.99999988013370742932740206713297868240007690621350154', 20, 6);
    T('0.0000089442719099991587856366946749251049417624734384461', '8E-11', 55, 3);
    T('0.99', '0.999993113013983408011715774682818573128', 2, 1);
    T('0.00000611316054732199642', '3.7370731877334170822166605E-11', 23, 2);
    T('0.013784843329905140561885790159255377437161367698', '0.0001900219056300302439144565717101815984692', 48, 1);
    T('0.9999999999817812359640851', '0.99999999996356247192850223214991806585959346352594', 25, 3);
    T('0.99997031055926855452699324838818843518195463349', '0.999940622', 47, 2);
    T('0.0000512692131367528306972385838679872115023412', '2.62853221566178902944066364E-9', 46, 4);
    T('0.057670466428', '0.003325882698071939316360450586446371022801794699158', 12, 5);
    T('0.99554', '0.9911', 5, 1);
    T('0.000015', '2.47381003984238390102286027217942815856570969E-10', 6, 1);
    T('0.999999970569604066925897611247667114936', '0.999999941139209', 39, 0);
    T('0.999999999989796679982837626159284554', '0.99999999997959335996577936005794173576329050240929182', 36, 1);
    T('0.1077032961', '0.0116', 10, 1);
    T('0.0141421356237309', '0.0002', 16, 1);
    T('0.9999998768394081493054516338497161179893323296', '0.9999997536788314671422882810678304235', 46, 0);
    T('0.238956062906970410270245989604192732', '0.0571', 36, 5);
    T('0.7660546671742345565286820470483750453763052102935', '0.5868397530994272793933020836897890562985049351', 49, 2);
    T('0.22', '0.0442337787336141631720176764858285725442961570', 2, 2);

    T('179.7241697915052581033756967731719513432692508', '32300.77720724581117', 43, 2);
    T('5789158040893098296794880930.864924830060829236', '33514350822437215971598406911256644694273118558749175249.0', 18, 0);
    T('2824221124337856947704.8012478416145', '7976224959156188833310273317864098707012606.0', 13, 0);
    T('86263.03333928827853610459', '7441310920.8951610508629', 20, 6);
    T('454426339695348359894208983.6714002627162343460178', '206503298208912140516268973108976781377094781756302253.10316325966', 22, 3);
    T('7.258787777583802594250936', '52.69', 24, 5);
    T('2001920924184258605558359872767592948024.8543', '4007687386686756091822637898553155471331634538521264108680442236674656851419653.351763057', 4, 4);
    T('6576613095964895855186161492615.465', '43251839814016972458988090681594663492334963080297610004985253.9613', 3, 3);
    T('218383218310958069785872.301927584', '47691230039851571982509649628982296504016241058.977955327', 9, 3);
    T('60728412086816922880678290796418806.6871171', '3687940034586251730077294264351100568013474596967205874685411761358250.20335920865', 7, 0);
    T('45974968665.29964986693286971876228807290133907777', '2113697743775284668710.510', 38, 6);
    T('939851185518650310576202.501936615855344549247', '883320250920812443260322053150977725907138537017.6109350', 21, 6);
    T('30792657782480416481776811935773797404.0003', '948187773308951760156602418478473823522730515817565059043672065724339252452.6444', 4, 2);
    T('26117518326666795322375597420041.35839298', '682124763543775920380116102948198971902395978922979169982661154.965777107019', 8, 0);
    T('2355735861300212899781.981274', '5549491448215855908992552523083142006504786.98270', 6, 5);
    T('2148665877305983003246901853661.043', '4616765052299089605130822036135055816921350866756700565077388.7685983542855487709', 3, 4);
    T('9679153997438690735.573001818', '93686022106133386382529030832963639287.904519199', 9, 6);
    T('800022506035493161330210575234.52459826997456', '640036010163310691747774480314864168983618826504059678896941.563828701', 14, 2);
    T('985.261772431685842274639340000514709128327111', '970740.7602152271', 42, 0);
    T('6995180742044463030779239979571353121953.293344', '48932553613869724437277762478094081188923485912516533320212206010023820963789544.269716913808775020', 6, 5);
    T('5013216366623851214.1896508660273321751', '25132338338585248190238248636234491768.941684121900', 19, 6);
    T('276902636348797086879219133.1612149', '76675070016914161693612991003380354517429223150278706.17849523', 7, 6);
    T('237564977178649182.665063047451442', '56437118381892106973704533609598484.60513868513', 15, 0);
    T('10.142', '102.85637382898348', 3, 6);
    T('30.397185674769991772025241876525529887', '923.9888969464420', 36, 6);
    T('3088127.79209138814087405', '9536533260287.231779393092478', 17, 6);
    T('27117.1218455773753951', '735338297.187889721906', 16, 1);
    T('24231197304168807169995.56301580963194716353', '587150922793557668101013208341938836941193210.8', 20, 4);
    T('37838002948618379.82724648128060201962219175', '1431714467139653206157054572446837.45539086584', 26, 5);
    T('6606.9228853267377281408395352', '43651430.012654185172', 25, 1);

    T('1947.1775', '3791500.199076232676051111', 4, 4);
    T('1336693769272.52325053384226966971097495236409237893207026280721123585', '1786750232811985622866694.6930741720574199999999999999', 56, 1);
    T('24311351864975.074869841182393118014836563', '591041829502627051005856744.915583060322969111', 27, 3);
    T('19829.25380894884324691326015863932659629897679659749823296445916409484', '393199306.619712207999', 65, 4);
    T('13978249.72', '195391465198312.4267582256299999999999999999999', 2, 4);
    T('2797202497359.7823516910213202974336269525752814583381669915057480062628591796293', '7824341811235803194182757.3598207579999', 67, 6);
    T('760894577043.5606489205461622645', '578960557374299052070386.48762474789323725999999999999999', 19, 6);
    T('42984058671789597132.7221112172629645', '1847629299899850465200384668635639967676.41', 16, 3);
    T('571382471180033.53319393302415277060828996943848848613781018317122334349', '326477928371801851150832020172.11648736659691111111111', 56, 6);
    T('84610737.37567766914912673289129529849968285511523295491', '7158976879255898.06343326264999', 47, 4);
    T('1831263691.5418751838', '3353526707959576179.2551166298974611111111111111111111', 10, 0);
    T('2.9750758830658420703', '8.85107651', 19, 4);
    T('20712.1421743511385380370509626991399201024596581971743168935176174593609271719', '428992833.450535108921311111111111111111111', 73, 5);
    T('18.9', '357.5299999999999999', 1, 1);
    T('104755563.3530892375895526651', '10973728053423092.8705407144035421', 19, 6);
    T('37.858905522559828896825267705453973368', '1433.2967273661111111111', 36, 1);
    T('18354976800132.7205784689505763982983253834776256454914296697190444244517734', '336905173333410406277377949.452223919511111111111111111111', 61, 0);
    T('3976.608172551970959396465282421461773647692323490773537804717860364788551', '15813412.55800712603999', 69, 4);
    T('24590607446647400780.287370661652288997152', '604697974595110599812980781794694145333.72111111111', 21, 1);
    T('2263343854329.638037260235405086446634', '5122725402931741767463814.840999999', 24, 4);
    T('973938933649.61947435573350430363376265175296752969591132312', '948557046478557884842093.364689883449666', 47, 3);
    T('218875865535828595.704079834928718536449349620686728272', '47906644514058120040043799772367208.9448736248559332241111111111111', 36, 1);
    T('9109612980191986763.27770675993117809970934764', '82985048648882330621522428607079458060.489945999999999999999', 26, 1);
    T('1767.77976097459671229637826840519162978769', '3125045.2833114022852699999999999999999999', 38, 0);
    T('5.099681356298784586945350294019810951758898271011656537155961056844', '26.0067499357814111111', 66, 1);
    T('17966734.447350027552808230157515476124640562819146055765307390247119289', '322803546701594.09999', 63, 0);
    T('63216259733951.59', '3996295494750428253401001163.9486336398226471847', 2, 0);

    T('0.0000000000000000000000000000000000000000025385115498645', '6.4440408887956349899441566682289191111111111111111111E-84', 55, 5);
    T('0.0000000000000000000000000000000000000000000000000000140471526414353053458592377645343274655348630930173397', '1.97322497331782876158645511100784258994087355048761111111111111E-106', 106, 5);
    T('0.0000000000000000000000000000000000000000008', '6.60007159131152131407609999E-85', 43, 4);
    T('0.00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002598902022257132166703490111187879281161770014282895', '6.7542917212922111E-232', 167, 5);

    log('\n ' + passed + ' of ' + total + ' tests passed in ' + (+new Date() - start) + ' ms \n');
    return [passed, total];
})(this.BigNumber);
if (typeof module !== 'undefined' && module.exports) module.exports = count;