#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Copyright 2022, Nils Hilbricht, Germany ( https://www.hilbricht.net )

This file is part of the Laborejo Software Suite ( https://www.laborejo.org ).

This application is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import logging; logger = logging.getLogger(__name__); logger.info("import")

from PyQt5 import QtCore

class EventLoop(object):
    def __init__(self):
        """The loop for all things GUI and controlling the GUI (e.g. by a control midi in port)

        By default use fastConnect.

        0 ms means "if there is time". 10ms-20ms is smooth. 100ms is still ok.
        Influences everything. Control Midi In Latency, playback cursor scrolling smoothnes etc.

        But not realtime. This is not the realtime loop. Converting midi into instrument sounds
        or playing back sequenced midi data is not handled by this loop at all.

        Creating a non-qt class for the loop is an abstraction layer that enables the engine to
        work without modification for non-gui situations. In this case it will use its own loop,
        like python async etc.

        A qt event loop needs the qt-app started. Otherwise it will not run.
        We init the event loop outside of main but call start from the mainWindow.
        """

        self.fastLoop = QtCore.QTimer()
        self.slowLoop = QtCore.QTimer()

    def fastConnect(self, function):
        self.fastLoop.timeout.connect(function)

    def slowConnect(self, function):
        self.slowLoop.timeout.connect(function)

    def fastDisconnect(self, function):
        """The function must be the exact instance that was registered"""
        self.fastLoop.timeout.disconnect(function)

    def slowDisconnect(self, function):
        """The function must be the exact instance that was registered"""
        self.slowLoop.timeout.disconnect(function)

    def start(self):
        """The event loop MUST be started after the Qt Application instance creation"""
        logger.info("Starting fast qt event loop")
        self.fastLoop.start(20)
        logger.info("Starting slow qt event loop")
        self.slowLoop.start(200)

    def stop(self):
        logger.info("Stopping fast qt event loop")
        self.fastLoop.stop()
        logger.info("Stopping slow qt event loop")
        self.slowLoop.stop()
