#!/bin/sh
# SPDX-License-Identifier: BSD-2-Clause
# SPDX-FileCopyrightText: © 2023 Antoine Le Gonidec <vv221@dotslashplay.it>
set -o errexit

###
# Frail Hearts
# send your bug reports to contact@dotslashplay.it
###

script_version=20240611.1

PLAYIT_COMPATIBILITY_LEVEL='2.29'

GAME_ID='frail-hearts'
GAME_NAME='Frail Hearts: Versicorae Domlion'

ARCHIVE_BASE_0_NAME='setup_frail_hearts_versicorae_domlion_1.1.1gog_(62511).exe'
ARCHIVE_BASE_0_MD5='0079a30ca10ed1eecc3ca0676c6134f8'
## Do not convert file names to lowercase.
ARCHIVE_BASE_0_EXTRACTOR='innoextract'
ARCHIVE_BASE_0_EXTRACTOR_OPTIONS=' '
ARCHIVE_BASE_0_SIZE='610000'
ARCHIVE_BASE_0_VERSION='1.1.1-gog62511'
ARCHIVE_BASE_0_URL='https://www.gog.com/game/frail_hearts_versicorae_domlion'

CONTENT_PATH_DEFAULT='www'
CONTENT_GAME_MAIN_FILES='
audio
data
fonts
icon
img
js
lib
locales
movies
save
www
index.html
translationengine.html
greenworks.js
package.json
steam_appid.txt.txt'

APP_MAIN_TYPE='custom'
APP_MAIN_EXE='index.html'
APP_MAIN_ICON='icon/icon.png'

PKG_MAIN_DEPENDENCIES_COMMANDS='
firefox
python3'

# Load common functions

PLAYIT_LIB_PATHS="
$PWD
${XDG_DATA_HOME:="${HOME}/.local/share"}/play.it
/usr/local/share/games/play.it
/usr/local/share/play.it
/usr/share/games/play.it
/usr/share/play.it"

if [ -z "$PLAYIT_LIB2" ]; then
	for playit_lib_path in $PLAYIT_LIB_PATHS; do
		if [ -e "${playit_lib_path}/libplayit2.sh" ]; then
			PLAYIT_LIB2="${playit_lib_path}/libplayit2.sh"
			break
		fi
	done
fi
if [ -z "$PLAYIT_LIB2" ]; then
	printf '\n\033[1;31mError:\033[0m\n'
	printf 'libplayit2.sh not found.\n'
	exit 1
fi
. "$PLAYIT_LIB2"

# Extract game data

archive_extraction_default

# Include game data

content_inclusion_icons
content_inclusion_default

# Write launchers

custom_launcher() {
	local application
	application="$1"

	launcher_headers
	game_id=$(game_id)
	path_game=$(path_game_data)
	web_path=$(application_exe_escaped "$application")
	cat <<- EOF
	# Set the environment

	GAME_ID='${game_id}'
	PATH_GAME='${path_game}'
	WEB_PATH='/${web_path}'

	EOF

	# Generate the game prefix
	launcher_prefix_symlinks_functions
	launcher_prefix_symlinks_build

	# A fake $HOME path is used to not mess up with the user real Firefox profiles.
	persistent_path_diversion
	fake_home_persistent

	{
		cat <<- 'EOF'
		# Compute the path to the Firefox profile for the current session
		firefox_profile_path() {
		    # Firefox profile path can be explicitely set using an environment variable
		    if [ -n "$PLAYIT_FIREFOX_PROFILE_PATH" ]; then
		        printf '%s' "$PLAYIT_FIREFOX_PROFILE_PATH"
		        return 0
		    fi
		    # Compute the default Firefox profile path if none has been explicitely set
		    printf '%s/play.it/firefox/%s' \
		        "${XDG_CACHE_HOME:="$HOME/.cache"}" \
		        "$GAME_ID"
		}

		EOF
	} | sed --regexp-extended 's/( ){4}/\t/g'
	{
		cat <<- 'EOF'
		cd "$PATH_PREFIX"

		# Start local Web server
		WEB_HOST='localhost'
		WEB_PORT='8000'
		python3 -m http.server \
		    --directory "$PWD" \
		    --bind "$WEB_HOST" "$WEB_PORT" &
		WEB_PID=$!

		# Set the path to the dedicated Firefox profile and the local URL of the game
		FIREFOX_PROFILE_PATH=$(firefox_profile_path)
		WEB_URL="http://${WEB_HOST}:${WEB_PORT}${WEB_PATH}"

		EOF
	} | sed --regexp-extended 's/( ){4}/\t/g'

	# Enable a fake $HOME path, to prevent messing up with the user regular Firefox profiles
	fake_home_enable

	cat <<- 'EOF'
	# Run Firefox, using a dedicated profile
	firefox -CreateProfile "./play.it $FIREFOX_PROFILE_PATH"
	## Ignore firefox failure exit code in case of crash, so post-run actions are not skipped.
	firefox -kiosk -no-remote -profile "$FIREFOX_PROFILE_PATH" "$@" "$WEB_URL" || true

	EOF

	# Disable the fake $HOME path
	fake_home_disable

	cat <<- 'EOF'
	# Stop local Web server
	kill -s TERM $WEB_PID

	EOF

	launcher_exit
}
launchers_generation

# Build packages

packages_generation
print_instructions

# Clean up

rm --recursive "$PLAYIT_WORKDIR"

exit 0
