/* livepatch-utils.c
 * Copyright (C) 2019 Canonical Ltd
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor
 * Boston, MA  02110-1301 USA.
 */

#include "livepatch-utils.h"

#include <gio/gdesktopappinfo.h>
#include <glib.h>
#include <json-glib/json-glib.h>

#define OS_RELEASE_PATH "/etc/os-release"

GQuark
livepatch_error_quark(void)
{
  static GQuark q = 0;

  if (G_UNLIKELY(!q))
    q = g_quark_from_static_string("livepatch-error-quark");

  return q;
}

static GKeyFile *
parse_osrelease()
{
    g_autoptr(GKeyFile) keys = NULL;
    g_autofree gchar *content = NULL;
    g_autofree gchar *content_with_group = NULL;
    const gchar *group = "[os-release]\n";
    g_autoptr(GError) error = NULL;

    if (!g_file_get_contents(OS_RELEASE_PATH, &content, NULL, &error))
      {
        g_warning("Failed to read '%s', error: %s",
                  OS_RELEASE_PATH, error->message);
        return NULL;
      }

    content_with_group = g_strdup_printf("%s%s", group, content);

    keys = g_key_file_new();
    if (!g_key_file_load_from_data(keys, content_with_group, -1,
                                   G_KEY_FILE_NONE, &error))
      {
        g_warning("Failed to parse /etc/os-release: %s", error->message);
        return NULL;
      }

    return g_steal_pointer(&keys);
}

static JsonNode *
livepatch_get_status(GError **error)
{
  g_autofree gchar *std_output = NULL;
  g_autofree gchar *std_error = NULL;
  g_autoptr(JsonParser) json_parser = NULL;
  JsonNode *root_node;

  if (!g_spawn_command_line_sync("canonical-livepatch status --format=json",
                                 &std_output, &std_error, NULL, error))
    return NULL;

  if (std_output == NULL || strlen(std_output) == 0)
    {
      if (std_error == NULL || strlen(std_error) == 0)
        {
          g_set_error(error,
                      LIVEPATCH_ERROR, LIVEPATCH_ERROR_CMD_FAILED,
                      "canonical-livepatch returned an empty status.");
        }
      else
        {
          g_set_error(error,
                      LIVEPATCH_ERROR, LIVEPATCH_ERROR_CMD_FAILED,
                      "canonical-livepatch status returned an error: %s",
                      std_error);
        }

      return NULL;
    }

  json_parser = json_parser_new();
  if (!json_parser_load_from_data(json_parser, std_output, -1, error))
    return NULL;

  root_node = json_parser_get_root(json_parser);
  if (root_node == NULL)
    {
      g_set_error(error,
                  LIVEPATCH_ERROR, LIVEPATCH_ERROR_CMD_FAILED,
                  "The output of canonical-livepatch has no json root node.");
      return NULL;
    }

  return json_node_copy(root_node);
}

static JsonNode *
livepatch_get_status_node(const gchar *expr, GError **error)
{
  g_autoptr(JsonNode) root = NULL;

  root = livepatch_get_status(error);
  if (root == NULL)
    return NULL;

  return json_path_query(expr, root, error);
}

gboolean
livepatch_has_settings_ui()
{
  g_autoptr(GDesktopAppInfo) info = NULL;

  info = g_desktop_app_info_new(LIVEPATCH_DESKTOP_FILE);
  return info != NULL;
}

gboolean
livepatch_is_supported()
{
  g_autoptr(GKeyFile) file = NULL;
  g_autofree gchar *version = NULL;
  g_autoptr(GError) error = NULL;

  file = parse_osrelease();
  if (file == NULL)
    return FALSE;

  version = g_key_file_get_string(file, "os-release", "VERSION", &error);
  if (version == NULL)
    {
      g_warning("Failed to get the version from the file %s: %s",
                OS_RELEASE_PATH, error->message);
      return FALSE;
    }

  return g_strstr_len(version, -1, "LTS") != NULL;
}

gboolean
livepatch_is_running()
{
  g_autoptr(JsonNode) result_node = NULL;
  JsonNode *running_node = NULL;
  JsonArray *result_array;

  result_node = livepatch_get_status_node("$.Status[0].Running", NULL);
  if (result_node == NULL)
    return FALSE;

  result_array = json_node_get_array(result_node);

  if (json_array_get_length(result_array) == 0)
    return FALSE;

  running_node = json_array_get_element(result_array, 0);
  return json_node_get_boolean(running_node);
}

gchar *
livepatch_get_state(GError **error)
{
  g_autoptr(JsonNode) result_node = NULL;
  JsonNode *state_node = NULL;
  JsonArray *result_array;
  const gchar *expr = "$.Status[0].Livepatch.State";

  result_node = livepatch_get_status_node(expr, error);
  if (result_node == NULL)
    return NULL;

  result_array = json_node_get_array(result_node);

  if (json_array_get_length(result_array) == 0)
    {
      g_set_error(error,
                  LIVEPATCH_ERROR, LIVEPATCH_ERROR_NOMATCH,
                  "No matches for: %s", expr);
      return NULL;
    }

  state_node = json_array_get_element(result_array, 0);
  return g_strdup(json_node_get_string(state_node));
}

gchar *
livepatch_get_check_state(GError **error)
{
  g_autoptr(JsonNode) result_node = NULL;
  JsonNode *state_node = NULL;
  JsonArray *result_array;
  const gchar *expr = "$.Status[0].Livepatch.CheckState";

  result_node = livepatch_get_status_node(expr, error);
  if (result_node == NULL)
    return NULL;

  result_array = json_node_get_array(result_node);

  if (json_array_get_length(result_array) == 0)
    {
      g_set_error(error,
                  LIVEPATCH_ERROR, LIVEPATCH_ERROR_NOMATCH,
                  "No matches for: %s", expr);
      return NULL;
    }

  state_node = json_array_get_element(result_array, 0);
  return g_strdup(json_node_get_string(state_node));
}

gssize
livepatch_get_num_fixes(GError **error)
{
  g_autoptr(JsonNode) node = NULL;
  JsonArray *array;

  node = livepatch_get_status_node("$.Status[0].Livepatch.Fixes[*]", error);
  if (node == NULL)
    return -1;

  array = json_node_get_array(node);
  return json_array_get_length(array);
}
