#!/usr/bin/env python

# This file is part of Window-Switch.
# Copyright (c) 2009-2013 Antoine Martin <antoine@nagafix.co.uk>
# Window-Switch is released under the terms of the GNU GPL v3

from winswitch.util.simple_logger import Logger
logger = Logger("user_base")
debug_import = logger.get_debug_import()

debug_import("globals")
from winswitch.globals import LOCAL_IPP_PORT, SAMBA_ENABLED
debug_import("common")
from winswitch.objects.common import ModifiedCallbackObject, ALLOW_FILE_SHARING, ALLOW_PRINTER_SHARING
debug_import("consts")
from winswitch.consts import XPRA_TYPE, NX_TYPE, SSH_TYPE, VNC_TYPE, SCREEN_TYPE, LIBVIRT_TYPE, WINDOWS_TYPE, VIRTUALBOX_TYPE, GSTVIDEO_TYPE, BINASCII
debug_import("crypt_util")
from winswitch.util.crypt_util import make_key_fingerprint, recreate_key

class UserBase(ModifiedCallbackObject):
	"""
	Common superclass for user information.
	See GlobalSettings and User for actual use.
	"""

	def __init__(self):
		ModifiedCallbackObject.__init__(self)
		self.uuid = ""
		self.name = ""

		self.crypto_modulus = 0l
		self.crypto_public_exponent = 0l

		self.avatar_icon = None
		self.avatar_icon_data = None
		self.locale = None
		self.preferred_session_type = XPRA_TYPE
		self.preferred_desktop_type = VNC_TYPE

		self.supports_xpra = True
		self.supports_nx = True
		self.supports_vnc = True
		self.supports_rdp = True
		self.supports_ssh = True
		self.supports_screen = True
		self.supports_gstvideo = True
		self.supports_virtualbox = True
		self.line_speed = 1000*1000				#Defaults to 1MBit/s

		self.tunnel_fs = SAMBA_ENABLED and ALLOW_FILE_SHARING
		self.tunnel_sink = True
		self.tunnel_source = False
		self.tunnel_clone = False
		self.tunnel_printer = LOCAL_IPP_PORT>0 and ALLOW_PRINTER_SHARING
		self.mount_points = []					#list of mount points on the client
		self.key = None
		self.xkbmap_print = ""
		self.xkbmap_query = ""
		self.xkbmap_layout = ""
		self.xkbmap_variant = ""
		self.binary_encodings = [BINASCII]
		self.gstaudio_codecs = []
		self.gstvideo_codecs = []


	def __str__(self):
		return	"User(%s : %s)" % (self.name, self.uuid)

	def __hash__(self):
		return self.uuid.__hash__()

	def __cmp__(self, other):
		if type(other)!=type(self):
			return -1
		c = cmp(self.name, other.name)
		if c==0:
			c = cmp(self.uuid, other.uuid)
		if c==0:
			c = ModifiedCallbackObject.__cmp__(self, other)
		return c

	def get_key(self):
		if not self.key:
			self.key = recreate_key(self.crypto_modulus, self.crypto_public_exponent)
		return self.key

	def regenerate_key_fingerprint(self):
		self.key_fingerprint = make_key_fingerprint(self.crypto_modulus, self.crypto_public_exponent)
		return	self.key_fingerprint

	def get_key_fingerprint(self):
		if not self.key_fingerprint or len(self.key_fingerprint)==0:
			return	self.regenerate_key_fingerprint()
		return	self.key_fingerprint

	def get_avatar_icon(self):
		if not self.avatar_icon and self.avatar_icon_data:
			from winswitch.util.icon_util import get_icon_from_data
			self.avatar_icon = get_icon_from_data(self.avatar_icon_data)
			if not self.avatar_icon:
				self.serror("invalid data for %s" % self)
				self.avatar_icon_data = None
		return self.avatar_icon

	def set_avatar_icon_data(self, data):
		self.avatar_icon = None
		self.avatar_icon_data = data

	def can_access_session(self, session):
		""" Can the user access this session?
			Check the session type and don't allow Windows users to try to resume their own session
			"""
		if not self.can_access_session_type(session.session_type):
			return False
		if session.owner==self.uuid and session.session_type==WINDOWS_TYPE:
			return False
		return True

	def can_access_session_type(self, session_type):
		if session_type==XPRA_TYPE:
			return	self.supports_xpra
		if session_type==NX_TYPE:
			return	self.supports_nx
		if session_type in [VNC_TYPE, LIBVIRT_TYPE]:
			return	self.supports_vnc
		if session_type in [WINDOWS_TYPE, VIRTUALBOX_TYPE]:
			return	self.supports_rdp
		if session_type==SSH_TYPE:
			return	self.supports_ssh
		if session_type==SCREEN_TYPE:
			return	self.supports_screen
		if session_type==GSTVIDEO_TYPE:
			return	self.supports_gstvideo
		return	False
