// @HEADER
//
// ***********************************************************************
//
//        MueLu: A package for multigrid based preconditioning
//                  Copyright 2012 Sandia Corporation
//
// Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
// the U.S. Government retains certain rights in this software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact
//                    Jonathan Hu       (jhu@sandia.gov)
//                    Andrey Prokopenko (aprokop@sandia.gov)
//                    Ray Tuminaro      (rstumin@sandia.gov)
//
// ***********************************************************************
//
// @HEADER
#ifndef MUELU_LOCALORDINALTRANSFER_FACTORY_DECL_HPP
#define MUELU_LOCALORDINALTRANSFER_FACTORY_DECL_HPP

#include "MueLu_ConfigDefs.hpp"
#include "MueLu_TwoLevelFactoryBase.hpp"
#include "Xpetra_MultiVector_fwd.hpp"
#include "Xpetra_MultiVectorFactory_fwd.hpp"
#include "Xpetra_CrsGraph_fwd.hpp"

#include "MueLu_CoarseMapFactory_fwd.hpp"
#include "MueLu_LocalOrdinalTransferFactory_fwd.hpp"
#include "MueLu_Utilities_fwd.hpp"

namespace MueLu {

/*!
  @class LocalOrdinalTransferFactory class.
  @brief Class for transferring a vector of local ordinals from a finer level to a coarser one, where each aggregate has a unique one
  

  ## Input/output of LocalOrdinalTransferFactory ##

  ### User parameters of LocalOrdinalTransferFactory ###
  Parameter | type | default | master.xml | validated | requested | description
  ----------|------|---------|:----------:|:---------:|:---------:|------------
  | TransferVec| Factory | null |   | * | (*) | Factory providing vector to be transered
  | Aggregates | Factory | null |   | * | (*) | Factory providing aggregates
  | CoarseMap  | Factory | null |   | * | (*) | Generating factory of the coarse map
  | write start|     int | -1   |   | * |     | first level at which coordinates should be written to file
  | write end  |     int | -1   |   | * |     | last level at which coordinates should be written to file

  The * in the @c master.xml column denotes that the parameter is defined in the @c master.xml file.<br>
  The * in the @c validated column means that the parameter is declared in the list of valid input parameters (see LocalOrdinalTransferFactory::GetValidParameters).<br>
  The * in the @c requested column states that the data is requested as input with all dependencies (see LocalOrdinalTransferFactory::DeclareInput).

  The LocalOrdinalTransferFact first checks whether there is already valid coarse TransferVec information
  available on the coarse level. If that is the case, we can skip the coarse TransferVec generation and just reuse
  the available information.
  Otherwise we try to build coarse grid TransverVec by using the information about the
  aggregates, the fine level TransferVec and the coarse map information.

  ### Variables provided by LocalOrdinalTransferFactory ###

  After LocalOrdinalTransferFactory::Build the following data is available (if requested)

  Parameter | generated by | description
  ----------|--------------|------------
  | TransferVec | LocalOrdinalTransferFactory   | coarse level transfervec
*/



  template<class LocalOrdinal = DefaultLocalOrdinal,
           class GlobalOrdinal = DefaultGlobalOrdinal,
           class Node = DefaultNode>
  class LocalOrdinalTransferFactory : public TwoLevelFactoryBase {
#undef MUELU_LOCALORDINALTRANSFERFACTORY_SHORT
#include "MueLu_UseShortNamesOrdinal.hpp"

  public:    
    //! @name Constructors/Destructors.
    //@{

    // Default constructor is distabled
    LocalOrdinalTransferFactory() = delete;

    /*! @brief Constructor.

       @param vectorName The name of the quantity to be restricted.
       @param restrictionName The name of the restriction Matrix.

       The operator associated with <tt>projectionName</tt> will be applied to the MultiVector associated with
       <tt>vectorName</tt>.
    */
    LocalOrdinalTransferFactory(const std::string & TransferVecName, const std::string & mode): TransferVecName_(TransferVecName) { 
      if(mode == "classical") useAggregatesMode_ = false;
      else useAggregatesMode_ = true;
    }

    //! Destructor.
    virtual ~LocalOrdinalTransferFactory() { }

    RCP<const ParameterList> GetValidParameterList() const;

    //@}

    //! @name Input
    //@{

    /*! @brief Specifies the data that this class needs, and the factories that generate that data.

        If the Build method of this class requires some data, but the generating factory is not specified in DeclareInput, then this class
        will fall back to the settings in FactoryManager.
    */
    void DeclareInput(Level &finelevel, Level &coarseLevel) const;

    //@}

    //! @name Build methods.
    //@{

    //! Build an object with this factory.
    void Build(Level & fineLevel, Level &coarseLevel) const;

    //@}

  private:

    void BuildAggregates(Level & fineLevel, Level &coarseLevel) const;

    void BuildFC(Level & fineLevel, Level &coarseLevel) const;
    
    //! Use aggregates mode (as opposed to FC mode)
    bool useAggregatesMode_;

    //! The name for the vector to be transfered.  This allows us to have multiple factories for different variables
    std::string TransferVecName_;

  }; // class LocalOrdinalTransferFactory

} // namespace MueLu

#define MUELU_LOCALORDINALTRANSFERFACTORY_SHORT
#endif // MUELU_LOCALORDINALTRANSFER_FACTORY_DECL_HPP
