/* Copyright (c) 2003-2004 Ecole centrale de Lyon
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "fdtd3d.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <limits.h>
#include <math.h>
#include <hdf5.h>
#include "file.h"
#include "utils.h"
#include "arrayh5.h"
#include "output.h"


SimulDesc *alloc_simuldesc()
{
  SimulDesc *blop;

  blop=myalloc(sizeof(SimulDesc));
  memset(blop,0,sizeof(SimulDesc));
  blop->space=myalloc(sizeof(varSpace));
  memset(blop->space,0,sizeof(varSpace));
  blop->simul=myalloc(sizeof(varSimul));
  memset(blop->simul,0,sizeof(varSimul));
  blop->inj=myalloc(sizeof(varInj));
  memset(blop->inj,0,sizeof(varInj));
  blop->output=myalloc(sizeof(varOutput));
  memset(blop->output,0,sizeof(varOutput));
  return blop;
}

void free_simuldesc(SimulDesc *blop)
{
  int i;
  
  free(blop->space->index);
  free(blop->space->absor);
  free(blop->space);
  free(blop->simul);
  free(blop->inj->injection);
  free(blop->inj);
  for(i=0;i<blop->output->nb_poncts;i++)
  {
    free(blop->output->poncts[i].valx);
    free(blop->output->poncts[i].valy);
    free(blop->output->poncts[i].valz);
  }
  free(blop->output->poncts);
  for(i=0;i<blop->output->nb_poynts;i++)
    free(blop->output->poynts[i].val);
  free(blop->output->poynts);
  free(blop->output->cartos);
  free(blop->output);
  free(blop);
}

int fill_varspace(varSpace *space, char *path)
{
  arrayh5 a;
  char *dataname;
  int errcode,i;

  dataname = split_dataname(path);
  if(dataname == NULL)
    dataname="index";
  errcode = arrayh5_read(&a,path,dataname,NULL,-1,0,0);
  if(errcode)
    {
      fprintf(stderr,"Problem while reading %s:%s : %s\n",path,dataname,arrayh5_read_strerror[errcode]);
      return -1;
    }
  if(a.rank!=3)
    {
      arrayh5_destroy(a);
      fprintf(stderr,"%s:%s is not a 3-dimensional array !\n",path,dataname);
      return -2;
    }
  space->nx=a.dims[0];
  space->ny=a.dims[1];
  space->nz=a.dims[2];

  for(i=0;i<3;i++)
      if(a.dims[i]<5)
	{
	  fprintf(stderr,"%s:%s: array too small in direction %c.\n",path,dataname,'x'+i);
	  arrayh5_destroy(a);
	  return -3;
	}

  space->index=myalloc(space->nx*space->ny*space->nz*sizeof(double));
  memcpy(space->index,a.data,space->nx*space->ny*space->nz*sizeof(double));
  arrayh5_destroy(a);

  space->min_index=space->index[0];
  for(i=0;i<space->nx*space->ny*space->nz;i++)
    if(space->index[i] < space->min_index)
      space->min_index=space->index[i];
  if(space->min_index < 1.)
  {
    fprintf(stderr,"Refraction index is lower than 1 !\n");
    return -4;
  }

  {
    H5E_auto_t h5errorfunc;
    void *h5clientdata;

    // Save and restore the error printing stuff,
    // but disable it for this call
    H5Eget_auto(&h5errorfunc, &h5clientdata);
    H5Eset_auto(NULL,NULL);
    dataname="absor";
    errcode = arrayh5_read(&a,path,dataname,NULL,-1,0,0);
    H5Eset_auto(h5errorfunc, h5clientdata);
  }
  if(errcode)
    return 0;

  printf("%s: absorption data detected.\n",path);
  
  if(a.rank!=3)
    {
      arrayh5_destroy(a);
      fprintf(stderr,"%s:%s is not a 3-dimensional array !\n",path,dataname);
      return -2;
    }
  if(space->nx != a.dims[0] || space->ny != a.dims[1] || space->nz != a.dims[2])
    {
      fprintf(stderr,"%s:%s: size does not match index array.\n",path,dataname);
      arrayh5_destroy(a);
      return -3;
    }

  space->absor=myalloc(space->nx*space->ny*space->nz*sizeof(double));
  memcpy(space->absor,a.data,space->nx*space->ny*space->nz*sizeof(double));
  arrayh5_destroy(a);

  return 0;
}

int extract_beginendinterv(char *gloo, int *begin, int *end, int *interv)
{
  int i;
  
  *begin=*interv=0;
  *end=INT_MAX;
  if(gloo[0]=='/')
    *interv=atoi(gloo+1);
  else if(gloo[0]=='-')
  {
    i=1;
    *end=atoi(gloo+1);
    while(gloo[i]>='0' && gloo[i]<='9') i++;
    if(gloo[i]=='/')
      *interv=atoi(gloo+i);
  }
  else if(gloo[0]<'0' || gloo[0]>'9')
    return -1;
  else
  {
    i=0;
    *begin=atoi(gloo);
    while(gloo[i]>='0' && gloo[i]<='9') i++;
    if(gloo[i]!='-')
    {
      *end=*begin+1;
      *interv=1;
    }
    else
    {
      i++;
      if(gloo[i]>='0' && gloo[i]<='9')
	*end=atoi(gloo+i);
      while(gloo[i]>='0' && gloo[i]<='9') i++;
      if(gloo[i]=='/')
	*interv=atoi(gloo+i+1);
    }
  }
  return 0;
}


int fill_varinj_file(SimulDesc *blop, char *path)
{
  char *contents;
  char *blip;
  varInj *inj=blop->inj;
  int it_max=blop->simul->it_max;
  int count=0;
   
  if(it_max<=0)
    return -2;

  contents=read_file(path);
  if(contents==NULL)
    return -1;

  free(inj->injection);
  inj->injection=myalloc(it_max*sizeof(double));
  
  blip=contents;
  while(count<it_max)
    {
      if(blip[0]=='\r' || blip[0]=='\n' || blip[0]==0)
	break;
      inj->injection[count]=atof(blip)*sin(2*M_PI*inj->basefreq*count*blop->simul->delta_t);
      count++;
      while(blip[0]!='\r' && blip[0]!='\n' && blip[0]!=0)
	blip++;
      if(blip[0]=='\r')
	blip++;
      if(blip[0]=='\n')
	blip++;
    }

  //if(count<it_max)
  //{
  //  inj->injection=myrealloc(inj->injection,count*sizeof(double));
  //}
  if(blip[0]!='\r' && blip[0]!='\n' && blip[0]!=0)
    {
      fprintf(stderr,"Warning: file %s contains more than %i values.\n",path,it_max);
    }

  //inj->inj_max=count;

  free(contents);

  return 0;
}
  
int fill_varinj_sin(SimulDesc *blop, double periods)
{
  varInj *inj=blop->inj;
  int i;
  int it_max=blop->simul->it_max;
  
  if(it_max<=0)
    return -2;

  free(inj->injection);
  inj->injection=myalloc(it_max*sizeof(double));
  //inj->inj_max=it_max;
  
  for(i=0;i<it_max;i++)
    inj->injection[i]=(.5*erf(5.*inj->basefreq*i*blop->simul->delta_t/periods-2.5)+.5)\
      		      * sin(2*M_PI*inj->basefreq*i*blop->simul->delta_t);

  return 0;
}

int fill_varinj_gauss(SimulDesc *blop, double relwidth)
{
  varInj *inj=blop->inj;
  int i;
  int it_max=blop->simul->it_max;
  
  if(it_max<=0)
    return -2;

  free(inj->injection);
  //inj->inj_max=ceil(320./(inj->basefreq*relwidth*blop->simul->delta_t));
  inj->injection=myalloc(it_max*sizeof(double));
  
  for(i=0;i<it_max;i++)
    inj->injection[i]=exp(-SQUARE(inj->basefreq*relwidth*i*blop->simul->delta_t/40.-4))\
      		      * sin(2*M_PI*inj->basefreq*i*blop->simul->delta_t);
  
  return 0;
}

int fill_varinj_defaults(SimulDesc *blop)
{
  int i;
  varInj *inj=blop->inj;
  int pospola;
  double constpola;
  double *tempinj;
  
  if(inj->injection==NULL)
  {
    int res;
    printf("Defaulting to sine wave with 10 periods of start time.\n");
    if((res=fill_varinj_sin(blop,10))<0)
      return res;
  }

  if(output_datfile_fft(blop->inj->injection,blop->simul->it_max,"p_inj",blop->simul->delta_t)<0)
    fprintf(stderr,"Warning: could not output injected polarization.\n");

  if( inj->x <= 0 || inj->x >= blop->space->nx-1 ||
      inj->y <= 0 || inj->y >= blop->space->ny-1 ||
      inj->z <= 0 || inj->z >= blop->space->nz-1 )
  {
    if(inj->x || inj->y || inj->z)
      fprintf(stderr,"Warning: injection position out of range.\n");
    printf("Centering injection position.\n");
    inj->x=blop->space->nx/2;
    inj->y=blop->space->ny/2;
    inj->z=blop->space->nz/2;
  }

  // For dipole injection, we convert all of that into the \Delta E that
  // will be injected. For each iteration:
  // \Delta E = -\Delta p/(\epsilon_0 \epsilon_r (\Delta x)^3 \Delta t)
  //
  // Let's be careful: because of that, the function is not idempotent. Maybe
  // a better solution is needed.
  pospola=(inj->x*blop->space->ny+inj->y)*blop->space->nz+inj->z;
  constpola=-1/(EPSILON_0*SQUARE(blop->space->index[pospola])*CUBE(blop->space->delta_x));
  tempinj=myalloc(blop->simul->it_max*sizeof(double));
  tempinj[0]=blop->inj->injection[0]*constpola;
  for(i=1;i<blop->simul->it_max;i++)
    tempinj[i]=(blop->inj->injection[i]-blop->inj->injection[i-1])*constpola;
  free(blop->inj->injection);
  blop->inj->injection=tempinj;
  
  if(inj->dir_x==0 && inj->dir_y==0 && inj->dir_z==0)
  {
    printf("Using default injection direction Z.\n");
    inj->dir_z=1;
  }
  
  return 0;
}

int validate_output (SimulDesc *blop)
{
  int i;

  if(blop->output->nb_cartos==0 && blop->output->nb_poncts==0 && blop->output->nb_poynts==0 && blop->output->total_loss==0)
    fprintf(stderr,"Warning: no output defined ! Simulation will be useless.\n");
  for(i=0;i<blop->output->nb_cartos;i++)
  {
    varOutputCarto *cur=&(blop->output->cartos[i]);
    
    if(cur->begin<0 || cur->begin>=blop->simul->it_max)
    {
      fprintf(stderr,"Incorrect begin iteration %i.\n",cur->begin);
      return -5;
    }
    
    if(cur->interv<0 || cur->interv>=blop->simul->it_max)
    {
      fprintf(stderr,"Incorrect interval %i.\n",cur->interv);
      return -5;
    }
    else if(cur->interv==0)
    {
      cur->interv=DEFAULT_INTERVAL;
      printf("No interval specified, using default value %i.\n",DEFAULT_INTERVAL);
    }
    
    if(cur->end < cur->begin+1)
    {
      fprintf(stderr,"Incorrect end iteration %i.\n",cur->end);
      return -5;
    }
    if(cur->end > blop->simul->it_max)
      cur->end=blop->simul->it_max;
  }
  
  for(i=0;i<blop->output->nb_poncts;i++)
  {
    varOutputPonct *cur=&(blop->output->poncts[i]);
    if( cur->x <= 0 || cur->x >= blop->space->nx-1 ||
        cur->y <= 0 || cur->y >= blop->space->ny-1 ||
        cur->z <= 0 || cur->z >= blop->space->nz-1 )
    {
      if(cur->x || cur->y || cur->z)
	fprintf(stderr,"Warning: output position out of range.\n");
      printf("Centering output position.\n");
      cur->x=blop->space->nx/2;
      cur->y=blop->space->ny/2;
      cur->z=blop->space->nz/2;
    }

    if(cur->valx==NULL)
      cur->valx=myalloc(blop->simul->it_max*sizeof(double));
    if(cur->valy==NULL)
      cur->valy=myalloc(blop->simul->it_max*sizeof(double));
    if(cur->valz==NULL)
      cur->valz=myalloc(blop->simul->it_max*sizeof(double));
  }

  if(blop->output->total_loss > 0)
  {
    // printf("total losses\n");
    blop->output->total_loss = -1;
    blop->output->nb_poynts += 6;
    blop->output->poynts=myrealloc(blop->output->poynts,blop->output->nb_poynts*sizeof(varOutputPoynt));
    if(blop->output->nb_poynts > 6)
      memmove(&(blop->output->poynts[6]),blop->output->poynts,(blop->output->nb_poynts-6)*sizeof(varOutputPoynt));
    memset(blop->output->poynts,0,6*sizeof(varOutputPoynt));
    for(i=0;i<6;i++)
    {
      varOutputPoynt *cur=&(blop->output->poynts[i]);
      cur->x2=blop->space->nx;
      cur->y2=blop->space->ny;
      cur->z2=blop->space->nz;
    }
    blop->output->poynts[0].x2=1;
    blop->output->poynts[1].x1=blop->space->nx-1;
    blop->output->poynts[2].y2=1;
    blop->output->poynts[3].y1=blop->space->ny-1;
    blop->output->poynts[4].z2=1;
    blop->output->poynts[5].z1=blop->space->nz-1;
    blop->output->poynts[0].dir=DIR_X;
    blop->output->poynts[1].dir=DIR_X;
    blop->output->poynts[2].dir=DIR_Y;
    blop->output->poynts[3].dir=DIR_Y;
    blop->output->poynts[4].dir=DIR_Z;
    blop->output->poynts[5].dir=DIR_Z;
  }

  for(i=0;i<blop->output->nb_poynts;i++)
  {
    varOutputPoynt *cur=&(blop->output->poynts[i]);
    if(cur->x1 < 0 || cur->x2 <= cur->x1 || blop->space->nx < cur->x2 ||
       cur->y1 < 0 || cur->y2 <= cur->y1 || blop->space->ny < cur->y2 ||
       cur->z1 < 0 || cur->z2 <= cur->z1 || blop->space->nz < cur->z2)
      {
	fprintf(stderr,"Output plane position out of range.\n");
	return -5;
      }

    if(cur->val==NULL)
      cur->val=myalloc(blop->simul->it_max*sizeof(double));
  }
  return 0;
}

int validate_simuldesc (SimulDesc *blop)
{
  double max_deltat,good_deltat;
  int i;
  
  if(blop->space->delta_x==0)
    {
      fprintf(stderr,"I need a space step !\n");
      return -1;
    }
  if(blop->simul->it_max==0)
    {
      fprintf(stderr,"I need an iteration number !\n");
      return -1;
    }
  if(blop->inj->basefreq==0)
    {
      fprintf(stderr,"I need a frequency/wavelength !\n");
      return -1;
    }
  if(blop->space->delta_x > .2*CELERITY/blop->inj->basefreq)
    {
      if(blop->space->delta_x > .5*CELERITY/blop->inj->basefreq)
	{
	  fprintf(stderr,"The half-wavelength should be shorter than the space step.\n");
	  return -2;
	}
      fprintf(stderr,"Warning: the wavelength is ridiculously low.\n");
    }
  max_deltat = blop->space->delta_x*blop->space->min_index/(CELERITY*SQRT_3);
  good_deltat = blop->space->delta_x*blop->space->min_index/(CELERITY*3);
  if(blop->simul->delta_t==0)
    {
      blop->simul->delta_t = good_deltat;
      printf("Setting iteration time to %e.\n",good_deltat);
    }
  else if(blop->simul->delta_t > max_deltat)
    {
      fprintf(stderr,"Error: iteration time to big, %le > %le.\n",blop->simul->delta_t,max_deltat);
      return -5;
    }
  if(blop->space->num_pml==0)
    {
      blop->space->num_pml=10;
      printf("Using %i-layer PMLs.\n",blop->space->num_pml);
    }
  if(blop->space->block_size==0)
    {
      blop->space->block_size=10;
      printf("Using default block size (%i).\n",blop->space->block_size);
    }
  if(blop->space->pml_refl==0)
    {
      blop->space->pml_refl=1.e-3;
      printf("PML theoretical reflection under normal incidence: %e\n",blop->space->pml_refl);
    }
  else if(blop->space->pml_refl>1)
    {
      fprintf(stderr,"Do you REALLY want a reflection greater than 1?\n");
      return -6;
    }
  blop->space->pml_decay=-CELERITY*log(blop->space->pml_refl)/(2*blop->space->num_pml*blop->space->delta_x);
  if(blop->space->pml_decay*blop->simul->delta_t > 1)
    {
      fprintf(stderr,"Such a reflection cannot be achieved with only %i PMLs.\n",blop->space->num_pml);
      return -6;
    }
  else if(blop->space->pml_decay*blop->simul->delta_t > 0.3)
    {
      printf("Warning: This is quite a small reflection for %i PMLs.\n",blop->space->num_pml);
    }

  if(blop->space->absor)
    for(i=0;i<blop->space->nx*blop->space->ny*blop->space->nz;i++)
      if(abs(blop->space->absor[i]*CELERITY*blop->simul->delta_t/blop->space->index[i])>.5)
	{
	  fprintf(stderr,"Absorbtion coefficient too large.\n");
	  return -7;
	}

  return 0;
}

SimulDesc *read_simul(char *path)
{
  char *contents;
  SimulDesc *blop;
  char *f1;
  char *f2;
  char *f3;
  int l,l1,l2,line;
  char *index_file=NULL;
  char *inj_file=NULL;
  int inj_type=INJ_NONE;
  double sin_periods=0, gauss_width=0;
  int result;
  
  contents=read_file(path);
  if(contents==NULL)
    return NULL;
  
  blop=alloc_simuldesc();
  
  f1=contents;
  line=1;
  while(*f1!='\0')
    {
      l=l1=l2=0;
      while(*f1==' ' || *f1=='\t') // Leading whitespace
	f1++;
      while(f1[l]!='\n' && f1[l]!='#' && f1[l]!='\0') // Go to the end of the line
	l++;
      while(f1[l1]!=' ' && f1[l1]!='=' && l1<l) // Get the variable name.
	l1++;
      f2=f1+l1;
      while(*f2==' ' || *f2=='=')
	{
	  if(*f2=='=')
	    l2++;
	  f2++;
	}
      f3=f1+l;
      if(*f3!='\0')
	{
	  while(*f3!='\n' && *f3!='\0') f3++;
	  if(*f3=='\n') f3++;
	}
      // printf("f1: %s, f2: %s, f3: %s.\n",f1,f2,f3);
      if(l1>1 && l2==1 && (l2=f1+l-f2)>0)
	{ // This is OK
	  f1[l1]=0;
	  f2[l2]=0;
	  // printf("%s\n",f2);
	  if(!strcasecmp(f1,"index_file"))
	    {
	      free(index_file);
	      index_file=mystrdup(f2);
	    }
	  else if(!strcasecmp(f1,"delta_x"))
	    blop->space->delta_x=atof(f2);
	  else if(!strcasecmp(f1,"num_pml"))
	    blop->space->num_pml=atoi(f2);
	  else if(!strcasecmp(f1,"block_size"))
	    blop->space->block_size=atoi(f2);
	  else if(!strcasecmp(f1,"pml_refl"))
	    blop->space->pml_refl=atof(f2);

	  else if(!strcasecmp(f1,"delta_t"))
	    blop->simul->delta_t=atof(f2);
	  else if(!strcasecmp(f1,"it_max"))
	    blop->simul->it_max=atoi(f2);
	  
	  else if(!strcasecmp(f1,"freq"))
	    blop->inj->basefreq=atof(f2);
	  else if(!strcasecmp(f1,"lambda"))
	    blop->inj->basefreq=CELERITY/atof(f2);
	  
	  else if(!strcasecmp(f1,"injection"))
	  {
	    double x;
	    if(sscanf(f2,"sinus %lf",&x)==1 || sscanf(f2,"sin %lf",&x)==1)
	    {
	      inj_type=INJ_SINUS;
	      sin_periods=x;
	    }
	    else if(sscanf(f2,"gauss %lf",&x)==1 || sscanf(f2,"gaussian %lf",&x)==1)
	    {
	      inj_type=INJ_GAUSS;
	      gauss_width=x;
	    }
	  }
	  else if(!strcasecmp(f1,"injection_pos"))
	  {
	    int x,y,z;
	    if(sscanf(f2,"%i,%i,%i",&x,&y,&z)<3)
	      fprintf(stderr,"Warning: %s: incorrect %s field line %i\n",path,f1,line);
	    else
	    {
	      blop->inj->x=x;
	      blop->inj->y=y;
	      blop->inj->z=z;
	    }
	  }
	  else if(!strcasecmp(f1,"injection_dir"))
	  {
	    double x,y,z;
	    if(sscanf(f2,"%lf,%lf,%lf",&x,&y,&z)<3)
	      fprintf(stderr,"Warning: %s: incorrect %s field line %i\n",path,f1,line);
	    else
	    {
	      blop->inj->dir_x=x;
	      blop->inj->dir_y=y;
	      blop->inj->dir_z=z;
	    }

	  }
	  else if((!strcasecmp(f1,"output_cartoE")) || (!strcasecmp(f1,"output_cartoH")))
	  {
	    int b,e,i;
	    varOutputCarto *current_carto;
	    
	    if(extract_beginendinterv(f2,&b,&e,&i))
	      fprintf(stderr,"Warning: %s: incorrect %s field line %i\n",path,f1,line);
	    else
	    {
	      blop->output->nb_cartos++;
	      blop->output->cartos=myrealloc(blop->output->cartos,blop->output->nb_cartos*sizeof(varOutputCarto));
	      current_carto = &(blop->output->cartos[blop->output->nb_cartos-1]);
	      switch(f1[12])
	      {
		case 'E':
		case 'e':
		  current_carto->field=FIELD_E;
		  break;
		case 'H':
		case 'h':
		  current_carto->field=FIELD_H;
		  break;
		default:
		  panic("GNE ? Unknown field. This shouldn't happen at all.");
	      }
	      current_carto->begin=b;
	      current_carto->end=e;
	      current_carto->interv=i;
	    }	    
	  }
	  else if((!strcasecmp(f1,"output_ponctE")) || (!strcasecmp(f1,"output_ponctH")))
	  {
	    int x,y,z;
	    varOutputPonct *current_ponct;
	    if(sscanf(f2,"%i,%i,%i",&x,&y,&z)<3)
	      fprintf(stderr,"Warning: %s: incorrect %s field line %i\n",path,f1,line);
	    else
	    {
	      blop->output->nb_poncts++;
	      blop->output->poncts=myrealloc(blop->output->poncts,blop->output->nb_poncts*sizeof(varOutputPonct));
	      current_ponct = &(blop->output->poncts[blop->output->nb_poncts-1]);
	      memset(current_ponct,0,sizeof(varOutputPonct));
	      switch(f1[12])
	      {
		case 'E':
		case 'e':
		  current_ponct->field=FIELD_E;
		  break;
		case 'H':
		case 'h':
		  current_ponct->field=FIELD_H;
		  break;
		default:
		  panic("GNE ? Unknown field. This shouldn't happen at all.");
	      }
	      current_ponct->x=x;
	      current_ponct->y=y;
	      current_ponct->z=z;
	    }
	  }
	  else if(!strcasecmp(f1,"output_poynting"))
	  {
	    int dir,x1,x2,y1,y2,z1,z2;
	    varOutputPoynt *current_poynt;
	    if(sscanf(f2,"%i,%i-%i,%i-%i",&x1,&y1,&y2,&z1,&z2)==5)
	      {
		dir=DIR_X;
		x2=x1+1;
	      }
	    else if(sscanf(f2,"%i-%i,%i,%i-%i",&x1,&x2,&y1,&z1,&z2)==5)
	      {
		dir=DIR_Y;
		y2=y1+1;
	      }
	    else if(sscanf(f2,"%i-%i,%i-%i,%i",&x1,&x2,&y1,&y2,&z1)==5)
	      {
		dir=DIR_Z;
		z2=z1+1;
	      }
	    else
	      {
		dir=-1;
		fprintf(stderr,"Warning: %s: incorrect %s field line %i\n",path,f1,line);
	      }
	    if(dir>=0)
	      {
		blop->output->nb_poynts++;
		blop->output->poynts=myrealloc(blop->output->poynts,blop->output->nb_poynts*sizeof(varOutputPoynt));
		current_poynt = &(blop->output->poynts[blop->output->nb_poynts-1]);
		memset(current_poynt,0,sizeof(varOutputPoynt));
		current_poynt->dir=dir;
		current_poynt->x1=x1;
		current_poynt->x2=x2;
		current_poynt->y1=y1;
		current_poynt->y2=y2;
		current_poynt->z1=z1;
		current_poynt->z2=z2;
	      }
	  }
	  else if(!strcasecmp(f1,"total_loss"))
	    blop->output->total_loss=atoi(f2);
	  else if(!strcasecmp(f1,"injection_file"))
	  {
	    inj_type=INJ_FILE;
	    free(inj_file);
	    inj_file=mystrdup(f2);
	  }
	  
	  else
	    fprintf(stderr,"Warning: %s: unknown field %s line %i\n",path,f1,line);
	}
      else if(l1 || l2)
	{
	  fprintf(stderr,"Warning: %s: parse error line %i\n",path,line);
	}
      line++;
      f1=f3;
    }

  free(contents);

  result=fill_varspace(blop->space,index_file);
  
  if(result>=0)
    result=validate_simuldesc(blop);
  free(index_file);
  
  if(result>=0)
  {
    switch(inj_type)
    {
      case INJ_FILE:
	result=fill_varinj_file(blop,inj_file);
	break;
      case INJ_SINUS:
	result=fill_varinj_sin(blop,sin_periods);
	break;
      case INJ_GAUSS:
	result=fill_varinj_gauss(blop,gauss_width);
	break;
    }
  }
  if(result>=0)
    result=fill_varinj_defaults(blop);
  free(inj_file);

  if(result>=0)
    result=validate_output(blop);
  
  if(result<0)
  {
    free_simuldesc(blop);
    return NULL;
  }
  return blop;
}
