/* ====================================================================
 * Copyright (c) 2003-2007, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "WcStatusLvi.h"
#include "ColorId.h"
#include "StatusId.h"
#include "DragDropMimeTypes.h"
#include "sublib/ColorStorage.h"
#include "util/Compare.h"
#include "svn/Path.h"
#include "svn/WcEntry.h"

// qt
#include <QtGui/QMimeSource>


WcStatusLvi::WcStatusLvi( Q3ListView* parent, const svn::WcStatusPtr status )
: super(parent), _status(status), _rename(false)
{
  init();
}

WcStatusLvi::WcStatusLvi( Q3ListViewItem* parent, const svn::WcStatusPtr status )
: super(parent), _status(status), _rename(false)
{
  init();
}

WcStatusLvi::~WcStatusLvi()
{
}

void WcStatusLvi::init()
{
  QString path = QString::fromUtf8( _status->getName().getStr() );
  path.remove( 0, _parentPath.length()+1 );

  setDropEnabled(_status->isDir());

  if( path.length() == 0 )
  {
    // ie. "."
    return;
  }

  setRenameEnabled( StatusColumnName, _status->isVersioned());
  setDragEnabled(_status->isVersioned());
}

void WcStatusLvi::startRename( int col )
{
  super::startRename(col);
}

void WcStatusLvi::okRename( int col )
{
  // tell the text method to return the new text and not the one
  // from the status.
  _rename = true;
  super::okRename(col);
  _rename = false;
}

void WcStatusLvi::cancelRename( int col )
{
  super::cancelRename(col);
}

#if 0
void WcStatusLvi::paintFocus( QPainter* p, const QColorGroup& cg, const QRect& r )
{
  super::paintFocus(p,cg,r);
}
#endif

void WcStatusLvi::paintCell( QPainter* p, const QColorGroup& cg, int column, int width, int alignment )
{
  QColorGroup g(cg);

  if( _status->getTextStatus() != svn::WcStatus_Normal )
  {
    g.setColor( QColorGroup::Background, getStatusColor(_status->getTextStatus()) );
  }
  else if(
    _status->getPropStatus() != svn::WcStatus_Normal &&
    _status->getPropStatus() != svn::WcStatus_None
    )
  {
    g.setColor( QColorGroup::Background, getStatusColor(_status->getPropStatus()) );
  }
#if 0
  else if( _status->getReposTextStatus() != svn::WcStatus_Normal )
  {
    g.setColor( QColorGroup::Base, getStatusColor(_status->getReposTextStatus()) );
  }
  else if( _status->getReposPropStatus() != svn::WcStatus_Normal )
  {
    g.setColor( QColorGroup::Base, getStatusColor(_status->getReposPropStatus()) );
  }
#endif
  //g.setColor( QColorGroup::Background, QColor(255,0,0) );
  //g.setColor( QColorGroup::Foreground, QColor(255,0,0) );
  //g.setColor( QColorGroup::Text, QColor(255,0,0) );
  //g.setColor( QColorGroup::Button, QColor(255,0,0) );
  //g.setColor( QColorGroup::ButtonText, QColor(255,0,0) );
  //g.setColor( QColorGroup::Light, QColor(255,0,0) );
  //g.setColor( QColorGroup::Midlight, QColor(255,0,0) );
  //g.setColor( QColorGroup::Dark, QColor(255,0,0) );
  //g.setColor( QColorGroup::Mid, QColor(255,0,0) );
  //g.setColor( QColorGroup::Shadow, QColor(255,0,0) );
  //g.setColor( QColorGroup::Highlight, QColor(255,0,0) );
  //g.setColor( QColorGroup::HighlightedText, QColor(255,0,0) );
  //g.setColor( QColorGroup::BrightText, QColor(255,0,0) );

  super::paintCell(p,g,column,width,alignment);
}

#if 0
void WcStatusLvi::paintBranches( QPainter* p, const QColorGroup& cg, int w, int y, int h, GUIStyle s )
{
  QColorGroup g(cg);
  if( _status->getTextStatus() != svn::WcStatus_Normal )
  {
    g.setColor( QColorGroup::Base, QColor(230,230,230) );
  }
  super::paintBranches(p,g,w,y,h,s);
}
#endif

bool WcStatusLvi::isInVisibleArea() const
{
  // the height correction is required for proper redraw when we scroll
  // the list.

  int top =       listView()->contentsY()     - height();
  int bot = top + listView()->visibleHeight() + height();
  int pos = itemPos();

  return !(pos < top || pos > bot);
}

QString WcStatusLvi::text( int column ) const
{
  // this heavily speeds up listview redrawing
#if 0
  if( ! isInVisibleArea() )
    return QString::null;
#endif

  switch( column )
  {
  case StatusColumnName:
    {
      if( _rename )
      {
        return super::text(StatusColumnName);
      }

      QString path = QString::fromUtf8( _status->getName().getStr() );
      path.remove( 0, _parentPath.length()+1 );

      if( path.length() == 0 )
      {
        return ".";
      }
      return path;
    }
  case StatusColumnTextStatus:
    {
      return getStatusCode( _status->getTextStatus() );
    }
  case StatusColumnPropStatus:
    {
      return getStatusCode( _status->getPropStatus() );
    }
  case StatusColumnWcLocked:
    {
      return _status->isLocked() ?
        getStatusCode(StatusWcLockLocked) : getStatusCode(StatusWcLockNone);
    }
  case StatusColumnCopied:
    {
      return _status->isCopied() ?
        getStatusCode(StatusHistory) : getStatusCode(StatusHistoryNone);
    }
  case StatusColumnSwitched:
    {
      return _status->isSwitched() ?
        getStatusCode(StatusSwitched) : getStatusCode(StatusSwitchedNo);
    }
  case StatusColumnRepLock:
    {
      return getLockStatusCode( _status->getLockStatus() );
    }
  case StatusColumnUpToDate:
    {
      bool t = _status->getReposTextStatus() != svn::WcStatus_None;
      bool p = _status->getReposTextStatus() != svn::WcStatus_None;

      return ( t || p ) ?
        getStatusCode(StatusUpToDateNo) : getStatusCode(StatusUpToDate);
    }
  case StatusColumnWcRev:
    {
      const svn::WcEntry* e = _status->getWcEntry();
      if( ! e )
      {
        return "";
      }
      else if( e->getRevnumber() < 0 )
      {
        return "?";
      }
      else if( _status->isCopied() )
      {
        return "-";
      }
      else
      {
        return QString().sprintf( "%ld", (unsigned long)e->getRevnumber() );
      }
    }
  case StatusColumnCmtRev:
    {
      const svn::WcEntry* e = _status->getWcEntry();
      if( ! e )
      {
        return "";
      }
      else if( e->getCmtRevnumber() < 0 )
      {
        return "?";
      }
      else
      {
        return QString().sprintf( "%ld", (unsigned long)e->getCmtRevnumber() );
      }
    }
  case StatusColumnCmtAuthor:
    {
      const svn::WcEntry* e = _status->getWcEntry();
      if( ! e )
      {
        return "";
      }
      else if( ! e->getCmtAuthor().getStr() )
      {
        return "?";
      }
      else
      {
        return QString::fromUtf8( e->getCmtAuthor().getStr() );
      }
    }
  case StatusColumnDeepStatus:
    {
      return super::text(StatusColumnDeepStatus);
    }
  case StatusColumnEmpty:
    {
      return "";
    }
  default:
    {
      return "invalid column";
    }
  }
}

QString WcStatusLvi::key( int column, bool ascending ) const
{
  switch( column )
  {
  case StatusColumnName:
  case StatusColumnTextStatus:
  case StatusColumnPropStatus:
  case StatusColumnWcLocked:
  case StatusColumnCopied:
  case StatusColumnSwitched:
  case StatusColumnRepLock:
  case StatusColumnUpToDate:
  case StatusColumnWcRev:
  case StatusColumnCmtRev:
  case StatusColumnCmtAuthor:
  default:
    {
      return text( column );
    }
  }
}

int WcStatusLvi::compare( Q3ListViewItem* i, int col, bool ascending ) const
{
  svn::WcStatusPtr status = dynamic_cast<WcStatusLvi*>(i)->getStatus();

  if( col == StatusColumnName )
  {
    return compare3( 
      _status->getName(), _status->isDir(), status->getName(), status->isDir() );
  }
  else // col != StatusColumnName
  {
    ///< \todo replace with direct compare on the status attributes
    return super::compare(i,col,ascending);
  }
}

void WcStatusLvi::setDeepStatus( const QString& status )
{
  setText( StatusColumnDeepStatus, status );
}

void WcStatusLvi::setParentPath( const QString& parentPath )
{
  _parentPath = parentPath;
}

const QString& WcStatusLvi::getParentPath() const
{
  return _parentPath;
}

svn::WcStatusPtr WcStatusLvi::getStatus() const
{
  return _status;
}

bool WcStatusLvi::acceptDrop( const QMimeSource* mime ) const
{
  return dropEnabled()
    && mime->provides( ScMimeTypeWcFilesItem );
}


#if 0
  QColorGroup g(cg);
  g.setColor( QColorGroup::Background, QColor(0,0,0) );
  g.setColor( QColorGroup::Foreground, QColor(0,0,0) );
  g.setColor( QColorGroup::Base, QColor(0,0,0) );
  g.setColor( QColorGroup::Text, QColor(0,0,0) );
  g.setColor( QColorGroup::Button, QColor(0,0,0) );
  g.setColor( QColorGroup::ButtonText, QColor(0,0,0) );
  g.setColor( QColorGroup::Light, QColor(0,0,0) );
  g.setColor( QColorGroup::Midlight, QColor(0,0,0) );
  g.setColor( QColorGroup::Dark, QColor(0,0,0) );
  g.setColor( QColorGroup::Mid, QColor(0,0,0) );
  g.setColor( QColorGroup::Shadow, QColor(0,0,0) );
  g.setColor( QColorGroup::Highlight, QColor(0,0,0) );
  g.setColor( QColorGroup::HighlightedText, QColor(0,0,0) );
  g.setColor( QColorGroup::BrightText, QColor(0,0,0) );
#endif

