/* factory.c - handle the SandUhr control center (TimerFactory CORBA object)
 *
 * Copyright (C) 2000, 2001  Jochen Voss.  */

static const  char  rcsid[] = "$Id: factory.c 5727 2004-06-01 22:11:03Z voss $";

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <time.h>

#include <gnome.h>

#include "interface.h"
#include "callbacks.h"
#include "support.h"

#include "sandcommon.h"


/**********************************************************************
 * Implement the CORBA servant
 */

static SandUhr_AlarmBeep
impl_SandUhr_TimerFactory_CreateAlarmBeep (struct factory *factory,
					   CORBA_unsigned_short Count,
					   CORBA_Environment *ev)
{
  struct alarm_beep *alarm;

  alarm = create_alarm_beep (factory->poa, Count, ev);
  return  PortableServer_POA_servant_to_reference (factory->poa, alarm, ev);
}

static SandUhr_AlarmSound
impl_SandUhr_TimerFactory_CreateAlarmSound (struct factory *factory,
					    CORBA_char *SoundFile,
					    CORBA_Environment *ev)
{
  struct alarm_sound *alarm;

  alarm = create_alarm_sound (factory->poa, SoundFile, ev);
  return  PortableServer_POA_servant_to_reference (factory->poa, alarm, ev);
}

static SandUhr_AlarmCommand
impl_SandUhr_TimerFactory_CreateAlarmCommand (struct factory *factory,
					      CORBA_char *CommandString,
					      CORBA_Environment *ev)
{
  struct alarm_command *alarm;

  alarm = create_alarm_command (factory->poa, CommandString, ev);
  return  PortableServer_POA_servant_to_reference (factory->poa, alarm, ev);
}

static SandUhr_Timer
impl_SandUhr_TimerFactory_CreateTimer (struct factory *factory,
				       CORBA_char *TimeSpec,
				       CORBA_char *Message,
				       CORBA_Environment *ev)
{
  struct timer *timer;

  timer = create_timer (factory, TimeSpec, Message, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)  return NULL;
  return  PortableServer_POA_servant_to_reference (timer->poa, timer, ev);
}

static SandUhr_TimerFactory_TimerVec *
impl_SandUhr_TimerFactory__get_Timers (struct factory *factory,
				       CORBA_Environment *ev)
{
  CORBA_unsigned_long  len, i;
  SandUhr_TimerFactory_TimerVec *retval;

  len = factory_timer_count (factory);
  retval = CORBA_sequence_SandUhr_Timer__alloc ();
  retval->_length = retval->_maximum = len;
  retval->_buffer = CORBA_sequence_SandUhr_Timer_allocbuf (len);
  CORBA_sequence_set_release (retval, TRUE);

  for (i=0; i<len; ++i) {
    struct timer *timer;

    timer = gtk_clist_get_row_data (GTK_CLIST (factory->clist), i);
    retval->_buffer[i]
      = PortableServer_POA_servant_to_reference (timer->poa, timer, ev);
  }
  return retval;
}

static void
impl_SandUhr_TimerFactory_ShowControl (struct factory *factory,
                                       CORBA_boolean Show,
                                       CORBA_Environment *ev)
{
  if (Show) {
    gtk_widget_show (factory->window);
  } else {
    gtk_widget_hide (factory->window);
  }
}

static CORBA_Object
impl_SandUhr_TimerFactory_createObject(struct factory *factory,
				       const CORBA_char *iid,
				       CORBA_Environment *ev)
{
  if (strcmp (iid, TIMER_IID) == 0) {
    struct timer *timer;

    timer = create_timer (factory, NULL, NULL, NULL);
    return  PortableServer_POA_servant_to_reference (timer->poa, timer, ev);
  } else {
    CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
			 ex_Bonobo_GenericFactory_CannotActivate,
			 NULL);
    return CORBA_OBJECT_NIL;
  }
}

static Bonobo_Unknown
impl_SandUhr_TimerFactory_queryInterface(struct factory *factory,
					 const CORBA_char *repoid,
					 CORBA_Environment *ev)
{
   Bonobo_Unknown retval;

   return retval;
}

static void
impl_SandUhr_TimerFactory_ref(struct factory *factory,
			      CORBA_Environment *ev)
{
}

static void
impl_SandUhr_TimerFactory_unref(struct factory *factory,
				CORBA_Environment *ev)
{
}

/**********************************************************************
 * epv structures
 */

static PortableServer_ServantBase__epv impl_SandUhr_TimerFactory_base_epv =
{
  NULL,				/* _private data */
  NULL,				/* finalize routine */
  NULL,				/* default_POA routine */
};

static POA_SandUhr_TimerFactory__epv impl_SandUhr_TimerFactory_epv =
{
  NULL,				/* _private */

  (gpointer) &impl_SandUhr_TimerFactory_CreateAlarmBeep,
  (gpointer) &impl_SandUhr_TimerFactory_CreateAlarmSound,
  (gpointer) &impl_SandUhr_TimerFactory_CreateAlarmCommand,

  (gpointer) &impl_SandUhr_TimerFactory_CreateTimer,
  (gpointer) &impl_SandUhr_TimerFactory__get_Timers,
  (gpointer) &impl_SandUhr_TimerFactory_ShowControl,
};

static POA_Bonobo_GenericFactory__epv
impl_SandUhr_TimerFactory_Bonobo_GenericFactory_epv = {
  NULL,				/* _private */
  (gpointer) &impl_SandUhr_TimerFactory_createObject,
};

static POA_Bonobo_Unknown__epv
impl_SandUhr_TimerFactory_Bonobo_Unknown_epv =
{
  NULL,				/* _private */
  (gpointer) &impl_SandUhr_TimerFactory_ref,
  (gpointer) &impl_SandUhr_TimerFactory_unref,
  (gpointer) &impl_SandUhr_TimerFactory_queryInterface,
};

static POA_SandUhr_TimerFactory__vepv impl_SandUhr_TimerFactory_vepv =
{
  &impl_SandUhr_TimerFactory_base_epv,
  &impl_SandUhr_TimerFactory_Bonobo_Unknown_epv,
  &impl_SandUhr_TimerFactory_Bonobo_GenericFactory_epv,
  &impl_SandUhr_TimerFactory_epv,
};

/**********************************************************************
 * Menus and callback functions
 */

static void
new_timer_cb (GtkMenuItem *menuitem, gpointer user_data)
{
  struct factory *factory = user_data;
  create_timer (factory, NULL, NULL, NULL);
}

static void
close_cb (GtkMenuItem *menuitem, gpointer user_data)
{
  struct factory *factory = user_data;
  gtk_widget_hide (factory->window);
}

static void
exit_cb (GtkMenuItem *menuitem, gpointer user_data)
{
  gtk_main_quit ();
}

static GnomeUIInfo timer_menu_uiinfo[] =
{
  GNOMEUIINFO_MENU_NEW_ITEM (N_("_New Timer"), N_("Create a new timer"),
			     new_timer_cb, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_CLOSE_ITEM (close_cb, NULL),
  GNOMEUIINFO_MENU_EXIT_ITEM (exit_cb, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo settings_menu_uiinfo[] =
{
  {
    GNOME_APP_UI_ITEM, N_("_Global Preferences"),
    N_("Change the default values for new timers"),
    (gpointer) on_preferences1_activate, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF,
    0, (GdkModifierType) 0, NULL
  },
  GNOMEUIINFO_END
};

static GnomeUIInfo help_menu_uiinfo[] =
{
  GNOMEUIINFO_HELP ("sanduhr"),
  GNOMEUIINFO_MENU_ABOUT_ITEM (on_about2_activate, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo menubar1_uiinfo[] =
{
  {
    GNOME_APP_UI_SUBTREE, N_("_Timer"),
    NULL,
    timer_menu_uiinfo, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL,
    0, (GdkModifierType) 0, NULL
  },
  GNOMEUIINFO_MENU_SETTINGS_TREE (settings_menu_uiinfo),
  GNOMEUIINFO_MENU_HELP_TREE (help_menu_uiinfo),
  GNOMEUIINFO_END
};

static void
close_timer_cb (GtkMenuItem *menuitem, gpointer user_data)
{
  struct timer *timer = user_data;
  delete_timer (timer);
}

static GnomeUIInfo menu3_uiinfo[] =
{
  GNOMEUIINFO_MENU_PROPERTIES_ITEM (on_properties1_activate, NULL),
  GNOMEUIINFO_MENU_CLOSE_ITEM (close_timer_cb, NULL),
  GNOMEUIINFO_END
};

static gboolean
window_delete_cb (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
  struct factory *factory = user_data;
  gtk_widget_hide (factory->window);
  return TRUE;
}

static gint
popup_cb (GtkCList *clist, GdkEventButton *event, gpointer data)
{
  struct factory *factory = data;
  int  row, column;
  
  if (event->button == 3
      && gtk_clist_get_selection_info (clist, event->x, event->y,
				       &row, &column)) {
    struct timer *timer;
    timer = gtk_clist_get_row_data (GTK_CLIST (factory->clist), row);
    gnome_popup_menu_do_popup (factory->popup_menu, NULL, NULL,
			       event, timer, factory->window);

    return  TRUE;
  }

  return  FALSE;
}

/**********************************************************************
 * external functions for the control center
 */

struct factory *
create_factory (PortableServer_POA poa, CORBA_Environment *ev)
{
  struct factory *factory;
  PortableServer_ObjectId *objid;
  GtkWidget *label, *scrolledwindow;

  factory = g_new (struct factory, 1);
  factory->servant._private = NULL;
  factory->servant.vepv = &impl_SandUhr_TimerFactory_vepv;
  factory->poa = poa;
  POA_SandUhr_TimerFactory__init ((PortableServer_Servant)factory, ev);

  factory->window = gnome_app_new ("SandUhr", _("SandUhr Control Center"));
  gtk_window_set_default_size (GTK_WINDOW (factory->window), 620, 300);
  gtk_window_set_policy (GTK_WINDOW (factory->window), TRUE, TRUE, FALSE);
  gtk_window_set_wmclass (GTK_WINDOW (factory->window), "sandUhr", "SandUhr");
  gnome_app_create_menus_with_data (GNOME_APP (factory->window),
				    menubar1_uiinfo, factory);
  
  gtk_signal_connect (GTK_OBJECT (factory->window),
		      "delete_event",
		      GTK_SIGNAL_FUNC (window_delete_cb),
		      factory);
  gtk_signal_connect (GTK_OBJECT (factory->window),
		      "show",
		      GTK_SIGNAL_FUNC (window_show_cb),
		      factory);
  gtk_signal_connect (GTK_OBJECT (factory->window),
		      "hide",
		      GTK_SIGNAL_FUNC (window_destroy_cb),
		      factory);

  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_ref (scrolledwindow);
  gnome_app_set_contents (GNOME_APP (factory->window), scrolledwindow);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  factory->clist = gtk_clist_new (2);
  gtk_container_add (GTK_CONTAINER (scrolledwindow), factory->clist);
  gtk_clist_set_column_width (GTK_CLIST (factory->clist), 0, 246);
  gtk_clist_set_column_width (GTK_CLIST (factory->clist), 1, 80);
  gtk_clist_column_titles_show (GTK_CLIST (factory->clist));

  label = gtk_label_new (_("Message"));
  gtk_clist_set_column_widget (GTK_CLIST (factory->clist), 0, label);
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  label = gtk_label_new (_("Alarm Time"));
  gtk_clist_set_column_widget (GTK_CLIST (factory->clist), 1, label);
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  /* Show all subwidgets, but leave the main window hidden.  */
  gtk_widget_show_all (scrolledwindow);

  factory->popup_menu = gnome_popup_menu_new (menu3_uiinfo);
  gtk_signal_connect (GTK_OBJECT (factory->clist),
		      "button_press_event",
		      GTK_SIGNAL_FUNC (popup_cb),
		      factory);
  
  objid = PortableServer_POA_activate_object (poa, factory, ev);
  CORBA_free (objid);

  return  factory;
}

void
delete_factory (struct factory *factory, CORBA_Environment *ev)
{
  PortableServer_ObjectId *objid;

  objid = PortableServer_POA_servant_to_id (factory->poa, factory, ev);
  PortableServer_POA_deactivate_object (factory->poa, objid, ev);
  CORBA_free (objid);

  POA_SandUhr_TimerFactory__fini ((PortableServer_Servant)factory, ev);

  g_free (factory);
}

void
factory_add_timer (struct factory *factory, struct timer *timer)
{
  char buffer [64];
  gchar *row[2];
  gint  n;

  {
    time_t target_time_abs = (int)(timer->target_time_abs + 0.5);
    strftime (buffer, 64, "%c", localtime (&target_time_abs));
  }
  row[0] = timer_get_message (timer);
  row[1] = buffer;
  n = gtk_clist_append (GTK_CLIST(factory->clist), row);
  g_free (row[0]);
  gtk_clist_set_row_data (GTK_CLIST(factory->clist), n, timer);
}

void
factory_remove_timer (struct factory *factory, struct timer *timer)
{
  gint  n;

  n = gtk_clist_find_row_from_data (GTK_CLIST(factory->clist), timer);
  gtk_clist_remove (GTK_CLIST(factory->clist), n);
}

void
factory_update_timer (struct factory *factory, struct timer *timer)
{
  gchar *message;
  char time_spec [64];
  gint  n;

  n = gtk_clist_find_row_from_data (GTK_CLIST(factory->clist), timer);

  message = timer_get_message (timer);
  gtk_clist_set_text (GTK_CLIST(factory->clist), n, 0, message);
  g_free (message);

  {
    time_t target_time_abs = (int)(timer->target_time_abs + 0.5);
    strftime (time_spec, 64, "%c", localtime (&target_time_abs));
  }
  gtk_clist_set_text (GTK_CLIST(factory->clist), n, 1, time_spec);
}

int
factory_timer_count (struct factory *factory)
/* Return the number of timers which are managed by FACTORY.  */
{
  return  GTK_CLIST(factory->clist)->rows;
}
