/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
    Copyright 2000-2025 the Rosegarden development team.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#define RG_MODULE_STRING "[CreateOrDeleteDeviceCommand]"
#define RG_NO_DEBUG_PRINT

#include "CreateOrDeleteDeviceCommand.h"

#include "misc/Debug.h"
#include "base/Device.h"
#include "gui/studio/DeviceManagerDialog.h"
#include "base/MidiDevice.h"
#include "gui/application/RosegardenMainWindow.h"
#include "sequencer/RosegardenSequencer.h"
#include "misc/Strings.h"  // For qstrtostr()...
#include "base/Studio.h"


namespace Rosegarden
{


CreateOrDeleteDeviceCommand::CreateOrDeleteDeviceCommand(
        Studio *studio,
        const std::string &name,
        Device::DeviceType type,
        MidiDevice::DeviceDirection direction,
        const std::string &connection,
        bool withData,
        const std::string &librarianName,
        const std::string &librarianEmail,
        MidiDevice::VariationType variationType,
        const BankList &bankList,
        const ProgramList &programList,
        const ControlList &controlList,
        const KeyMappingList &keyMappingList) :
    NamedCommand(getGlobalName(false)),
    m_studio(studio),
    m_deviceName(name),
    m_type(type),
    m_direction(direction),
    m_connection(connection),
    m_deviceId(Device::NO_DEVICE),
    m_baseInstrumentId(MidiInstrumentBase),
    m_deviceCreated(false),
    m_withData(withData),
    m_librarianName(librarianName),
    m_librarianEmail(librarianEmail),
    m_variationType(variationType),
    m_bankList(bankList),
    m_programList(programList),
    m_controlList(controlList),
    m_keyMappingList(keyMappingList)
{
}

CreateOrDeleteDeviceCommand::CreateOrDeleteDeviceCommand(Studio *studio,
                                                         DeviceId deviceId) :
    NamedCommand(getGlobalName(true)),
    m_studio(studio),
    m_deviceId(deviceId),
    m_baseInstrumentId(0),
    m_deviceCreated(true),  // We are doing delete.
    m_withData(false),
    m_variationType(MidiDevice::NoVariations)
{
    RG_DEBUG << "ctor" << m_deviceName << m_type << m_direction;
    Device *device = m_studio->getDevice(m_deviceId);

    if (!device) {
        RG_WARNING << "CreateOrDeleteDeviceCommand(): WARNING: No such device as " << m_deviceId;
        return;
    }

    // Save for undo.

    m_deviceName = device->getName();
    m_type = device->getType();
    m_direction = MidiDevice::Play;

    const MidiDevice *midiDevice = dynamic_cast<MidiDevice *>(device);
    if (midiDevice) {
        m_direction = midiDevice->getDirection();

        m_connection = qstrtostr(RosegardenSequencer::getInstance()->
                                 getConnection(midiDevice->getId()));
    }
}

void
CreateOrDeleteDeviceCommand::execute()
{
    RG_DEBUG << "execute" << m_deviceName << m_type << m_direction;
    // Create
    if (!m_deviceCreated) {

        // !DEVPUSH: Not ideal; we probably just want to add it to the
        //           studio (and then trigger a re-push) rather than add
        //           it twice to studio and sequencer

        // don't want to do this again on undo even if it fails -- only on redo
        m_deviceCreated = true;

        m_deviceId = m_studio->getSpareDeviceId(m_baseInstrumentId);

        bool success = RosegardenSequencer::getInstance()->addDevice(
                m_type,
                m_deviceId,
                m_baseInstrumentId,
                m_direction,
                m_deviceName);

        if (!success) {
            RG_WARNING << "execute(): WARNING: addDevice() failed";
            return;
        }

        //RG_DEBUG << "execute() - added device " << m_deviceId << " with base instrument id " << m_baseInstrumentId;

        // Make the connection.
        RosegardenSequencer::getInstance()->setConnection(
                m_deviceId, strtoqstr(m_connection));

        //RG_DEBUG << "execute() - reconnected device " << m_deviceId << " to " << m_connection;

        // Add to Studio.
        m_studio->addDevice(m_deviceName, m_deviceId,
                            m_baseInstrumentId, m_type);

        Device *device = m_studio->getDevice(m_deviceId);
        if (device) {
            MidiDevice *midiDevice = dynamic_cast<MidiDevice *>(device);
            if (midiDevice) {
                midiDevice->setDirection(m_direction);
                midiDevice->setUserConnection(m_connection);
                if (m_withData) {
                    midiDevice->setLibrarian(m_librarianName, m_librarianEmail);
                    midiDevice->setVariationType(m_variationType);
                    midiDevice->clearBankList();
                    for(const MidiBank& bank : m_bankList) {
                        midiDevice->addBank(bank);
                    }
                    midiDevice->clearProgramList();
                    for(const MidiProgram& program : m_programList) {
                        midiDevice->addProgram(program);
                    }
                    midiDevice->clearControlList();
                    for(const ControlParameter& control : m_controlList) {
                        midiDevice->addControlParameter(control, false);
                    }
                    midiDevice->clearKeyMappingList();
                    for(const MidiKeyMapping& keyMapping : m_keyMappingList) {
                        midiDevice->addKeyMapping(keyMapping);
                    }
                }
            }
        }

        // Update view automatically (without pressing refresh button).
        DeviceManagerDialog *deviceManagerDialog =
                RosegardenMainWindow::self()->getDeviceManager();
        if (deviceManagerDialog)
            deviceManagerDialog->slotResyncDevicesReceived();

    } else {  // Delete

        // Delete the device from the sequencer.
        RosegardenSequencer::getInstance()->removeDevice(m_deviceId);

        //RG_DEBUG << "execute() - removed device " << m_deviceId;

        // Delete the device from the Studio.
        m_studio->removeDevice(m_deviceId);

        m_deviceId = Device::NO_DEVICE;

        m_deviceCreated = false;
    }

    // ??? Instead of this kludge, we should be calling a Studio::hasChanged()
    //     which would then notify all observers (e.g. MIPP) who, in turn,
    //     would update themselves.
    RosegardenMainWindow::self()->uiUpdateKludge();
}


}
