///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
/*Prog:wathen
NAME: @code{wathen} - generate a N-by-N finite element matrix 
SYNOPSIS:
  @example
   wathen @var{nx} @var{ny} [-check|-scale]
  @end example
DESCRIPTION:       
  A = @code{wathen}(@var{nx},@var{ny})
  is a random N-by-N finite element matrix
  where N = 3*nx*ny + 2*nx + 2*ny + 1.
  
  A is precisely the mass matrix for a regular
  @var{nx}-by-@var{ny}
  grid of 8-node (serendipity, i.e. incomplete Q2) elements in the
  bidimensional space.
  
  A is symmetric positive definite for any (positive) values of 
  the density matrix, rho(@var{nx},@var{ny}), which is chosen randomly in this routine.
  
  In particular, if D=diag(diag(A)), then 
  0.25 <= eig(inv(D)*A) <= 4.5
  for any positive integers @var{nx} and @var{ny} and any densities rho(@var{nx},@var{ny}).
  
  This diagonally scaled matrix M=inv(D)*A is precisely
  returned by the command: 
  @example
	wathen @var{nx} @var{ny} -scale
  @end example

  Reference: A. J. Wathen, Realistic eigenvalue bounds for the Galerkin
  mass matrix , IMA J. Numer. Anal., 7 (1987), pp. 449-457.

EXAMPLE: 
  Matrix visualisation:
  @example
     	wathen 5 5 | hb2ps | ghostview -
    
     	wathen 5 5 -scale | hb2ps | ghostview -
    
     	wathen 20 20 | hb2ps | ghostview -
  @end example
  Check with
  @example
    	wathen 2 2 -check | octave
  @end example
  BEWARE - octave uses a dense format and this is a sparse matrix that
  quickly gets large !

OPTIONS:
  By default, generate the matrix on standard output in Harwell-Boeing format.
  The "wathen" command recognizes the following options:
  @table @code 
  @itemx -scale
	scale the matrix by its diagonal
  @itemx -check
	scale the matrix by its diagonal and generate matlab output format,
        usefull for the direct computation of eigenvalues.
  @end table

AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Pierre.Saramito@imag.fr
SEE ALSO:
    "makefish"(1), class "csr", class "dns"
DATE:
    20 february 1997

METHOD: @wathen
End:
*/
// wathen:
# include "rheolef/skit.h"
using namespace rheolef;
using namespace std;

Float e1[4][4] = {{ 6, -6,  2, -8},
                   {-6, 32, -6, 20},
		   { 2, -6,  6, -6},
		   {-8, 20, -6, 32}};

Float e2[4][4] = {{ 3, -8,  2, -6},
                   {-8, 16, -8, 20},
		   { 2, -8,  3, -8},
		   {-6, 20, -8, 16}};

csr<Float> wathen(unsigned int nx, unsigned int ny, bool scaled = 0)
{
    // e = [e1, e2; e2',e1]/45;
    Float e [8][8];
    for (unsigned int i = 0; i < 4; i++) {
      for (unsigned int j = 0; j < 4; j++) {
	e[i]  [j]   = e1[i][j]/45;
        e[i+4][j+4] = e1[i][j]/45;
        e[i]  [j+4] = e2[i][j]/45;
        e[i+4][j]   = e2[j][i]/45;
      }
    }
    unsigned int N = 3*nx*ny + 2*nx + 2*ny + 1;
    asr<Float> a(N,N);
    for (unsigned int j = 0; j < ny; j++) {
      for (unsigned int i = 0; i < nx; i++) {

        unsigned int nn [8];
	nn[0] = 3*(j+1)*nx + 2*(i+1) + 2*(j+1);
	nn[1] = nn[0] - 1;
	nn[2] = nn[1] - 1;
	nn[3] = (3*(j+1) - 1)*nx + 2*(j+1) + (i+1) - 2;
	nn[4] = 3*j*nx + 2*(i+1) + 2*(j+1) - 4;
	nn[5] = nn[4] + 1;
	nn[6] = nn[5] + 1;
	nn[7] = nn[3] + 1;

        // rand(): pseudo-random numbers 
	//         in the range from 0 to 2^15-1.
        const Float rho_0 = 100;
        const Float coef  = rho_0/(1<<15);
	const Float rho_ij = coef*rand();

	for (unsigned int krow = 0; krow < 8; krow++)
	  for (unsigned int kcol = 0; kcol < 8; kcol++)
            a.entry(nn[krow],nn[kcol])
	      += rho_ij*e[krow][kcol];
      }
    }
    csr<Float> A = csr<Float>(a);
    return A;
#ifdef TODO
    if (!scaled) 
	return A;
    else    // A := inv(diag(A))*A
	return A.left_div(diag<Float>(A));
#endif // TODO
}
int main(int argc, char *argv[])
{
    if (argc < 2 || atoi(argv[1]) < 1 || atoi(argv[2]) < 1) {
        
	cerr << argv[0] << ": usage: " 
	     << argv[0] << " nx ny [-check|-scale]\n";
        exit(1); 
    }
    unsigned int nx = atoi(argv[1]);
    unsigned int ny = atoi(argv[2]);
    
    if (argc < 4 || strcmp (argv[3], "-check") != 0) {

	// print hb output
	cout << wathen (nx, ny, argc >= 4);
    
    } else {
	
	// print matlab check code
	cout << ml << "a = " 
	     << wathen (nx, ny, argc >= 4) << ";" << endl
	     << "eigenvalues = eig(a)"     << endl
	     << "eig_min=min(eigenvalues)" << endl
	     << "eig_max=max(eigenvalues)" << endl;
    }
    return 0;
} 
// wathen:
