/* Copyright (C) 2009 J.F.Dockes
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#ifndef _fsindexer_h_included_
#define _fsindexer_h_included_

#include <list>
#ifndef NO_NAMESPACES
using std::list;
#endif

#include "indexer.h"
#include "fstreewalk.h"
#ifdef IDX_THREADS
#include "workqueue.h"
#endif // IDX_THREADS

class DbIxStatusUpdater;
class FIMissingStore;
struct stat;

#ifdef IDX_THREADS
class IndexingTask {
public:
    IndexingTask(const string& u, const string& p, const Rcl::Doc& d)
	:udi(u), parent_udi(p), doc(d)
    {}
    string udi;
    string parent_udi;
    Rcl::Doc doc;
};
extern void *FsIndexerIndexWorker(void*);
#endif // IDX_THREADS

/** Index selected parts of the file system
 
Tree indexing: we inherits FsTreeWalkerCB so that, the processone()
method is called by the file-system tree walk code for each file and
directory. We keep all state needed while indexing, and finally call
the methods to purge the db of stale entries and create the stemming
databases.

Single file(s) indexing: there are also calls to index or purge lists of files.
No database purging or stem db updating in this case.
*/
class FsIndexer : public FsTreeWalkerCB {
 public:
    /** Constructor does nothing but store parameters 
     *
     * @param cnf Configuration data
     * @param updfunc Status updater callback
     */
    FsIndexer(RclConfig *cnf, Rcl::Db *db, DbIxStatusUpdater *updfunc = 0);
    virtual ~FsIndexer();

    /** 
     * Top level file system tree index method for updating a given database.
     *
     * We create the temporary directory, open the database,
     * then call a file system walk for each top-level directory.
     */
    bool index();

    /** Index a list of files. No db cleaning or stemdb updating */
    bool indexFiles(list<string> &files, ConfIndexer::IxFlag f = 
		    ConfIndexer::IxFNone);

    /** Purge a list of files. */
    bool purgeFiles(list<string> &files);

    /**  Tree walker callback method */
    FsTreeWalker::Status 
    processone(const string &fn, const struct stat *, FsTreeWalker::CbFlag);

    /** Make signature for file up to date checks */
    static void makesig(const struct stat *stp, string& out);

 private:
    FsTreeWalker m_walker;
    RclConfig   *m_config;
    Rcl::Db     *m_db;
    TempDir     m_tmpdir;
    string       m_reason;
    DbIxStatusUpdater *m_updater;
    list<string> m_tdl;
    FIMissingStore *m_missing;

    // The configuration can set attribute fields to be inherited by
    // all files in a file system area. Ie: set "rclaptg = thunderbird"
    // inside ~/.thunderbird. The boolean is set at init to avoid
    // further wasteful processing if no local fields are set.
    bool         m_havelocalfields;
    map<string, string> m_localfields;

#ifdef IDX_THREADS
    friend void *FsIndexerIndexWorker(void*);
    WorkQueue<IndexingTask*> m_wqueue;
#endif // IDX_THREADS

    bool init();
    void localfieldsfromconf();
    void setlocalfields(Rcl::Doc& doc);
    string getDbDir() {return m_config->getDbDir();}
};

#endif /* _fsindexer_h_included_ */
