% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/datagrid.R
\name{datagrid}
\alias{datagrid}
\title{Data grids}
\usage{
datagrid(
  ...,
  model = NULL,
  newdata = NULL,
  by = NULL,
  grid_type = "mean_or_mode",
  response = FALSE,
  FUN = NULL,
  FUN_character = NULL,
  FUN_factor = NULL,
  FUN_logical = NULL,
  FUN_numeric = NULL,
  FUN_integer = NULL,
  FUN_binary = NULL,
  FUN_other = NULL
)
}
\arguments{
\item{...}{named arguments with vectors of values or functions for user-specified variables.
\itemize{
\item Functions are applied to the variable in the \code{model} dataset or \code{newdata}, and must return a vector of the appropriate type.
\item Character vectors are automatically transformed to factors if necessary.
\item The output will include all combinations of these variables (see Examples below.)
}}

\item{model}{Model object}

\item{newdata}{data.frame (one and only one of the \code{model} and \code{newdata} arguments can be used.)}

\item{by}{character vector with grouping variables within which \verb{FUN_*} functions are applied to create "sub-grids" with unspecified variables.}

\item{grid_type}{character. Determines the functions to apply to each variable. The defaults can be overridden by defining individual variables explicitly in \code{...}, or by supplying a function to one of the \verb{FUN_*} arguments.
\itemize{
\item "mean_or_mode": Character, factor, logical, and binary variables are set to their modes. Numeric, integer, and other variables are set to their means.
\item "balanced": Each unique level of character, factor, logical, and binary variables are preserved. Numeric, integer, and other variables are set to their means. Warning: When there are many variables and many levels per variable, a balanced grid can be very large. In those cases, it is better to use \code{grid_type="mean_or_mode"} and to specify the unique levels of a subset of named variables explicitly.
\item "dataframe": Similar to "mean_or_mode" but creates a data frame by binding columns element-wise rather than taking the cross-product. All explicitly specified vectors must have the same length (or length 1), and the result has as many rows as the longest vector. This differs from other grid types which use \code{expand.grid()} or \code{data.table::CJ()} to create all combinations.
\item "counterfactual": the entire dataset is duplicated for each combination of the variable values specified in \code{...}. Variables not explicitly supplied to \code{datagrid()} are set to their observed values in the original dataset.
}}

\item{response}{Logical should the response variable be included in the grid, even if it is not specified explicitly.}

\item{FUN}{a function to be applied to all variables in the grid. This is useful when you want to apply the same function to all variables, such as \code{mean} or \code{median}. If you specify \code{FUN}, it will override the \code{grid_type} defaults, but not other \verb{FUN_*} arguments below.}

\item{FUN_character}{the function to be applied to character variables.}

\item{FUN_factor}{the function to be applied to factor variables. This only applies if the variable in the original data is a factor. For variables converted to factor in a model-fitting formula, for example, \code{FUN_character} is used.}

\item{FUN_logical}{the function to be applied to logical variables.}

\item{FUN_numeric}{the function to be applied to numeric variables.}

\item{FUN_integer}{the function to be applied to integer-ish variables (including columns without decimal places).}

\item{FUN_binary}{the function to be applied to binary variables.}

\item{FUN_other}{the function to be applied to other variable types.}
}
\value{
A \code{data.frame} in which each row corresponds to one combination of the named
predictors supplied by the user via the \code{...} dots. Variables which are not
explicitly defined are held at their mean or mode.
}
\description{
Generate a data grid of user-specified values for use in the \code{newdata} argument of the \code{predictions()}, \code{comparisons()}, and \code{slopes()} functions. This is useful to define where in the predictor space we want to evaluate the quantities of interest. Ex: the predicted outcome or slope for a 37 year old college graduate.
}
\details{
If \code{datagrid} is used in a \code{predictions()}, \code{comparisons()}, or \code{slopes()} call as the
\code{newdata} argument, the model is automatically inserted in the \code{model} argument of \code{datagrid()}
call, and users do not need to specify either the \code{model} or \code{newdata} arguments. The same behavior will occur when the value supplied to \verb{newdata=} is a function call which starts with "datagrid". This is intended to allow users to create convenience shortcuts like:

\strong{Warning about hierarchical grouping variables:} When using the default \code{grid_type = "mean_or_mode"} with hierarchical models (such as mixed models with nested grouping factors), \code{datagrid()} may create invalid combinations of grouping variables. For example, if you have students nested within schools, or countries nested within regions, the modal values of each grouping variable may not correspond to valid nested relationships in the data. This can cause prediction errors. To avoid this issue, explicitly specify valid combinations of hierarchical grouping variables in the \code{datagrid()} call, or use \code{grid_type = "counterfactual"} to preserve the original data structure.

\preformatted{
mod <- lm(mpg ~ am + vs + factor(cyl) + hp, mtcars)
datagrid_bal <- function(...) datagrid(..., grid_type = "balanced")
predictions(model, newdata = datagrid_bal(cyl = 4))
}

If users supply a model, the data used to fit that model is retrieved using
the \code{insight::get_data} function.
}
\examples{
# The output only has 2 rows, and all the variables except `hp` are at their
# mean or mode.
datagrid(newdata = mtcars, hp = c(100, 110))

# We get the same result by feeding a model instead of a data.frame
mod <- lm(mpg ~ hp, mtcars)
datagrid(model = mod, hp = c(100, 110))

# Use in `marginaleffects` to compute "Typical Marginal Effects". When used
# in `slopes()` or `predictions()` we do not need to specify the
# `model` or `newdata` arguments.
slopes(mod, newdata = datagrid(hp = c(100, 110)))

# datagrid accepts functions
datagrid(hp = range, cyl = unique, newdata = mtcars)
comparisons(mod, newdata = datagrid(hp = fivenum))

# The full dataset is duplicated with each observation given counterfactual
# values of 100 and 110 for the `hp` variable. The original `mtcars` includes
# 32 rows, so the resulting dataset includes 64 rows.
dg <- datagrid(newdata = mtcars, hp = c(100, 110), grid_type = "counterfactual")
nrow(dg)

# We get the same result by feeding a model instead of a data.frame
mod <- lm(mpg ~ hp, mtcars)
dg <- datagrid(model = mod, hp = c(100, 110), grid_type = "counterfactual")
nrow(dg)

# Use `by` to hold variables at group-specific values
mod2 <- lm(mpg ~ hp + cyl, mtcars)
datagrid(model = mod2, hp = mean, by = "cyl")

# Use `FUN` to apply function to all variables
datagrid(model = mod2, FUN = median)

# Use `grid_type="dataframe"` for column-wise binding instead of cross-product
datagrid(model = mod2, hp = c(100, 200), cyl = c(4, 6), grid_type = "dataframe")
}
