% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ambientContribMaximum.R
\name{ambientContribMaximum}
\alias{ambientContribMaximum}
\alias{maximumAmbience}
\alias{ambientContribMaximum,ANY-method}
\alias{ambientContribMaximum,SummarizedExperiment-method}
\title{Ambient contribution by maximum scaling}
\usage{
maximumAmbience(...)

ambientContribMaximum(y, ...)

\S4method{ambientContribMaximum}{ANY}(
  y,
  ambient,
  threshold = 0.1,
  dispersion = 0,
  num.points = 100,
  num.iter = 5,
  mode = c("scale", "profile", "proportion"),
  BPPARAM = SerialParam()
)

\S4method{ambientContribMaximum}{SummarizedExperiment}(y, ..., assay.type = "counts")
}
\arguments{
\item{...}{For the generic, further arguments to pass to individual methods.

For the SummarizedExperiment method, further arguments to pass to the ANY method.

For \code{controlAmbience}, arguments to pass to \code{ambientContribMaximum}.}

\item{y}{A numeric matrix-like object containing counts, where each row represents a gene and each column represents a cluster of cells (see Caveats).

Alternatively, a \linkS4class{SummarizedExperiment} object containing such a matrix.

\code{y} can also be a numeric vector of counts; this is coerced into a one-column matrix.}

\item{ambient}{A numeric vector of length equal to \code{nrow(y)},
containing the proportions of transcripts for each gene in the ambient solution.
Alternatively, a matrix where each row corresponds to a row of \code{y}
and each column contains a specific ambient profile for the corresponding column of \code{y}.}

\item{threshold}{Numeric scalar specifying the p-value threshold to use, see Details.}

\item{dispersion}{Numeric scalar specifying the dispersion to use in the negative binomial model.
Defaults to zero, i.e., a Poisson model.}

\item{num.points}{Integer scalar specifying the number of points to use for the grid search.}

\item{num.iter}{Integer scalar specifying the number of iterations to use for the grid search.}

\item{mode}{String indicating the output to return, see Value.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object specifying how parallelization should be performed.}

\item{assay.type}{Integer or string specifying the assay containing the count matrix.}
}
\value{
If \code{mode="scale"},
a numeric vector is returned quantifying the maximum \dQuote{contribution} of the ambient solution to each column of \code{y}.
Scaling \code{ambient} by each entry yields the maximum ambient profile for the corresponding column of \code{y}.

If \code{mode="profile"}, a numeric matrix is returned containing the maximum ambient profile for each column of \code{y}.
This is computed by scaling as described above; if \code{ambient} is a matrix, each column is scaled by the corresponding entry of the scaling vector.

If \code{mode="proportion"}, a numeric matrix is returned containing the maximum proportion of counts in \code{y} that are attributable to ambient contamination.
This is computed by simply dividing the output of \code{mode="profile"} by \code{y} and capping all values at 1.
}
\description{
Compute the maximum contribution of the ambient solution to an expression profile for a group of droplets,
by scaling the ambient profile and testing for significant deviations in the count profile.
}
\details{
On occasion, it is useful to estimate the maximum possible contribution of the ambient solution to a count profile.
This represents the most pessimistic explanation of a particular expression pattern
and can be used to identify and discard suspect genes or clusters prior to downstream analyses.

This function implements the following algorithm:
\enumerate{ 
\item We compute the mean ambient contribution for each gene by scaling \code{ambient} by some factor.
\code{ambient} itself is usually derived by summing counts across barcodes with low total counts,
see the output of \code{\link{emptyDrops}} for an example.
\item We compute a p-value for each gene based on the probability of observing a count equal to or below that in \code{y}, using the lower tail of a negative binomial (or Poisson) distribution with mean set to the ambient contribution.
The per-gene null hypothesis is that the expected count in \code{y} is equal to the sum of the scaled ambient proportion and some (non-negative) contribution from actual intra-cellular transcripts.
\item We combine p-values across all genes using Simes' method.
This represents the evidence against the joint null hypothesis (that all of the per-gene nulls are true).
\item We find the largest scaling factor that fails to reject this joint null at the specified \code{threshold}.
If \code{sum(ambient)} is equal to unity, this scaling factor can be interpreted as the maximum number of transcript molecules contributed to \code{y} by the ambient solution.
}

The process of going from a scaling factor to a combined p-value has no clean analytical solution,
so we use an iterative grid search to identify to largest possible scaling factor at a decent resolution.
\code{num.points} and \code{num.iter} control the resolution of the grid search,
and generally do not need to be changed.

\code{maximumAmbience} is soft-deprecated; use \code{ambientContribMaximum} instead.
}
\section{Caveats}{

The above algorithm is rather \emph{ad hoc} and offers little in the way of theoretical guarantees.
The p-value is used as a score rather than providing any meaningful error control.
Empirically, increasing \code{threshold} will return a higher scaling factor by making the estimation more robust to drop-outs in \code{y}, at the cost of increasing the risk of over-estimation of the ambient contribution.

Our abuse of the p-value machinery means that the reported scaling often exceeds the actual contribution, especially at low counts where the reduced power fails to penalize overly large scaling factors.
Hence, the function works best when \code{y} contains aggregated counts for one or more groups of droplets with the same expected expression profile, e.g., clusters of related cells.
Higher counts provide more power to detect deviations, hopefully leading to a more accurate estimate of the scaling factor.
(On a practical note, this function is rather slow so it is more feasible to calculate on cluster-level profiles rather than per cell.)

Note that this function returns the \emph{maximum possible} contribution of the ambient solution to \code{y}, not the actual contribution.
In the most extreme case, if the ambient profile is similar to the expectation of \code{y} (e.g., due to sequencing a relatively homogeneous cell population), the maximum possible contribution of the ambient solution would be 100\% of \code{y}, and subtraction would yield an empty count vector!
}

\examples{
# Making up some data for, e.g., a single cluster.
ambient <- c(runif(900, 0, 0.1), runif(100))
y <- rpois(1000, ambient * 100)
y[1:100] <- y[1:100] + rpois(100, 20) # actual biology.

# Estimating the maximum possible scaling factor:
scaling <- ambientContribMaximum(y, ambient)
scaling

# Estimating the maximum contribution to 'y' by 'ambient'.
contribution <- ambientContribMaximum(y, ambient, mode="profile")
DataFrame(ambient=drop(contribution), total=y)

}
\seealso{
\code{\link{ambientProfileEmpty}} and \code{\link{ambientProfileBimodal}}, to estimate the ambient profile.

\code{\link{ambientContribSparse}} and \code{\link{ambientContribNegative}}, for other methods to estimate the ambient contribution.

\code{\link{emptyDrops}}, which uses the ambient profile to call cells.

\code{\link{ambientProfileEmpty}} or \code{\link{ambientProfileBimodal}}, to obtain an estimate to use in \code{ambient}.

\code{\link{ambientContribNegative}} or \code{\link{ambientContribSparse}}, for other methods of estimating the contribution.
}
\author{
Aaron Lun
}
