///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/ParameterUI.h>
#include <core/gui/properties/PropertiesEditor.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(ParameterUI, RefMaker)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(ReferenceParameterUI, ParameterUI)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(PropertyParameterUI, ParameterUI)

DEFINE_FLAGS_REFERENCE_FIELD(ParameterUI, RefTarget, "EditObject", PROPERTY_FIELD_NO_UNDO | PROPERTY_FIELD_NO_CHANGE_MESSAGE, _editObject)

DEFINE_FLAGS_REFERENCE_FIELD(ReferenceParameterUI, RefTarget, "ParameterObject", PROPERTY_FIELD_NO_UNDO | PROPERTY_FIELD_NO_CHANGE_MESSAGE, _parameterObject)

///////////////////////////////////// ParameterUI /////////////////////////////////////////

/******************************************************************************
* The constructor.
******************************************************************************/
ParameterUI::ParameterUI(PropertiesEditor* parentEditor) : _enabled(true)
{
	INIT_PROPERTY_FIELD(ParameterUI, _editObject);
	
	CHECK_OBJECT_POINTER(parentEditor);
	setParent(parentEditor);
	if(parentEditor->editObject())
		setEditObject(parentEditor->editObject());
	
	// Connect to the contentsReplaced() signal of the editor to synchronize the
	// parameter UI's edit object with the editor's edit object.
	connect(parentEditor, SIGNAL(contentsReplaced(RefTarget*)), this, SLOT(setEditObject(RefTarget*)));
}

///////////////////////////////////// ReferenceParameterUI /////////////////////////////////////////

/******************************************************************************
* The constructor.
******************************************************************************/
ReferenceParameterUI::ReferenceParameterUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField) : 
	ParameterUI(parentEditor), _refField(refField)
{
	INIT_PROPERTY_FIELD(ReferenceParameterUI, _parameterObject);
	OVITO_ASSERT_MSG(refField.isReferenceField(), "ReferenceParameterUI", "This controller UI cannot be used with non-animatable property fields.");
	OVITO_ASSERT_MSG(refField.isVector() == false, "ReferenceParameterUI", "This controller UI cannot be used with vector reference fields.");
}

/******************************************************************************
* This method is called when parameter object has been assigned to the reference field of the editable object 
* this parameter UI is bound to. It is also called when the editable object itself has
* been replaced in the editor. 
******************************************************************************/
void ReferenceParameterUI::resetUI() 
{
	if(editObject()) {
		CHECK_OBJECT_POINTER(editObject());
		OVITO_ASSERT(editObject() == NULL || editObject()->pluginClassDescriptor()->isKindOf(referenceField().definingClass()));

		// Bind this parameter UI to the parameter object of the new edited object.
		_parameterObject = editObject()->getReferenceField(referenceField());
	}
	else _parameterObject = NULL;
	
	ParameterUI::resetUI(); 
}

/******************************************************************************
* This method is called when a reference target changes.
******************************************************************************/
bool ReferenceParameterUI::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(source == editObject() && msg->type() == REFERENCE_FIELD_CHANGED && referenceField() == ((ReferenceFieldMessage*)msg)->field()) {
		// The parameter value object stored in the reference field of the edited object
		// has been replaced by another one, so update our own reference to the parameter value object.
		if(editObject()->getReferenceField(referenceField()) != parameterObject())
			resetUI();
	}
	else if(source == parameterObject() && msg->type() == REFTARGET_CHANGED) {
		// The parameter value object has changed -> update value shown in UI.
		updateUI();
	}
	return ParameterUI::onRefTargetMessage(source, msg);
}

///////////////////////////////////// PropertyParameterUI /////////////////////////////////////////

/******************************************************************************
* Constructor for a Qt property.
******************************************************************************/
PropertyParameterUI::PropertyParameterUI(PropertiesEditor* parentEditor, const char* propertyName) : 
	ParameterUI(parentEditor), _propertyName(propertyName), _propField(NULL)
{
	OVITO_ASSERT(propertyName != NULL);
}

/******************************************************************************
* Constructor for a PropertyField property.
******************************************************************************/
PropertyParameterUI::PropertyParameterUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& propField) : 
	ParameterUI(parentEditor), _propertyName(NULL), _propField(&propField)
{
	OVITO_ASSERT_MSG(!propField.isReferenceField(), "PropertyParameterUI", "This property UI can only be used for non-animatable properties.");
}

/******************************************************************************
* This method is called when a reference target changes.
******************************************************************************/
bool PropertyParameterUI::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(source == editObject() && msg->type() == REFTARGET_CHANGED) {
		// The edited object has changed -> update value shown in UI.
		updateUI();
	}
	return ParameterUI::onRefTargetMessage(source, msg);
}

};

