[comment {-*- Tcl -*- nx::Object manpage}]

[include version.inc]
[manpage_begin nx::Object 3 [vset VERSION]]

[comment {For the time being, we do not render keywords & terms; and
the corresponding reverse index}] 
[proc keywords args {}]
[proc term v {return $v}]

[keywords baseclass]
[keywords NX]
[keywords "mixin class"]
[keywords "re-classification"]
[keywords "submethod"]
[keywords "method ensemble"]
[keywords "linearization"]

[vset SCOPE "object"]
[vset CMD "obj"]
[vset MODIFIER "object"]

[copyright {2014-19 Stefan Sobernig <stefan.sobernig@wu.ac.at>, Gustaf Neumann <gustaf.neumann@wu.ac.at>; available under the Creative Commons Attribution 3.0 Austria license (CC BY 3.0 AT).}]

[moddesc "NX API"]
[titledesc {API reference of the base class in the NX object system}]


[description]
[para]

[cmd nx::Object] is the [term "base class"] of the [term NX] object system. All
objects defined in [term NX] are (direct or indirect) instances of this
[term "base class"]. The methods provided by the [cmd nx::Object]
[term "base class"] are available to all objects and to all classes defined in
NX.

[example {
 +---------+
 | ::nx::* |
 +---------+--------------------------------------Y
 |                                                |
 |  +---------+     instance of     +----------+  |
 |  |         |<....................|          |  |
 |  |  Class  |                     |  Object  |  |
 |  |         |....................>|          |  |
 |  +----+----+     subclass of     +-----+----+  |
 |       ^                           ^    ^       |
instance.|...........................|....|......./
      of |                           |    |
   +-----+-----+    subclass of      |    | instance
   |           |.....................|    | of
   |   /cls/   |    (by default)          |
   |           |                          |
   +-----------+                          |
         ^                                |
instance |.............(xor)..............|
      of |         +-----------+          |
         |.........|           |..........|
                   |   /obj/   |
                   |           |
                   +-----------+
}]

[term NX] allows for creating and for using objects (e.g. [emph obj]) which are
instantiated from the [term "base class"] [cmd nx::Object]
directly. Typical use cases are singletons and anonymous, inline
objects. In such use cases, [term NX] does not require creating an
intermediate application class (e.g. [emph cls]), which specializes the [term "base class"]
[cmd nx::Object] by default, beforehand.

[para]

Objects (e.g. [emph obj]) which are creating by instantiating a
previously defined application class (e.g. [emph cls]) are indirect
instances of [cmd nx::Object].

[para]
Direct instances of [cmd nx::Object] can be created as follows:

[list_begin definitions]

[call [cmd nx::Object] [method create] [arg obj] [opt "[option -object-mixins] [arg mixinSpec]"] [opt "[option -class] [arg newClassName]"] [opt "[option -object-filters] [arg filterSpec]"] [opt [arg initBlock]]]

[para] 

To create a direct instance of [cmd nx::Object] having an explicit name
[arg obj], use [method create] on [cmd nx::Object]. Note that
[method create] is defined by [cmd nx::Class] and is available to [cmd nx::Object] being
an instance of [cmd nx::Class]. This way, singleton objects can be
created, for example.

[call [cmd nx::Object] [method new] [opt "[option -object-mixins] [arg mixinSpec]"] [opt "[option -class] [arg newClassName]"] [opt "[option -object-filters] [arg filterSpec]"] [opt [arg initBlock]]]

To create a direct instance of [cmd nx::Object] having an
automatically assigned, implicit object name, use [method new] on [cmd nx::Object]. Note
that [method new] is defined by [cmd nx::Class] and is available to
[cmd nx::Object] being an instance of [cmd nx::Class]. Using [method new] allows
for creating anonymous, inline objects, for example.

[list_end]

The configuration options for direct and indirect instances of [cmd nx::Object], which
can be passed when calling [method create] and [method new], are
documented in the subsequent section.

[section {Configuration Options for Instances of nx::Object}]

[para] Configuration options can be used for configuring objects
during their creation by passing the options as non-positional
arguments into calls of [method new] and [method create] (see [cmd nx::Class]). An
existing object can be queried for its current configuration using
[method cget] and it can be re-configured using [method configure]. Legal
configuration options are:

[list_begin options]

[opt_def -class [opt [arg className]]]

Retrieves the current class of the object or sets the object's class to [arg className], if provided.

[opt_def -object-filters [opt [arg filterMethods]]]

Retrieves the list of currently active per-object filter methods or sets
a list of per-object filter methods, if [arg filterMethods] is
provided.

[opt_def -object-mixins [opt [arg mixinSpecs]]]

If [arg mixinSpecs] is not specified, retrieves the list of currently
active per-object mixin specifications. If [arg mixinSpecs] is
specified, sets a list of per-object mixin specifications to become
active. [term "mixin class"]es are returned or set in terms of a list
of [term "mixin specification"]s.

[list_end]

[section {Methods for Instances of nx::Object}]

[list_begin commands]

[cmd_def alias]
[list_begin definitions]

[include alias.man.inc]

[list_end]

[cmd_def cget]

[list_begin definitions]

[call [arg obj] [method cget] [arg configurationOption]] 

The method is used to obtain the current value of [arg configurationOption] for
[arg obj]. The configuration options
available for querying through [method cget] are determined by the
configurable properties defined by the class hierarchy of [arg obj]. The
queryable configuration options for [arg obj] can be
obtained by calling [method "info lookup syntax configure"]. The [arg configurationOption] can
be set and modified using [method configure].

[example_begin]
% nx::Object create obj
::obj
% ::obj info lookup syntax configure
?-object-mixins /mixinreg .../? ?-class /class/? ?-object-filters /filterreg .../? ?/__initblock/?
% ::obj cget -class
::nx::Object
[example_end]

[list_end]

[cmd_def configure]

[list_begin definitions]

[call [arg obj] [method configure] [opt "[arg configurationOption] [arg value] ..."]]

This method sets configuration options on an object. The configuration
options available for setting on [arg obj] are determined by the
configurable properties defined by the class hierarchy of [arg obj]. The
settable configuration options for [arg obj] can be
obtained by calling [method "info lookup syntax configure"]. Furthermore, [method configure] is
also called during object construction. Under object construction, it receives
the arguments passed into calls of [method create] and [method new]. Options
set using [method configure] can be retrieved using [method cget].

[example_begin]
% nx::Class create Foo {:property x}
::Foo
% Foo create f1 -x 101
::f1
% f1 cget -x
101
% f1 configure -x 200
% f1 cget -x
200
[example_end]

[list_end]


[cmd_def contains]

[list_begin definitions]

[call [arg obj] [method contains] [opt "-withnew [arg trueFalse]"] [opt "-object [arg objectName]"] [opt "-class [arg className]"] [arg cmds]]

This method acts as a builder for nested object structures. Object
and class construction statements passed to this method as its last
argument [arg cmds] are evaluated in a way so that the receiver object
[arg obj] becomes the parent of the newly constructed objects and
classes. This is realized by setting explicitly the namespace for
constructing relatively named objects. Fully qualified object names in
[arg cmds] evade the nesting.

[para]

[option "-withnew"] requests the automatic rescoping of
objects created using [method new] so that they become nested into the
receiver object [arg obj], rather than being created in the default
namespace for autonamed objects (i.e., ::nsf). If turned off,
autonamed objects do not become children of [arg obj].

[para]

The parent object [arg objectName] to be used instead of [arg obj] can be specified
using [option "-object"]. If this explicitly set parent
object does not exist prior to calling [method contains], it will be
created on the fly as a direct instance of [cmd nx::Object]. Alternatively,
using [option "-class"], a class [arg className] other
than [cmd nx::Object] for the on-the-fly creation of [arg objectName]
can be provided.

[example {
% nx::Class create Window {
  :contains {
    #
    # Become children of Window, implicitly
    #
    nx::Class create Header; # Window::Header
    nx::Object create Panel; # Window::Panel
  }
  #
  # Explicitly declared a child of Window using [self]
  #
  nx::Class create [self]::Slider; # Window::Slider
  #
  # Fully-qualified objects do not become nested
  #
  nx::Class create ::Door; # ::Door
}
::Window
% ::Window info children
::Window::Panel ::Window::Header ::Window::Slider
}]

[list_end]

[cmd_def copy]

[list_begin definitions]

[call [arg obj] [method copy] [opt [arg newObjectName]]]

Creates a full and deep copy of a source object [arg obj]. The
object's copy features all structural and behavioral properties of the
source object, including object variables, per-object methods, nested
objects, slot objects, namespaces, filters, mixins, and traces. The
copy can be named explicitly, if [arg newObjectName] is provided,
or it is named automatically (in the spirit of [method new] of
[cmd nx::Class]).

[list_end]

[comment {
  [cmd_def defaultmethod]
  
  [list_begin definitions]
  
  [call [arg obj] [cmd defaultmethod]]

  This method is called implicitly when the object command is invoked
  without an argument. By default, the [cmd defaultmethod]
  implementation returns the fully qualified object name.

  [example_begin]
  % Object create ::foo
  ::foo
  % ::foo defaultmethod
  ::foo
  % ::foo
  ::foo
  [example_end]
  
  [list_end]
}]

[cmd_def delete]

[list_begin definitions]

[include delete.man.inc]

[list_end]


[cmd_def destroy]

[list_begin definitions]

[call [arg obj] [method destroy]]

This method allows for explicitly destructing an object [arg obj],
potentially prior to [arg obj] being destroyed by the object system
(e.g. during the shutdown of the object system upon calling [cmd exit]):

[example {[nx::Object new] destroy}]

By providing a custom implementation of [method destroy], the
destruction procedure of [arg obj] can be customized. Typically, once
the application-specific destruction logic has completed, a custom
[method destroy] will trigger the actual, physical object destruction
via [cmd next].

[example {
% [nx::Object create obj {
  :public method destroy {} {
    puts "destroying [self]"
    next; # physical destruction
  }
}] destroy
destroying ::obj
}]

A customized object-destruction scheme can be made shared between the instances
of a class, by defining the custom [method destroy] for an
application class:

[example_begin]
% nx::Class create Foo {
    :method destroy {} {
      puts "destroying [lb]self[rb]"
      next; # physical destruction
    }
}
::Foo
% Foo create f1
::f1
% f1 destroy
destroying ::f1
[example_end]

Physical destruction is performed by clearing the in-memory object
storage of [arg obj]. This is achieved by passing [arg obj] into a
call to [method dealloc] provided by [cmd nx::Class]. A near, scripted
equivalent to the C-implemented [method destroy] provided by [cmd nx::Object] would look
as follows:

[example {
% Object method destroy {} {
  [:info class] dealloc [self]
}
}]

Note, however, that [method destroy] is protected against
application-level redefinition. Trying to evaluate the above script snippet yields: 

[example {
refuse to overwrite protected method 'destroy'; derive e.g. a subclass!
}]

A custom [method destroy] must be provided as a refinement in a
subclass of [cmd nx::Object] or in a [term "mixin class"].

[list_end]

[cmd_def eval]

[list_begin definitions]

[call [arg obj] [method eval] [arg arg] [opt "[arg arg] ..."]]

Evaluates a special Tcl script for the scope of [arg obj] in the style
of Tcl's [cmd eval]. There are, however, notable differences to the
standard [cmd eval]: In this script, the colon-prefix notation is available to
dispatch to methods and to access variables of [arg obj]. Script-local
variables, which are thrown away once the evaluation of the script has
completed, can be defined to store intermediate results.

[example { 
  % nx::Object create obj {
    :object property {bar 1}
    :public object method foo {x} { return $x }
  }
  ::obj
  % ::obj eval {
    set y [:foo ${:bar}]
  }
  1
}]

[list_end]

[cmd_def filters]

[list_begin definitions]

[include filter.man.inc]

[list_end]

[cmd_def forward]

[list_begin definitions]

[include forward.man.inc]

[list_end]

[cmd_def info]

[list_begin definitions]

[call [arg obj] [method "info baseclass"]]

Returns the [term "base class"] of [arg obj]. The [term "base class"]
is the class from which all [term NX] objects are instantiated
directly or indirectly (typically [cmd nx::Object]).

[call [arg obj] [method "info children"] [opt "[option -type] [arg className]"] [opt [arg pattern]]]

Retrieves the list of nested (or aggregated) objects of [arg obj]. The
resulting list contains the fully qualified names of the nested
objects. If [option -type] is set, only nested objects which are
direct or indirect instances of class [arg className] are
returned. Using [arg pattern], only nested objects whose names match
[arg pattern] are returned. The [arg pattern] string can contain
special matching characters (see [cmd "string match"]). This method
allows for introspecting on [method contains].

[call [arg obj] [method "info class"]]

Returns the fully qualified name of the current [cmd nx::Class] of
[arg obj]. In case of [term "re-classification"] (see [method configure]), the
returned class will be different from the [cmd nx::Class] from which [arg obj] was
originally instantiated using [method create] or [method new].

[call [arg obj] [method "info has"] [opt "[method mixin] | [method namespace] | [method type]"] [opt "[arg arg] ..."]]

[list_begin definitions]
[def "[arg obj] [method "info has mixin"] [arg className]"] 

Verifies whether [arg obj] has a given [cmd nx::Class] [arg className] registered as a [term "mixin class"] (returns: [const true]) or not (returns: [const false]).

[def "[arg obj] [method "info has namespace"]"]

Checks whether the object has a companion Tcl namespace (returns:
[const true]) or not (returns: [const false]). The namespace could
have been created using, for example, [method "object require namespace"]. 

[comment {Note that the results do not necessarily correspond
  to those yielded by '''[namespace exists /obj/]'''.}]

[def "[arg obj] [method "info has type"] [arg className]"]

Tests whether the [cmd nx::Class] [arg className] is a type of the
object (returns: [const true]) or not (returns: [const false]). That
is, the method checks whether the object is a direct instance of [arg className] or
an indirect instance of one of the superclasses of [arg className].

[list_end]

[call [arg obj] [method "info lookup"] [arg submethod] [opt "[arg arg] ..."]]

A collection of submethods to retrieve structural features (e.g.
configuration options, [term "slot object"]s) and behavioral features
(e.g. methods, [term "filter"]s) available for [arg obj] from the
perspective of a client to [arg obj]. Features provided by [arg obj]
itself and by the classes in its current linearization list are
considered.

[list_begin definitions]

[def "[arg obj] [method "info lookup filter"] [arg name]"]

Returns the [term "method handle"] for the [term "filter"] method [arg name], if
currently registered. If there is no filter [arg name] registered, an
empty string is returned.

[def "[arg obj] [method "info lookup filters"] [opt [option -guards]] [opt [arg namePattern]]"]

Returns the [term "method handle"]s of all filters which are active on [arg obj]. By
turning on the [term switch] [option -guards], the corresponding guard
expressions, if any, are also reported for each filter as a three-element list: [arg methodHandle] -guard [arg guardExpr]. The returned filters can be limited to
those whose names match [arg namePattern] (see [cmd "string match"]).

[def "[arg obj] [method "info lookup method"] [arg name]"]

Returns the [term "method handle"] for a method [arg name] if a
so-named method can be invoked on [arg obj]. If there is no method
[arg name], an empty string is returned.

[def "[arg obj] [method "info lookup methods"] [opt [arg namePattern]]"]

Returns the names of all methods (including aliases and forwarders)
which can be invoked on [arg obj]. The returned methods can be limited
to those whose names match [arg namePattern] (see [cmd "string match"]).

[def "[arg obj] [method "info lookup mixins"] [opt [option -guards]] [opt [arg namePattern]]"]

Returns the object names of all [term "mixin class"]es which are
currently active on [arg obj]. By turning on the [term switch]
[option -guards], the corresponding guard expressions, if any, are also reported as a
three-element list for each [term "mixin class"]: [arg className]
-guard [arg guardExpr]. The returned [term "mixin class"]es can be
limited to those whose names match [arg namePattern] (see [cmd "string match"]).

[def "[arg obj] [method "info lookup parameters"] [arg methodName] [opt [arg namePattern]]"]

Returns the [term "parameter specification"] of the method [arg methodName] callable
on [arg obj] as a list of parameter names and type specifications. The
resulting [term "parameter specification"] can be limited to those
parameters whose names match [arg namePattern] (see [cmd "string match"]).

[def "[arg obj] [method "info lookup slots"] [opt "[option "-type"] [arg className]"] [opt "[option "-source"] all | application | system"] [opt [arg namePattern]]"]

Returns the command names of all [term "slot object"]s responsible for
managing properties, variables, and relations of [arg obj]. The
returned [term "slot object"]s can be limited according to any or a
combination of the following criteria: First, [term "slot object"]s
can be filtered based on their command names matching [arg namePattern]
(see [cmd "string match"]). Second, [option "-type"]
allows one to select [term "slot object"]s which are instantiated from
a subclass [arg className] of [cmd nx::Slot] (default: [cmd nx::Slot])
. Third, [option -source] restricts [term "slot object"]s returned
according to their provenance in either the NX [emph system] classes
or the [emph application] classes present in the linearization list of
[arg obj] (default: [emph all]).

[para]

To extract details of each [term "slot object"], use the [method info]
submethods available for each [term "slot object"].

[def "[arg obj] [method "info lookup syntax"] [arg methodName] [opt [arg namePattern]]"]

Returns the method parameters of the method [arg methodName] callable
on [arg obj] as a concrete-syntax description to be used in
human-understandable messages (e.g., errors or warnings, documentation
strings). The result can be limited to those parameters matching the
[arg namePattern] (see [cmd "string match"]).

[def "[arg obj] [method "info lookup variables"]"]

Returns the command names of all [term "slot object"]s responsible for
managing properties and variables of [arg obj], if provided by [arg obj]
or the classes in the linearization list of [arg obj].

[para]

This is equivalent to calling: [arg obj] [method "info lookup slots"] -type ::nx::VariableSlot -source all [opt [arg namePattern]].

[para]

To extract details of each [term "slot object"], use the [method info]
submethods available for each [term "slot object"].


[list_end]

[call [arg obj] [method {info name}]]

Returns the unqualified name of an object, i.e., the object name
without any namespace qualifiers.

[include info.man.inc]

[call [arg obj] [method {info parent}]]

Returns the fully qualified name of the parent object of [arg obj], if
any. If there is no parent object, the name of the Tcl
namespace containing [arg obj] (e.g. "::") will be reported.

[call [arg obj] [method {info precedence}] [opt [option -intrinsic]] [opt [arg pattern]]]

Lists the classes from which [arg obj] inherits structural (e.g.
properties) and behavioral features (e.g. methods) and methods, in
order of the [term linearization] scheme in [term NX]. By setting the
[term switch] [option -intrinsic], only classes which participate in
superclass/subclass relationships (i.e., intrinsic classes) are
returned. If a [arg pattern] is provided only classes whose
names match [arg pattern] are returned. The [arg pattern] string can
contain special matching characters (see [cmd "string match"]).

[call [arg obj] [method {info variable}] [arg option] [arg handle]]

Retrieves selected details about a variable represented by the given
[arg handle]. A [arg handle] can be obtained by querying [arg obj] using 
[method "info [vset SCOPE] variables"] and [method "info lookup variables"].

Valid values for [arg option] are:
[list_begin itemized]
[item] [const name] returns the variable name.
[item] [const parameter] returns a canonical parameter specification
eligible to (re-)define the given variable (e.g. using [method "[vset SCOPE] variable"]) in a new context.
[item] [const definition] returns a canonical representation of the definition command used to create the variable in its current configuration.
[list_end]

[call [arg obj] [method {info vars}] [opt [arg pattern]]]

Yields a list of Tcl variable names created and defined for the scope of
[arg obj], i.e., object variables. The list can be limited to object variables whose names
match [arg pattern]. The [arg pattern] string can contain special
matching characters (see [cmd "string match"]).

[list_end]

[comment { 
[cmd_def init]
  
[list_begin definitions]

[call [arg obj] [cmd init] [opt "[arg arg] ..."]]

The method [method init] is called during the object construction
process. It is invoked as the last step during object construction
(i.e. after method [method configure]) to provide the fully initialized 
state of the object. Note that the definition of an [method init] 
method must contain an empty parameter specification, 
since [method init] is always called with an empty argument list.

[example_begin]
% nx::Class create Foo {:property x}
% Foo method init {} {set :y [lb]expr {${:x} + 1}[rb]}
% Foo public method bar {} {return ${:y}}
% Foo create f1 -x 101
% f1 cget -x
101
% f1 bar
102
[example_end]
[list_end]
}]


[cmd_def method]

[list_begin definitions]

[include method.man.inc]

[list_end]

[cmd_def move]

[list_begin definitions]

[call [arg obj] [method move] [arg newObjectName]]

Effectively renames an object. First, the source object [arg obj] is
cloned into a target object [arg newObjectName] using [method copy]. Second,
the source object [arg obj] is destroyed by invoking [method destroy]. 
[method move] is also called internally when [cmd rename] is
performed for a Tcl command representing an object.

[list_end]

[cmd_def mixins]
[list_begin definitions]

[include mixin.man.inc]

[list_end]

[cmd_def __object_configureparameter]

[list_begin definitions]

[def "[arg obj] [method "__object_configureparameter"]"]

Computes and returns the configuration options available for [arg obj], to be consumed as method-parameter specification by [method configure].

[list_end]

[cmd_def property]

[list_begin definitions]

[call [arg obj] [method "object property"] [opt "[option -accessor] [const public] | [const protected] | [const private]"] [opt "[option -class] [arg className]"] [opt "[option -configurable] [arg trueFalse]"] [opt [option -incremental]] [opt [option -nocomplain]] [opt "[option -trace] [const set] | [const get] | [const default]"] [arg spec] [opt [arg initBlock]]]

[include property.man.inc]

[para]

By default, the [term property] will ascertain that no (potentially)
pre-existing and equally named object variable will be overwritten
when defining the property. In case of a conflict, an error exception
is thrown:

[example {
% Object create obj { set :x 1 }
::obj
% ::obj object property {x 2}
object ::obj has already an instance variable named 'x'
}]

If the [term switch] [option -nocomplain] is on, this check is omitted (continuing the above example):

[example {
% ::obj object property -nocomplain {x 2}
% ::obj eval {set :x}
2
}]

[list_end]

[cmd_def require]

[list_begin definitions]

[call [arg obj] [method "require namespace"]]

Create a Tcl namespace named after the object [arg obj]. All object
variables become available as namespace variables.

[include require.man.inc]

[list_end]

[cmd_def unknown]

[list_begin definitions]

[call [arg obj] [method unknown] [arg unknownMethodName] [opt "[arg arg] ..."]]

This method is called implicitly whenever an unknown method is invoked.

[arg unknownMethodName] indicates the unresolvable method name,
followed by the remainder of the original argument vector as a number
of [arg arg] of the calling method invocation.

[list_end]

[cmd_def uplevel]

[list_begin definitions]

[call [arg obj] [method uplevel] [opt [arg level]] [arg arg1] [opt "[arg arg2] ..."]]

Evaluate a script or a command at a different stack-frame
level. The command behaves in essence like Tcl's [cmd uplevel], but can
be used to achieve identical results when filters or mixins are
registered.

[list_begin itemized]

[item] If the [arg level] specifier is omitted, [method uplevel]
will skip any auxiliary frames added to the stack by active
[term "filter"]s and [term mixin]s. The resulting stack-frame level
corresponds to the [term callinglevel] as indicated by [cmd nx::current]. In
this case method [method uplevel] can be used to evaluate the command
in the next enclosing procedure call, i.e., a frame corresponding to a
proc, method, or apply call, while skipping frames of filters and
mixins.

[item] If the [arg level] specifier is provided (relative, or
absolute), [method "uplevel"] will execute the command in the
stack-frame level. In such cases, method [method uplevel]
behaves like Tcl's [method uplevel] command.


[list_end]

[example { 
  % nx::Object create ::obj
  ::obj  
  % ::obj public object method foo {varName} {
      :uplevel set $varName 1; return
  }
  ::obj::foo
  % namespace eval ::ns1 {
       ::obj foo BAR
  }
  % namespace eval ::ns1 {
      info exists BAR
  }
  1 
}]

[list_end]

Note, in the example above, [method "uplevel"] is guaranteed to
resolve to the calling context of [method "foo"] ([term "ns1"]) despite
mixins and filters being (potentially) registered on [cmd "obj"].

[cmd_def upvar]

[list_begin definitions]

[call [arg obj] [method upvar] [opt [arg level]] [arg otherVar1] [arg localVar1] [opt "[arg otherVar2] [arg localVar2] ..."]]

Links one or more local variables to variables defined in other
scopes (namespaces, objects, call frames).
The command behaves in essence like Tcl's [cmd upvar], but can
be used to achieve identical results when filters or mixins are
registered.

[list_begin itemized]

[item] If the [arg level] specifier is omitted, [method upvar]
will skip any auxiliary frames added to the stack by active [term "filter"]s and [term "mixin"]s. The
resulting stack-frame level corresponds to the [term "callinglevel"]
as indicated by [cmd nx::current]. Therefore, method [method upvar] gives
access to the next enclosing procedure call, i.e., a frame
corresponding to a proc, method, or apply call, while skipping
frames of filters and mixins.

[item] If the [arg level] specifier is provided (relative, or
absolute), [method "upvar"] will link into the requested
stack-frame level. In these cases, method [method upvar]
behaves like Tcl's  [method upvar] command.

[list_end]

[example { 
  % nx::Object create ::obj
  ::obj  
  % ::obj public object method foo {varName} {
      :upvar $varName x; set x 1; return
  }
  ::obj::foo
  % namespace eval ::ns1 {
       ::obj foo BAR
  }
  % namespace eval ::ns1 {
      info exists BAR
  }
  1 
}]

[list_end]

Note, in the example above, [method "upvar"] is guaranteed to
resolve to the calling context of [method "foo"] ([term "ns1"]) despite
mixins and filters being (potentially) registered on [cmd "obj"].

[cmd_def variable]

[list_begin definitions]

[comment {::nx::Object variable ?-accessor /value/? ?-incremental? ?-class /value/? ?-configurable /boolean/? ?-initblock /value/? ?-nocomplain? /spec/ ?/defaultValue/?}]

[call [arg obj] [method "object variable"] [opt "[option -accessor] [const public] | [const protected] | [const private]"] [opt [option -incremental]] [opt "[option -class] [arg className]"] [opt "[option -configurable] [arg trueFalse]"] [opt "[option -initblock] [arg script]"] [opt "[option -trace] [const set] | [const get] | [const default]"] [opt [option -nocomplain]] [arg spec] [opt [arg defaultValue]]]

[include variable.man.inc]

[para]

By default, the [term variable] will ascertain that a
pre-existing and equally named object variable will not be overwritten
when defining the [term variable]. In case of a conflict, an error exception
is thrown:

[example {
% Object create obj { set :x 1 }
::obj
% ::obj object variable x 2
object ::obj has already an instance variable named 'x'
}]

If the [term switch] [option -nocomplain] is on, this check is omitted (continuing the above example):

[example {
% ::obj object variable -nocomplain x 2
% ::obj eval {set :x}
2
}]

[list_end]

[list_end]

[comment { COMMANDS list }]

[comment {
  [cmd nx::Object] provides a set of default implementations for
  internally called methods, which are called primarily during the 
  creation or destruction of NX objects.
  
  Application developers can provide custom implementations of these
  methods by providing tailored implementations for these methods in application
  classes (i.e., subclasses of [cmd nx::Object]). An adequate method
  implementation must comply with the method signature interfaces described below.
}]

[section {Object Self-Reference}]

Objects are naturally recursive, with methods of an object [const ::obj]
frequently invoking other methods in the same object [const ::obj] and
accessing [const ::obj]'s object variables. To represent these
self-references effectively in method bodies, and depending on the
usage scenario, NX offers two alternative notations for self-references: one based on a
special-purpose syntax token ("colon prefix"), the other based on the
command [cmd nx::current].  

[para] 

Both, the colon-prefix notation and
[cmd nx::current], may be used only in method bodies and scripts
passed to [method eval]. If they appear anywhere else, an error will be
reported.

There are three main use cases for self-references:

[list_begin enumerated]
[enum] As a [emph placeholder] for the currently active object, [cmd nx::current]
       can be used to retrieve the object name.
[enum] Reading and writing [emph "object variables"] directly (i.e. without getter/setter methods in place) require the use
of variable names carrying the prefix [const :] ("colon-prefix
notation"). Internally, colon-prefixed variable names are processed
using Tcl's variable resolvers. Alternatively, one can provide for getter/setter methods for object variables (see [method property] and [method variable]).

[enum] [emph {Self-referential method calls}] can be defined via
prefixing ([const :]) the method names or, alternatively, via [cmd nx::current]. Internally,
colon-prefixed method names are processed using Tcl's command
resolvers. The colon-prefix notation is recommended, also because it
has a (slight) performance advantage over [cmd nx::current] which
requires two rather than one command evaluation per method call.
[list_end]

See the following listing for some examples corresponding to use cases 1--3:


[example {
  Object create ::obj {
    # 1) print name of currently active object ('::obj')
    puts [current];
    # 2) object variables
    set :x 1; :object variable y 2;	
    :public object method print {} {
      # 2.a) method-local variable
      set z 3;
      # 2.b) variable substitution using '$' and ':'
      puts ${:x}-${:y}-$z;
      # 2.c) reading variables using 'set'
      puts [set :x]-[set :y]-[set z];
      # 2.d) writing variables using 'set', 'incr', ...
      set :x 1; incr :y; 			
    }
    :public object method show {} {
      # 3.a) self-referential method call using ':'	
      :print;
      # 3.b) self-referential method call using 'nx::current'
      [current] print;
      # 3.c) self-referential method call using 'nx::current object'
      [current object] print;              
    }
    :show
  }
}]


[manpage_end]
