<!--

@license Apache-2.0

Copyright (c) 2019 The Stdlib Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

-->

# iterator2arrayview

> Fill an array-like object view with values returned from an iterator.

<!-- Section to include introductory text. Make sure to keep an empty line after the intro `section` element and another before the `/section` close. -->

<section class="intro">

</section>

<!-- /.intro -->

<!-- Package usage documentation. -->

<section class="usage">

## Usage

```javascript
var iterator2arrayview = require( '@stdlib/iter/to-array-view' );
```

#### iterator2arrayview( iterator, dest\[, begin\[, end]]\[, mapFcn\[, thisArg]] )

Fills an array-like `object` view with values returned from an `iterator`.

```javascript
var Uint8Array = require( '@stdlib/array/uint8' );
var array2iterator = require( '@stdlib/array/to-iterator' );

var iter = array2iterator( [ 1, 2, 3, 4 ] );

var arr = iterator2arrayview( iter, new Uint8Array( 10 ) );
// returns <Uint8Array>[ 1, 2, 3, 4, 0, 0, 0, 0, 0, 0 ]
```

By default, the function begins filling from the first element of a provided array-like `object`. To specify an alternative starting index, provide a `begin` argument (zero-based and inclusive).

```javascript
var Uint8Array = require( '@stdlib/array/uint8' );
var array2iterator = require( '@stdlib/array/to-iterator' );

var iter = array2iterator( [ 1, 2, 3, 4 ] );

var arr = iterator2arrayview( iter, new Uint8Array( 10 ), 3 );
// returns <Uint8Array>[ 0, 0, 0, 1, 2, 3, 4, 0, 0, 0 ]
```

If `begin` is less than `0`, the starting index is resolved relative to the last element of the provided array-like `object`. For example, the following achieves the same behavior as the previous example

```javascript
var Uint8Array = require( '@stdlib/array/uint8' );
var array2iterator = require( '@stdlib/array/to-iterator' );

var iter = array2iterator( [ 1, 2, 3, 4 ] );

var arr = iterator2arrayview( iter, new Uint8Array( 10 ), -7 );
// returns <Uint8Array>[ 0, 0, 0, 1, 2, 3, 4, 0, 0, 0 ]
```

By default, the function assumes that a view extends through the last element of a provided array-like `object`. To specify an alternative last view element, provide an `end` argument (zero-based and non-inclusive).

```javascript
var Uint8Array = require( '@stdlib/array/uint8' );
var array2iterator = require( '@stdlib/array/to-iterator' );

var iter = array2iterator( [ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 ] );

var arr = iterator2arrayview( iter, new Uint8Array( 10 ), 0, 4 );
// returns <Uint8Array>[ 1, 2, 3, 4, 0, 0, 0, 0, 0, 0 ]
```

If `end` is less than `0`, the last view element is resolved relative to the last element of the provided array-like `object`. For example, the following achieves the same behavior as the previous example

```javascript
var Uint8Array = require( '@stdlib/array/uint8' );
var array2iterator = require( '@stdlib/array/to-iterator' );

var iter = array2iterator( [ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 ] );

var arr = iterator2arrayview( iter, new Uint8Array( 10 ), 0, -6 );
// returns <Uint8Array>[ 1, 2, 3, 4, 0, 0, 0, 0, 0, 0 ]
```

To invoke a function for each iterated value, provide a callback function.

```javascript
var Float64Array = require( '@stdlib/array/float64' );
var array2iterator = require( '@stdlib/array/to-iterator' );

function fcn( v ) {
    return v * 10.0;
}

var iter = array2iterator( [ 1, 2, 3, 4 ] );

var arr = iterator2arrayview( iter, new Float64Array( 4 ), fcn );
// returns <Float64Array>[ 10.0, 20.0, 30.0, 40.0 ]
```

The invoked function is provided three arguments:

-   `value`: iterated value
-   `index`: destination index
-   `n`: iteration count (zero-based)

```javascript
var Uint8Array = require( '@stdlib/array/uint8' );
var array2iterator = require( '@stdlib/array/to-iterator' );

function fcn( v, i ) {
    return v * (i+1);
}

var iter = array2iterator( [ 1, 2, 3, 4 ] );

var arr = iterator2arrayview( iter, new Uint8Array( 4 ), fcn );
// returns <Uint8Array>[ 1, 4, 9, 16 ]
```

To set the callback function execution context, provide a `thisArg`.

```javascript
var Float64Array = require( '@stdlib/array/float64' );
var randu = require( '@stdlib/random/iter/randu' );

function fcn( v ) {
    this.count += 1;
    return v * 10.0;
}

var ctx = {
    'count': 0
};

var arr = iterator2arrayview( randu(), new Float64Array( 10 ), fcn, ctx );
// returns <Float64Array>

var count = ctx.count;
// returns 10
```

</section>

<!-- /.usage -->

<!-- Package usage notes. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="notes">

## Notes

-   Iteration stops when an output array view is full **or** an iterator finishes; whichever comes first.

</section>

<!-- /.notes -->

<!-- Package usage examples. -->

<section class="examples">

## Examples

<!-- eslint no-undef: "error" -->

```javascript
var Float64Array = require( '@stdlib/array/float64' );
var randu = require( '@stdlib/random/iter/randu' );
var iterator2arrayview = require( '@stdlib/iter/to-array-view' );

function scale( v, i ) {
    return v * (i+1);
}

// Create an iterator for generating uniformly distributed pseudorandom numbers:
var it = randu();

// Fill an array view with scaled iterator values:
var arr = iterator2arrayview( it, new Float64Array( 100 ), 40, 60, scale );

var i;
for ( i = 0; i < arr.length; i++ ) {
    console.log( arr[ i ] );
}
```

</section>

<!-- /.examples -->

<!-- Section to include cited references. If references are included, add a horizontal rule *before* the section. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="references">

</section>

<!-- /.references -->

<!-- Section for all links. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="links">

</section>

<!-- /.links -->
