/*
 *      netlink.c
 *      
 *      Copyright 2012-2013 Alex <alex@linuxonly.ru>
 *      
 *      This program is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 3 of the License, or
 *      (at your option) any later version.
 *      
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *      
 *      You should have received a copy of the GNU General Public License
 *      along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <netinet/tcp.h>
#include <netinet/in.h>
#include <linux/netlink.h>
#include <linux/inet_diag.h>
#include <dirent.h>
#include <ctype.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <net/if.h>

#include "netlink.h"

#define MMGUI_NETLINK_INTERNAL_SEQUENCE_NUMBER 100000

static gboolean mmgui_netlink_numeric_name(gchar *dirname)
{
	if ((dirname == NULL) || (!*dirname)) return FALSE;
	
	while (*dirname) {
		if (!isdigit(*dirname)) {
			return FALSE;
		} else {
			*dirname++;
		}
	}
	
	return TRUE;
}

static gboolean mmgui_netlink_process_access(gchar *dirname, uid_t uid)
{
	gchar fullpath[128];
	struct stat pathstat;
	
	if (!mmgui_netlink_numeric_name(dirname)) return FALSE;
	
	memset(fullpath, 0, sizeof(fullpath));
	
	sprintf(fullpath, "/proc/%s", dirname);
	
	if (stat(fullpath, &pathstat) == -1) {
		return FALSE;
	}
	
	if (pathstat.st_uid != uid) {
		return FALSE;
	}
	
	return TRUE;
}

static gboolean mmgui_netlink_socket_access(gchar *dirname, gchar *sockname, guint inode)
{
	gchar fullpath[128];
	struct stat fdstat;
	
	if (!mmgui_netlink_numeric_name(sockname)) return FALSE;
		
	memset(fullpath, 0, sizeof(fullpath));
	
	sprintf(fullpath, "/proc/%s/fd/%s", dirname, sockname);
	
	if (stat(fullpath, &fdstat) == -1) {
		return FALSE;
	}
	
	if (((fdstat.st_mode & S_IFMT) == S_IFSOCK) && (fdstat.st_ino == inode)) {
		return TRUE;
	}
	
	return FALSE;
}

static gchar *mmgui_netlink_process_name(gchar *dirname, gchar *appname, gsize appsize)
{
	gint fd, i;
	gchar fpath[128];
	ssize_t linkchars;
		
	if ((dirname == NULL) || (dirname[0] == '\0')) return NULL;
	if ((appname == NULL) || (appsize == 0)) return NULL;
	
	memset(fpath, 0, sizeof(fpath));
	sprintf(fpath, "/proc/%s/exe", dirname);
	
	linkchars = readlink(fpath, appname, appsize-1);
	
	if (linkchars == 0) {
		memset(fpath, 0, sizeof(fpath));
		sprintf(fpath, "/proc/%s/comm", dirname);
		
		fd = open(fpath, O_RDONLY);
		if (fd != -1) {
			linkchars = read(fd, appname, appsize-1);
			close(fd);
		} else {
			return NULL;
		}
	}
		
	appname[linkchars] = '\0';
	
	for (i=linkchars; i>=0; i--) {
		if (appname[i] == '/') {
			memmove(appname+0, appname+i+1, linkchars-i-1);
			linkchars -= i+1;
			break;
		}
	}
	
	appname[linkchars] = '\0';
	
	return appname;
}

static gboolean mmgui_netlink_get_process(guint inode, gchar *appname, gsize namesize, pid_t *apppid)
{
	DIR *procdir, *fddir;
	struct dirent *procde, *fdde;
	char fdirpath[128], fdpath[128];
	
	if ((appname == NULL) || (namesize == 0) || (apppid == NULL)) return FALSE;
	
	procdir = opendir("/proc");
	if (procdir != NULL) {
		while (procde = readdir(procdir)) {
			if (mmgui_netlink_process_access(procde->d_name, getuid())) {
				memset(fdirpath, 0, sizeof(fdirpath));
				sprintf(fdirpath, "/proc/%s/fd", procde->d_name);
				//enumerate file descriptors
				fddir = opendir(fdirpath);
				if (fddir != NULL) {
					while (fdde = readdir(fddir)) {
						if (mmgui_netlink_socket_access(procde->d_name, fdde->d_name, inode)) {
							//printf("%s:%s (%s)\n", procde->d_name, fdde->d_name, nlconnections_process_name(procde->d_name, appname, sizeof(appname)));
							*apppid = atoi(procde->d_name);
							mmgui_netlink_process_name(procde->d_name, appname, namesize);
							closedir(fddir);
							closedir(procdir);
							return TRUE;
						}
					}
					closedir(fddir);
				}
			}
		}
		closedir(procdir);
	}
	
	return FALSE;
}

gboolean mmgui_netlink_terminate_application(pid_t pid)
{
	if (kill(pid, 0) == 0) {
		if (kill(pid, SIGTERM) == 0) {
			return TRUE;
		}
	}
	
	return FALSE;
}

gchar *mmgui_netlink_socket_state(guchar state)
{
	switch (state) {
		case TCP_ESTABLISHED:
			return "Established";
		case TCP_SYN_SENT:
			return "SYN sent";
		case TCP_SYN_RECV:
			return "SYN recv";
		case TCP_FIN_WAIT1:
			return "FIN wait";
		case TCP_FIN_WAIT2:
			return "FIN wait";
		case TCP_TIME_WAIT:
			return "Time wait";
		case TCP_CLOSE:
			return "Close";
		case TCP_CLOSE_WAIT:
			return "Close wait";
		case TCP_LAST_ACK:
			return "Last ACK";
		case TCP_LISTEN:
			return "Listen";
		case TCP_CLOSING:
			return "Closing";
		default:
			return "Unknown";
	}
}

void mmgui_netlink_hash_destroy(gpointer data)
{
	mmgui_netlink_connection_t connection;
	
	connection = (mmgui_netlink_connection_t)data;
	
	if (connection == NULL) return;
	
	if (connection->appname != NULL) g_free(connection->appname);
	if (connection->dsthostname != NULL) g_free(connection->dsthostname);
	
	g_free(connection);
}

static gboolean mmgui_netlink_hash_clear_foreach(gpointer key, gpointer value, gpointer user_data)
{
	mmgui_netlink_connection_t connection;
	time_t currenttime;
	
	connection = (mmgui_netlink_connection_t)value;
	currenttime = *(time_t *)user_data;
	
	if (connection->updatetime == currenttime) {
		return FALSE;
	} else {
		//printf("Remove: %u\n", socket->inode);
		return TRUE;
	}
}

gboolean mmgui_netlink_request_connections_list(mmgui_netlink_t netlink, guint family)
{
	struct _mmgui_netlink_connection_info_request request;
	gint status;
	
	if ((netlink == NULL) || ((!(family & AF_INET)) && (!(family & AF_INET6)))) return FALSE;
	
	memset(&request.msgheader, 0, sizeof(struct nlmsghdr));
	
	request.msgheader.nlmsg_len = sizeof(struct _mmgui_netlink_connection_info_request);
	request.msgheader.nlmsg_type = TCPDIAG_GETSOCK;
	request.msgheader.nlmsg_flags = NLM_F_REQUEST | NLM_F_ROOT;
	request.msgheader.nlmsg_pid = getpid();
	request.msgheader.nlmsg_seq = 0;
	
	memset(&request.nlreq, 0, sizeof(struct inet_diag_req));
	request.nlreq.idiag_family = family;
	request.nlreq.idiag_states = ((1 << TCP_CLOSING + 1) - 1);
	
	request.msgheader.nlmsg_len = NLMSG_ALIGN(request.msgheader.nlmsg_len);
	
	status = send(netlink->connsocketfd, &request, request.msgheader.nlmsg_len, 0);
	
	if (status != -1) {
		return TRUE;
	} else {
		return FALSE;
	}
}

gboolean mmgui_netlink_read_connections_list(mmgui_netlink_t netlink, gchar *data, gsize datasize)
{
	time_t currenttime;
	struct nlmsghdr *msgheader;
	struct inet_diag_msg *entry;
	mmgui_netlink_connection_t connection;
	struct hostent *dsthost;
	gchar srcbuf[INET6_ADDRSTRLEN];
	gchar dstbuf[INET6_ADDRSTRLEN];
	gchar appname[1024];
	pid_t apppid;
			
	if ((netlink == NULL) || (data == NULL) || (datasize == 0)) return FALSE;
	
	//Get current time
	currenttime = time(NULL);
	
	//Work with data
	for (msgheader = (struct nlmsghdr *)data; NLMSG_OK(msgheader, (unsigned int)datasize); msgheader = NLMSG_NEXT(msgheader, datasize)) {
		if ((msgheader->nlmsg_type == NLMSG_ERROR) || (msgheader->nlmsg_type == NLMSG_DONE)) {
			break;
		}
		//New connections list
		if (msgheader->nlmsg_type == TCPDIAG_GETSOCK) {
			entry = (struct inet_diag_msg *)NLMSG_DATA(msgheader);
			if (entry != NULL) {
				if ((entry->idiag_uid == netlink->userid) || (netlink->userid == 0)) {
					if (!g_hash_table_contains(netlink->connections, (gconstpointer)&entry->idiag_inode)) {
						//Add new connection
						if (mmgui_netlink_get_process(entry->idiag_inode, appname, sizeof(appname), &apppid)) {
							connection = g_new(struct _mmgui_netlink_connection, 1);
							connection->inode = entry->idiag_inode;
							connection->family = entry->idiag_family;
							connection->userid = entry->idiag_uid;
							connection->updatetime = currenttime;
							connection->dqueue = entry->idiag_rqueue + entry->idiag_wqueue;
							connection->state = entry->idiag_state;
							connection->srcport = ntohs(entry->id.idiag_sport);
							g_snprintf(connection->srcaddr, sizeof(connection->srcaddr), "%s:%u", inet_ntop(entry->idiag_family, entry->id.idiag_src, srcbuf, INET6_ADDRSTRLEN), ntohs(entry->id.idiag_sport));
							g_snprintf(connection->dstaddr, sizeof(connection->dstaddr), "%s:%u", inet_ntop(entry->idiag_family, entry->id.idiag_dst, dstbuf, INET6_ADDRSTRLEN), ntohs(entry->id.idiag_dport));
							connection->appname = g_strdup(appname);
							connection->apppid = apppid;
							connection->dsthostname = NULL;
							/*dsthost = gethostbyaddr(entry->id.idiag_dst, sizeof(entry->id.idiag_dst), entry->idiag_family);
							if (dsthost != NULL) {
								connection->dsthostname = g_strdup(dsthost->h_name);
							} else {*/
								connection->dsthostname = g_strdup(connection->dstaddr);
							/*}*/
							g_hash_table_insert(netlink->connections, (gpointer)&connection->inode, connection);
							g_debug("Connection added: inode %u\n", entry->idiag_inode);
						}
					} else {
						//Update connection information (state, buffers fill, time)
						connection = g_hash_table_lookup(netlink->connections, (gconstpointer)&entry->idiag_inode);
						if (connection != NULL) {
							connection->updatetime = currenttime;
							connection->dqueue = entry->idiag_rqueue + entry->idiag_wqueue;
							connection->state = entry->idiag_state;
							g_debug("Connection updated: inode %u\n", entry->idiag_inode);
						}	
					}
				}
			}
		}
	}
	
	//Remove connections that disappear
	g_hash_table_foreach_remove(netlink->connections, mmgui_netlink_hash_clear_foreach, &currenttime);
	
	return TRUE;
}

gboolean mmgui_netlink_request_interface_statistics(mmgui_netlink_t netlink, gchar *interface)
{
	struct _mmgui_netlink_interface_info_request request;
	guint ifindex;
	gint status;
		
	if ((netlink == NULL) || (interface == NULL)) return FALSE;
	if ((netlink->intsocketfd == -1) || (interface[0] == '\0')) return FALSE;
	
	ifindex = if_nametoindex(interface);
	
	if ((ifindex == 0) && (errno == ENXIO)) return FALSE;
	
	memset(&request, 0, sizeof(request));
	
	request.msgheader.nlmsg_len = NLMSG_LENGTH(sizeof(struct ifinfomsg));
	request.msgheader.nlmsg_type = RTM_GETLINK;
	request.msgheader.nlmsg_flags = NLM_F_REQUEST;
	request.msgheader.nlmsg_seq = MMGUI_NETLINK_INTERNAL_SEQUENCE_NUMBER;
	request.msgheader.nlmsg_pid = getpid();
	
	request.ifinfo.ifi_family = AF_UNSPEC;
	request.ifinfo.ifi_index = ifindex;
	request.ifinfo.ifi_type = 0;
	request.ifinfo.ifi_flags = 0;
	request.ifinfo.ifi_change = 0xFFFFFFFF;
	
	request.msgheader.nlmsg_len = NLMSG_ALIGN(request.msgheader.nlmsg_len);
	
	status = send(netlink->intsocketfd, &request, request.msgheader.nlmsg_len, 0);
	
	if (status != -1) {
		return TRUE;
	} else {
		return FALSE;
	}
}

gboolean mmgui_netlink_read_interface_event(mmgui_netlink_t netlink, gchar *data, gsize datasize, mmgui_netlink_interface_event_t event)
{
	struct nlmsghdr *msgheader;
	struct ifinfomsg *ifi;
	struct rtattr *rta;
	struct rtnl_link_stats *ifstats;
	struct rtnl_link_stats64 *ifstats64;
	gchar ifname[IFNAMSIZ];
	gboolean have64bitstats;
	
	if ((netlink == NULL) || (data == NULL) || (datasize == 0) || (event == NULL)) return FALSE;
	
	//Work with data
	for (msgheader = (struct nlmsghdr *)data; NLMSG_OK(msgheader, (unsigned int)datasize); msgheader = NLMSG_NEXT(msgheader, datasize)) {
		if ((msgheader->nlmsg_type == NLMSG_ERROR) || (msgheader->nlmsg_type == NLMSG_DONE)) {
			break;
		}
		//New event
		if ((msgheader->nlmsg_type == RTM_NEWLINK) || (msgheader->nlmsg_type == RTM_DELLINK) || (msgheader->nlmsg_type == RTM_GETLINK)) {
			ifi = NLMSG_DATA(msgheader);
			rta = IFLA_RTA(ifi);
			//Copy valuable data first
			event->running = (ifi->ifi_flags & IFF_RUNNING);
			event->up = (ifi->ifi_flags & IFF_UP);
			if (if_indextoname(ifi->ifi_index, ifname) != NULL) {
				strncpy(event->ifname, ifname, sizeof(event->ifname));
			}
			event->type = MMGUI_NETLINK_INTERFACE_EVENT_TYPE_UNKNOWN;
			//Detrmine type of event
			if (msgheader->nlmsg_seq == MMGUI_NETLINK_INTERNAL_SEQUENCE_NUMBER) {
				event->type = MMGUI_NETLINK_INTERFACE_EVENT_TYPE_STATS;
			} else {
				if (msgheader->nlmsg_type == RTM_NEWLINK) {
					event->type = MMGUI_NETLINK_INTERFACE_EVENT_TYPE_ADD;
					g_debug("[%u] New interface state: Running: %s, Up: %s, Name: %s\n", msgheader->nlmsg_seq, (ifi->ifi_flags & IFF_RUNNING) ? "Yes" : "No", (ifi->ifi_flags & IFF_UP) ? "Yes" : "No",  if_indextoname(ifi->ifi_index, ifname));
				} else if (msgheader->nlmsg_type == RTM_DELLINK) {
					event->type = MMGUI_NETLINK_INTERFACE_EVENT_TYPE_REMOVE;
					g_debug("[%u] Deleted interface state: Running: %s, Up: %s, Name: %s\n", msgheader->nlmsg_seq, (ifi->ifi_flags & IFF_RUNNING) ? "Yes" : "No", (ifi->ifi_flags & IFF_UP) ? "Yes" : "No",  if_indextoname(ifi->ifi_index, ifname));
				} else if (msgheader->nlmsg_type == RTM_GETLINK) {
					event->type = MMGUI_NETLINK_INTERFACE_EVENT_TYPE_STATS;
					g_debug("[%u] Requested interface state: Running: %s, Up: %s, Name: %s\n", msgheader->nlmsg_seq, (ifi->ifi_flags & IFF_RUNNING) ? "Yes" : "No", (ifi->ifi_flags & IFF_UP) ? "Yes" : "No",  if_indextoname(ifi->ifi_index, ifname));
				}
			} 
			//If 64bit traffic statistics values are not available, 32bit values will be used anyway
			have64bitstats = FALSE;
			//Use tags to get additional data
			while (RTA_OK(rta, msgheader->nlmsg_len)) {
				if (rta->rta_type == IFLA_IFNAME) {
					strncpy(event->ifname, (char *)RTA_DATA(rta), sizeof(event->ifname));
					g_debug("Tag: Device name: %s\n", (char *)RTA_DATA(rta));
				} else if (rta->rta_type == IFLA_STATS) {
					ifstats = (struct rtnl_link_stats *)RTA_DATA(rta);
					if (!have64bitstats) {
						event->rxbytes = ifstats->rx_bytes;
						event->txbytes = ifstats->tx_bytes;
						if (!(event->type & MMGUI_NETLINK_INTERFACE_EVENT_TYPE_STATS)) {
							event->type |= MMGUI_NETLINK_INTERFACE_EVENT_TYPE_STATS;
						}
					}
					g_debug("Tag: Interface Statistics (32bit): RX: %u, TX: %u\n", ifstats->rx_bytes, ifstats->tx_bytes);
				} else if (rta->rta_type == IFLA_STATS64) {
					ifstats64 = (struct rtnl_link_stats64 *)RTA_DATA(rta);
					event->rxbytes = ifstats64->rx_bytes;
					event->txbytes = ifstats64->tx_bytes;
					have64bitstats = TRUE;
					if (!(event->type & MMGUI_NETLINK_INTERFACE_EVENT_TYPE_STATS)) {
						event->type |= MMGUI_NETLINK_INTERFACE_EVENT_TYPE_STATS;
					}
					g_debug("Tag: Interface Statistics (64bit): RX: %llu, TX: %llu\n", ifstats64->rx_bytes, ifstats64->tx_bytes);
				} else if (rta->rta_type == IFLA_LINK) {
					g_debug("Tag: Link type\n");
				} else if (rta->rta_type == IFLA_ADDRESS) {
					g_debug("Tag: interface L2 address\n");
				} else if (rta->rta_type == IFLA_UNSPEC) {
					g_debug("Tag: unspecified\n");
				} else {
					g_debug("Tag: %u\n", rta->rta_type);
				}
				rta = RTA_NEXT(rta, msgheader->nlmsg_len);
			}
		}
	}
	
	return TRUE;
}

gint mmgui_netlink_get_connections_monitoring_socket_fd(mmgui_netlink_t netlink)
{
	if (netlink == NULL) return -1;
	
	return netlink->connsocketfd;
}

gint mmgui_netlink_get_interfaces_monitoring_socket_fd(mmgui_netlink_t netlink)
{
	if (netlink == NULL) return -1;
	
	return netlink->intsocketfd;
}

struct sockaddr_nl *mmgui_netlink_get_connections_monitoring_socket_address(mmgui_netlink_t netlink)
{
	if (netlink == NULL) return NULL;
	
	return &(netlink->connaddr);
}

struct sockaddr_nl *mmgui_netlink_get_interfaces_monitoring_socket_address(mmgui_netlink_t netlink)
{
	if (netlink == NULL) return NULL;
	
	return &(netlink->intaddr);
}

GHashTable *mmgui_netlink_get_connections_list(mmgui_netlink_t netlink)
{
	if (netlink == NULL) return NULL;
	
	return netlink->connections;
}

void mmgui_netlink_close(mmgui_netlink_t netlink)
{
	if (netlink == NULL) return;
	
	if (netlink->connsocketfd != -1) {
		close(netlink->connsocketfd);
		g_hash_table_destroy(netlink->connections);
	}
	
	if (netlink->intsocketfd != -1) {
		close(netlink->intsocketfd);
	}
	
	g_free(netlink);
}

mmgui_netlink_t mmgui_netlink_open(void)
{
	mmgui_netlink_t netlink;
		
	netlink = g_new(struct _mmgui_netlink, 1);
	
	#ifndef NETLINK_SOCK_DIAG
		netlink->connsocketfd = socket(AF_NETLINK, SOCK_RAW, NETLINK_INET_DIAG);
	#else
		netlink->connsocketfd = socket(AF_NETLINK, SOCK_RAW, NETLINK_SOCK_DIAG);
	#endif
	
	if (netlink->connsocketfd != -1) {
		memset(&netlink->connaddr, 0, sizeof(struct sockaddr_nl));
		netlink->connaddr.nl_family = AF_NETLINK;
		netlink->connaddr.nl_pid = getpid();
		netlink->connaddr.nl_groups = 0;
		
		netlink->userid = getuid();
		
		netlink->connections = g_hash_table_new_full(g_int_hash, g_int_equal, NULL, (GDestroyNotify)mmgui_netlink_hash_destroy);
	} else {
		netlink->connections = NULL;
		g_debug("Failed to open connections monitoring netlink socket\n");
	}
	
	netlink->intsocketfd = socket(AF_NETLINK, SOCK_RAW, NETLINK_ROUTE);
	
	if (netlink->intsocketfd != -1) {
		memset(&netlink->intaddr, 0, sizeof(netlink->intaddr));
		netlink->intaddr.nl_family = AF_NETLINK;
		netlink->intaddr.nl_groups = RTMGRP_LINK;
		netlink->intaddr.nl_pid = getpid();
		
		if (bind(netlink->intsocketfd, (struct sockaddr *)&(netlink->intaddr), sizeof(netlink->intaddr)) == -1) {
			g_debug("Failed to bind network interfaces monitoring netlink socket\n");
			close(netlink->intsocketfd);
			netlink->intsocketfd = -1;
		}
	} else {
		g_debug("Failed to open network interfaces monitoring netlink socket\n");
	}
	
	return netlink;
}
