////////////////////////////////////////////////////////////////////////////////
//
//	Pico Technology USB Device Driver
//
///	\file     PicoUsbDevice_Linux.h
///	\brief    Generic Pico USB device class
//
//	Copyright (c) 2007, Pico Technology.
//	All rights reserved.
//   
//	Redistribution and use in source and binary forms, with or without
//	modification, are permitted provided that the following conditions are met:
//		* Redistributions of source code must retain the above copyright
//		  notice, this list of conditions and the following disclaimer.
//		* Redistributions in binary form must reproduce the above copyright
//		  notice, this list of conditions and the following disclaimer in the
//		  documentation and/or other materials provided with the distribution.
//		* The name of Pico Technology may not be used to endorse or promote
//		  products derived from this software without specific prior written
//		  permission.
//
//	THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY ``AS IS'' AND ANY
//	EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
//	DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
//	ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
//	THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//	Version $Id: PicoUsbDevice_Linux.h,v 1.10 2008/03/11 15:17:03 douglas Exp $
//
////////////////////////////////////////////////////////////////////////////////

#include "PicoPortability.h"
#include "LinuxUsbFS.h"

/* Avoid including this header if we are not compiling for Linux */
#ifdef PICO_OS_LINUX

/* Avoid including this header more than once */
#ifndef PICOUSBDEVICE_LINUX_H
#define PICOUSBDEVICE_LINUX_H

#include <vector.h>

// Format of USB Descriptors is defined in USB2.0 spec, section 9.6
#pragma pack (1)
typedef struct {
	uint8_t  bLength;
	uint8_t  bDescriptorType;
	uint16_t bcdUSB;
	uint8_t  bDeviceClass;
	uint8_t  bDeviceSubClass;
	uint8_t  bDeviceProtocol;
	uint8_t  bMaxPacketSize0;
	uint16_t idVendor;
	uint16_t idProduct;
	uint16_t bcdDevice;
	uint8_t  iManufacturer;
	uint8_t  iProduct;
	uint8_t  iSerialNumber;
	uint8_t  bNumConfigurations;
} DeviceDescriptor;

typedef struct {
	uint8_t  bLength;
	uint8_t  bDescriptorType;
	uint8_t  bEndpointAddress;
	uint8_t  bmAttributes;
	uint16_t wMaxPacketSize;
	uint8_t  bInterval;
} EndpointDescriptor;
#pragma pack()

typedef struct _UrbContext {
	class PicoLinuxUsbDevice * dev;
	int completed;
} UrbContext;

#include "PicoUsbDevice.h"

/// Provides platform-specific implementation of the PicoUsbDevice interface for 
/// Linux.
class PicoLinuxUsbDevice : PicoUsbDevice
{
public:
	PicoLinuxUsbDevice(int newDevice, unsigned char * device_desc);
	~PicoLinuxUsbDevice(void);
	PICODEVICE_STATES Open (void);
	void              Close(void);
	
	// Getting device information
	int GetPID(void);
	int GetDID(void);
	int GetDeviceClass(void);
	int GetPipes(void);
	short GetHandle(void);
	const char *      GetSerialString(void);
	PICODEVICE_STATES GetDeviceState (void);
	PicoUsbDevice::PipeInformation GetPipeInformation(int pipeNumber);
	void ProcessUrb(UrbContext * urb);
	static std::vector<char *>  * ListUSBDevices(void);
	
	// Data transfer
	PICO_RETURNS ReadPipe (int pipeNumber,void *buf, unsigned int *size);
	PICO_RETURNS WritePipe(int pipeNumber,void *buf, unsigned int size);
	PICO_RETURNS BulkTransfer(int pipeNumber,void *buf, unsigned int *size);
	PICO_RETURNS ResetPipe(int pipeNumber);

	private:
		/// Pointer to the usbfs file for our device
		int device;
		/// The connection state of the USB device
		volatile PICODEVICE_STATES state;
		/// Mutex used for synchronising access to the device
		pthread_mutex_t mutex;
		/// Mutex used for synchronising access to urb status with the signal handler
		pthread_mutex_t urbMutex;
		/// Mutex to protect deviceHandles array
		static pthread_mutex_t handlesMutex;
		/// Indicates if the handles array has been initialised
		static bool handlesInitialized;
		/// Array of bitfields to indicate whether each device handle is in use
		static unsigned char * deviceHandles;
		/// Handle of this device (for use with C API)
		short handle;		
		/// USB Serial number of device
		char *serialString;
		/// Vector containing USB Endpoint info
		std::vector<EndpointDescriptor *> Endpoints;
		/// USB Device Descriptor (PID, VID etc)
		DeviceDescriptor Descriptor;
		/// Is the device operating in high-speed USB mode
		bool highSpeed;
};

#endif // not defined PICOUSBDEVICE_LINUX_H

#endif // defined PICO_OS_LINUX
