/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file audio_winmm.c
 * \brief Windows MM audio plugin
 */

#include <ffgtk.h>
#include <preferences.h>
#include <windows.h>
#include <mmsystem.h>

#define BLOCK_SIZE 160
#define BLOCK_COUNT 200

/** predefined backup values */
static gint nWinMmChannels = 2;
static gint nWinMmSampleRate = 8000;
static gint nWinMmBitsPerSample = 16;

struct sBufferNode {
	LPBYTE anData;
	DWORD nBytes;
	struct sBufferNode *psNext;
	DWORD nNum;
};

struct sWinMmPrivate {
	CRITICAL_SECTION nWaveCriticalSection;
	WAVEHDR *psWaveOutBlocks;
	WAVEHDR *psWaveInBlocks;
	WAVEHDR asWaveInBlocks[ 2 ];
	volatile int nWaveFreeBlockCount;
	int nWaveCurrentBlock;
	int nDoneAll;
	int nInRecord;
	volatile long nNumUsedIn;
	HANDLE nWaveInThread;
	HWAVEOUT nWaveOut;
	HWAVEIN nWaveIn;
	WAVEFORMATEX sWinFx;
	WAVEFORMATEX sWinOutFx;
	CRITICAL_SECTION sInLock;
};

#define NUM_HEADERS 4
#define NUM_BUFFERS	160

static HGLOBAL *apnMemBuff[ NUM_BUFFERS ];
struct sBufferNode *apsBuffer[ NUM_BUFFERS ];
static HGLOBAL *apnMemBuffData[ NUM_BUFFERS ];
static long nChunkSize = 320;
struct sBufferNode *psFirstFree;
struct sBufferNode *psFirstUsed;

void CALLBACK waveOutProc(HWAVEOUT nWaveOut, UINT nMsg, DWORD nInstance, DWORD nParam1, DWORD nParam2 ) {
	struct sWinMmPrivate *psPrivate = ( struct sWinMmPrivate * ) nInstance;

	/* ignore calls that occur due to openining and closing the device.*/
	if ( nMsg != WOM_DONE ) {
		return;
	}

	//EnterCriticalSection( &psPrivate -> nWaveCriticalSection );
	psPrivate -> nWaveFreeBlockCount++;
	//LeaveCriticalSection( &psPrivate -> nWaveCriticalSection );
}

WAVEHDR *allocateBlocks( int nSize, int nCount ) {
	char *pnBuffer;
	int nIndex;
	WAVEHDR *psBlocks;
	DWORD nTotalBufferSize = ( nSize + sizeof( WAVEHDR ) ) * nCount;

	/* allocate memory for the entire set in one go */
	if ( ( pnBuffer = HeapAlloc( GetProcessHeap(), HEAP_ZERO_MEMORY, nTotalBufferSize ) ) == NULL ) {
		fprintf( stderr, "Memory allocation error\n" );
		return NULL;
	}

	/* and set up the pointers to each bit */
	psBlocks = ( WAVEHDR * ) pnBuffer;
	pnBuffer += sizeof( WAVEHDR ) * nCount;

	for ( nIndex = 0; nIndex < nCount; nIndex++ ) {
		psBlocks[ nIndex ].dwBufferLength = nSize;
		psBlocks[ nIndex ].lpData = pnBuffer;
		pnBuffer += nSize;
	}

	return psBlocks;
}

void freeBlocks( WAVEHDR *psBlockArray ) {
	/* and this is why allocateBlocks works the way it does */
	HeapFree( GetProcessHeap(), 0, psBlockArray );
}

/**
 * \brief Detect supported audio devices
 * \return 0
 */
static int winmmAudioDetectDevices( void ) {
	return 0;
}

/**
 * \brief Initialize audio device
 * \param nChannels number of channels
 * \param nSampleRate sample rate
 * \param nBitsPerSample number of bits per samplerate
 * \return TRUE on success, otherwise error
 */
static int winmmAudioInit( unsigned char nChannels, unsigned short nSampleRate, unsigned char nBitsPerSample ) {
	/* TODO: Check if configuration is valid and usable */
	nWinMmChannels = nChannels;
	nWinMmSampleRate = nSampleRate;
	nWinMmBitsPerSample = nBitsPerSample;

	return 0;
}

void StoreBuffer( struct sWinMmPrivate *psPrivate, WAVEHDR *psHeader ) {
	MMRESULT nRes;

	if ( psPrivate -> nNumUsedIn < NUM_BUFFERS ) {
		//EnterCriticalSection( &psPrivate -> sInLock );
		//Debug( KERN_DEBUG, "Adding data %d/%d\n", psPrivate -> nNumUsedIn, NUM_BUFFERS );

		ZeroMemory( psFirstFree -> anData, psHeader -> dwBytesRecorded );
		memcpy( ( LPBYTE ) psFirstFree -> anData, psHeader -> lpData, psHeader -> dwBytesRecorded );
		psFirstFree -> nBytes = psHeader -> dwBytesRecorded;
		psFirstFree = psFirstFree -> psNext;
		InterlockedIncrement( &psPrivate -> nNumUsedIn );

		ZeroMemory( psHeader -> lpData, psHeader -> dwBufferLength );
		nRes = waveInPrepareHeader( psPrivate -> nWaveIn, psHeader, sizeof( WAVEHDR ) );
		if ( nRes == MMSYSERR_NOERROR ) {
			nRes = waveInAddBuffer( psPrivate -> nWaveIn, psHeader, sizeof( WAVEHDR ) );
		}

		//LeaveCriticalSection( &psPrivate -> sInLock );
	} else {
		EnterCriticalSection( &psPrivate -> sInLock );
		//Debug( KERN_DEBUG, "Full\n" );
		ZeroMemory( psHeader -> lpData, psHeader -> dwBufferLength );
		nRes = waveInPrepareHeader( psPrivate -> nWaveIn, psHeader, sizeof( WAVEHDR ) );
		if ( nRes == MMSYSERR_NOERROR ) {
			nRes = waveInAddBuffer( psPrivate -> nWaveIn, psHeader, sizeof( WAVEHDR ) );
		}
		LeaveCriticalSection( &psPrivate -> sInLock );
	}
}

DWORD WINAPI waveInProc( LPVOID arg ) {
	MSG msg;
	struct sWinMmPrivate *psPrivate = arg;

	while ( GetMessage( &msg, 0, 0, 0 ) == 1 ) {
		switch ( msg.message ) {
			case MM_WIM_DATA:
				StoreBuffer( psPrivate, ( ( WAVEHDR * ) msg.lParam ) );
				break;
			case MM_WIM_OPEN:
				Debug( KERN_DEBUG, "MM_WIM_OPEN\n" );
				psPrivate -> nDoneAll = 0;
				break;
			case MM_WIM_CLOSE:
				Debug( KERN_DEBUG, "MM_WIM_CLOSE\n" );
				break;
			default:
				Debug( KERN_DEBUG, "MM_WIM_??? (%x)\n", msg.message );
				break;
		}
	}

	return 0;
}

/**
 * \brief Open audio device
 * \return private data or NULL on error
 */
static void *winmmAudioOpen( void ) {
	struct sWinMmPrivate *psPrivate = malloc( sizeof( struct sWinMmPrivate ) );
	MMRESULT nErr = 0;
	DWORD nId;
	int nI;

	if ( psPrivate == NULL ) {
		return psPrivate;
	}
	memset( psPrivate, 0, sizeof( struct sWinMmPrivate ) );

	psPrivate -> psWaveOutBlocks = allocateBlocks( BLOCK_SIZE, BLOCK_COUNT );
	psPrivate -> nWaveFreeBlockCount = BLOCK_COUNT;
	psPrivate -> nWaveCurrentBlock = 0;

	InitializeCriticalSection( &psPrivate -> nWaveCriticalSection );

	/* Set up the WAVEFORMATEX structure */
	psPrivate -> sWinFx.nSamplesPerSec = nWinMmSampleRate;
	psPrivate -> sWinFx.wBitsPerSample = nWinMmBitsPerSample;
	psPrivate -> sWinFx.nChannels = nWinMmChannels;
	psPrivate -> sWinFx.cbSize = 0;
	psPrivate -> sWinFx.wFormatTag = WAVE_FORMAT_PCM;
	psPrivate -> sWinFx.nBlockAlign = ( psPrivate -> sWinFx.wBitsPerSample * psPrivate -> sWinFx.nChannels ) >> 3;
	psPrivate -> sWinFx.nAvgBytesPerSec = psPrivate -> sWinFx.nBlockAlign * psPrivate -> sWinFx.nSamplesPerSec;

	if ( waveOutOpen( &psPrivate -> nWaveOut, WAVE_MAPPER, &psPrivate -> sWinFx, ( DWORD_PTR ) waveOutProc, ( DWORD_PTR ) psPrivate, CALLBACK_FUNCTION ) != MMSYSERR_NOERROR ) {
		Debug( KERN_WARNING, "Unable to open wave mapper device\n" );
		return NULL;
	}

	/**
	 *  Now setup input data
	 */

	psPrivate -> nWaveInThread = CreateThread( 0, 0, ( LPTHREAD_START_ROUTINE ) waveInProc, psPrivate, 0, &nId );
	if ( psPrivate -> nWaveInThread == 0 ) {
		Debug( KERN_WARNING, "Can't create WAVE recording thread! -- %08X\n", GetLastError() );
		// TODO: Free structure
		return NULL;
	}
	CloseHandle( psPrivate -> nWaveInThread );

	ZeroMemory( &psPrivate -> asWaveInBlocks[ 0 ], sizeof( WAVEHDR ) * 2 );

	psPrivate -> sWinOutFx.nSamplesPerSec = nWinMmSampleRate;
	psPrivate -> sWinOutFx.wBitsPerSample = nWinMmBitsPerSample;
	psPrivate -> sWinOutFx.nChannels = nWinMmChannels;
	psPrivate -> sWinOutFx.cbSize = 0;
	psPrivate -> sWinOutFx.wFormatTag = WAVE_FORMAT_PCM;
	psPrivate -> sWinOutFx.nBlockAlign = ( psPrivate -> sWinFx.wBitsPerSample * psPrivate -> sWinFx.nChannels ) >> 3;
	psPrivate -> sWinOutFx.nAvgBytesPerSec = nChunkSize >> 1;

	nErr = waveInOpen( &psPrivate -> nWaveIn, WAVE_MAPPER, &psPrivate -> sWinOutFx, nId, 0, CALLBACK_THREAD );
	//nErr = waveInOpen( &psPrivate -> nWaveIn, WAVE_MAPPER, &psPrivate -> sWinOutFx, (DWORD)(VOID*)waveInProc, 0, CALLBACK_FUNCTION );
	if ( nErr ) {
		Debug( KERN_WARNING, "Unable to open wave in device\n" );
		return NULL;
	}

	psPrivate -> asWaveInBlocks[ 1 ].dwBufferLength = psPrivate -> asWaveInBlocks[ 0 ].dwBufferLength = psPrivate -> sWinOutFx.nAvgBytesPerSec << 1;
	if ( !( psPrivate -> asWaveInBlocks[ 0 ].lpData = ( char * ) VirtualAlloc( 0, psPrivate -> asWaveInBlocks[ 0 ].dwBufferLength * 2, MEM_COMMIT, PAGE_READWRITE ) ) ) {
		Debug( KERN_WARNING, "Can't allocate memory for WAVE buffer\n" );
		return NULL;
	}
	psPrivate -> asWaveInBlocks[ 1 ].lpData = psPrivate -> asWaveInBlocks[ 0 ].lpData + psPrivate -> asWaveInBlocks[ 0 ].dwBufferLength;

	if ( ( nErr = waveInPrepareHeader( psPrivate -> nWaveIn, &psPrivate -> asWaveInBlocks[ 0 ], sizeof( WAVEHDR ) ) ) ) {
		Debug( KERN_WARNING, "Error preparing WAVEHDR 1\n" );
		return NULL;
	}

	if ( ( nErr = waveInPrepareHeader( psPrivate -> nWaveIn, &psPrivate -> asWaveInBlocks[ 1 ], sizeof( WAVEHDR ) ) ) ) {
		Debug( KERN_WARNING, "Error preparing WAVEHDR 2\n" );
		return NULL;
	}

	if ( ( nErr = waveInAddBuffer( psPrivate -> nWaveIn, &psPrivate -> asWaveInBlocks[ 0 ], sizeof( WAVEHDR ) ) ) ) {
		Debug( KERN_WARNING, "Error queuing WAVEHDR 1\n" );
		return NULL;
	}

	if ( ( nErr = waveInAddBuffer( psPrivate -> nWaveIn, &psPrivate -> asWaveInBlocks[ 1 ], sizeof( WAVEHDR ) ) ) ) {
		Debug( KERN_WARNING, "Error queuing WAVEHDR 1\n" );
		return NULL;
	}
	psPrivate -> nInRecord = 1;

	InitializeCriticalSection( &psPrivate -> sInLock );

	EnterCriticalSection( &psPrivate -> sInLock );

	for ( nI = 0; nI < NUM_BUFFERS; nI++ ) {
		apnMemBuff[ nI ] = GlobalAlloc( GHND, sizeof( struct sBufferNode ) );
		apsBuffer[ nI ] = GlobalLock( apnMemBuff[ nI ] );

		if ( apsBuffer[ nI ] ) {
			apnMemBuffData[ nI ] = GlobalAlloc( GHND, nChunkSize );
			apsBuffer[ nI ] -> anData = GlobalLock( apnMemBuffData[ nI ] );
			apsBuffer[ nI ] -> nBytes = 0;
			apsBuffer[ nI ] -> nNum = nI;
		}
	}

	for ( nI = 0; nI < NUM_BUFFERS; nI++ ) {
		if ( nI == NUM_BUFFERS - 1 ) {
			apsBuffer[ nI ] -> psNext = apsBuffer[ 0 ];
		} else {
			apsBuffer[ nI ] -> psNext = apsBuffer[ nI + 1 ];
		}
	}

	psFirstUsed = apsBuffer[ 0 ];
	psFirstFree = apsBuffer[ 0 ];

	LeaveCriticalSection( &psPrivate -> sInLock );

	if ( ( nErr = waveInStart( psPrivate -> nWaveIn ) ) ) {
		Debug( KERN_WARNING, "Could not start record!\n" );
		return NULL;
	}

	Debug( KERN_DEBUG, "Done...\n" );

	return psPrivate;
}

/**
 * \brief Write audio data
 * \param pPriv private data
 * \param pnData audio data
 * \param nSize size of audio data
 * \return bytes written or error code
 */
static int winmmAudioWrite( void *pPriv, unsigned char *pnData, unsigned int nSize ) {
	struct sWinMmPrivate *psPrivate = pPriv;
	WAVEHDR *psCurrentBlock = NULL;
	int nRemain;

	if ( psPrivate == NULL ) {
		return 0;
	}

	psCurrentBlock = &psPrivate -> psWaveOutBlocks[ psPrivate -> nWaveCurrentBlock ];

	while (nSize > 0) {
		/* first make sure the header we're going to use is unprepared */
		if (psCurrentBlock->dwFlags & WHDR_PREPARED) {
			waveOutUnprepareHeader( psPrivate -> nWaveOut, psCurrentBlock, sizeof( WAVEHDR ) );
		}

		if ( nSize < ( int )( BLOCK_SIZE - psCurrentBlock -> dwUser ) ) {
			memcpy( psCurrentBlock -> lpData + psCurrentBlock -> dwUser, pnData, nSize );
			psCurrentBlock -> dwUser += nSize;
			break;
		}

		nRemain = BLOCK_SIZE - psCurrentBlock -> dwUser;
		memcpy( psCurrentBlock -> lpData + psCurrentBlock -> dwUser, pnData, nRemain );
		nSize -= nRemain;
		pnData += nRemain;
		psCurrentBlock -> dwBufferLength = BLOCK_SIZE;
		waveOutPrepareHeader( psPrivate -> nWaveOut, psCurrentBlock, sizeof( WAVEHDR ) );
		waveOutWrite( psPrivate -> nWaveOut, psCurrentBlock, sizeof( WAVEHDR ) );
		EnterCriticalSection( &psPrivate -> nWaveCriticalSection );
		psPrivate -> nWaveFreeBlockCount--;
		LeaveCriticalSection( &psPrivate -> nWaveCriticalSection );

		/* wait for a block to become free */
		while ( !psPrivate -> nWaveFreeBlockCount ) {
			Sleep( 10 );
		}

		/* point to the next block */
		psPrivate -> nWaveCurrentBlock++;
		psPrivate -> nWaveCurrentBlock %= BLOCK_COUNT;
		psCurrentBlock = &psPrivate -> psWaveOutBlocks[ psPrivate -> nWaveCurrentBlock ];
		psCurrentBlock->dwUser = 0;
	}

	return nSize;
}

static int winmmAudioRead( void *pPriv, unsigned char *pnData, unsigned int nSize ) {
	struct sWinMmPrivate *psPrivate = pPriv;
	int nRead = 0;

	if ( nSize >= nChunkSize && psPrivate -> nNumUsedIn > 0 ) {
		memcpy( pnData, psFirstUsed -> anData, psFirstUsed -> nBytes );
		nRead = psFirstUsed -> nBytes;

		ZeroMemory( psFirstUsed -> anData, psFirstUsed -> nBytes );
		psFirstUsed -> nBytes = 0;
		psFirstUsed = psFirstUsed -> psNext;

		InterlockedDecrement( &psPrivate -> nNumUsedIn );
	}

	return nRead;
}

/**
 * \brief Stop and remove pipeline
 * \param pPriv private data
 * \param bForce force quit
 * \return error code
 */
int winmmAudioClose( void *pPriv, gboolean bForce ) {
	struct sWinMmPrivate *psPrivate = pPriv;
	int nIndex;

	if ( psPrivate == NULL ) {
		return 0;
	}

	if ( bForce == TRUE ) {
		waveOutReset( psPrivate -> nWaveOut );
		waveInReset( psPrivate -> nWaveIn );
		return 0;
	}

	//if ( bForce == FALSE ) {
		while ( psPrivate -> nWaveFreeBlockCount < BLOCK_COUNT ) {
			Sleep( 10 );
		}
	//}

	for ( nIndex = 0; nIndex < psPrivate -> nWaveFreeBlockCount; nIndex++ ) {
		if ( psPrivate -> psWaveOutBlocks[ nIndex ].dwFlags & WHDR_PREPARED ) {
			waveOutUnprepareHeader( psPrivate -> nWaveOut, &psPrivate -> psWaveOutBlocks[ nIndex ], sizeof( WAVEHDR ) );
		}
	}

	DeleteCriticalSection( &psPrivate -> nWaveCriticalSection );
	freeBlocks( psPrivate -> psWaveOutBlocks );
	waveOutClose( psPrivate -> nWaveOut );
	waveInClose( psPrivate -> nWaveIn );

	psPrivate -> nInRecord = 0;


	//while ( psPrivate -> nDoneAll < 2 ) {
	//	Sleep( 100 );
	//}

	waveInUnprepareHeader( psPrivate -> nWaveIn, &psPrivate -> asWaveInBlocks[ 1 ], sizeof( WAVEHDR ) );
	waveInUnprepareHeader( psPrivate -> nWaveIn, &psPrivate -> asWaveInBlocks[ 0 ], sizeof( WAVEHDR ) );

	waveInReset( psPrivate -> nWaveIn );
	waveInClose( psPrivate -> nWaveIn );

	if ( psPrivate -> asWaveInBlocks[ 0 ].lpData ) {
		VirtualFree( psPrivate -> asWaveInBlocks[ 0 ].lpData, 0, MEM_RELEASE );
	}

	free( psPrivate );

	return 0;
}

int winmmAudioDeinit( void ) {
	return 0;
}

/** audio definition */
struct sAudio sWinMm = {
	winmmAudioInit,
	winmmAudioOpen,
	winmmAudioWrite,
	winmmAudioRead,
	winmmAudioClose,
	winmmAudioDeinit,
	NULL
};

MODULE_INIT( PLUGIN_TYPE_AUDIO, _( "Windows MM" ), &sWinMm, NULL, winmmAudioDetectDevices );
