/* $Id: CheckLoops.hpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * Check, if next statements occur within a loop statement.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __CHECK_LOOPS_HPP_INCLUDED
#define __CHECK_LOOPS_HPP_INCLUDED

#include "frontend/visitor/TopDownVisitor.hpp"
#include <list>

namespace ast {

//! check loop statements.
/** This class will check, if next statements will occur within loop 
 *  statements.
 *  TODO check if loop statements have an effect in the first place.
 */
class CheckLoops : public TopDownVisitor {
private:
	/** visit a node which affects a loop (Exit statement or
	 *  Next statement).
	 *  @param node exit or next statement node.
	 *  @param kind textual represention of node kind that is checked.
	 */
	template <typename T>
	void
	visitLoopCFNode(T &node, const char *kind) const;

	/** Visit a ExitStat
	 *  @param node ExitStat node that get's visited.
	 */
	virtual void visit(ExitStat &node);

	/** Visit a NextStat
	 *  @param node NextStat node that get's visited.
	 */
	virtual void visit(NextStat &node);

	//! Process a generic LoopStat.
        /** This function will get called for each LoopStat (or class
         *  derived from LoopStat) that get's visited.
         *
         *  @param node LoopStat instance.
         */
	virtual void process(LoopStat &node);

	//! search the stack for the loop label and return it.
	/** @param label look for loop statement with given label (optional)
	 *  @return corresponding loop statement or NULL if not found.
	 */
	LoopStat *lookup(SimpleName *label) const;

	//! report an error that no corresponding loop was found.
	/** @param node incorrect node (e.g. next statement).
	 *  @param label optional referred to loop label.
	 *  @param kind textual representation of kind of loop statement.
	 */
	void
	missingLoop(
		const AstNode &node, 
		const SimpleName *label, 
		const char *kind) const;

	//! stack of current loop statements.
	std::list<LoopStat*> loops;
};

}; /* namespace ast */

#endif /* __CHECK_LOOPS_HPP_INCLUDED */
