# -*- coding: utf-8 -*-

# Copyright (c) 2008 - 2020 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a PostScript lexer with some additional methods.
"""


from PyQt5.Qsci import QsciLexerPostScript

from .Lexer import Lexer
import Preferences


class LexerPostScript(Lexer, QsciLexerPostScript):
    """
    Subclass to implement some additional lexer dependant methods.
    """
    def __init__(self, parent=None):
        """
        Constructor
        
        @param parent parent widget of this lexer
        """
        QsciLexerPostScript.__init__(self, parent)
        Lexer.__init__(self)
        
        self.commentString = "%"
        
        self.keywordSetDescriptions = [
            self.tr("PS Level 1 operators"),
            self.tr("PS Level 2 operators"),
            self.tr("PS Level 3 operators"),
            self.tr("RIP specific operators"),
            self.tr("User defined operators"),
        ]
    
    def initProperties(self):
        """
        Public slot to initialize the properties.
        """
        self.setTokenize(Preferences.getEditor("PostScriptTokenize"))
        self.setLevel(Preferences.getEditor("PostScriptLevel"))
        self.setFoldAtElse(Preferences.getEditor("PostScriptFoldAtElse"))
        self.setFoldCompact(Preferences.getEditor("AllFoldCompact"))
    
    def isCommentStyle(self, style):
        """
        Public method to check, if a style is a comment style.
        
        @param style style to check (integer)
        @return flag indicating a comment style (boolean)
        """
        return style in [QsciLexerPostScript.Comment]
    
    def isStringStyle(self, style):
        """
        Public method to check, if a style is a string style.
        
        @param style style to check (integer)
        @return flag indicating a string style (boolean)
        """
        return False
    
    def defaultKeywords(self, kwSet):
        """
        Public method to get the default keywords.
        
        @param kwSet number of the keyword set (integer)
        @return string giving the keywords (string) or None
        """
        return QsciLexerPostScript.keywords(self, kwSet)
    
    def maximumKeywordSet(self):
        """
        Public method to get the maximum keyword set.
        
        @return maximum keyword set (integer)
        """
        return 5
