/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
   \file cdw_xorriso.c

   \brief Functions creating calls to xorriso and calling xorriso

   Functions for creating xorriso command line strings, creating string
   with xorriso parameters, and for calling cdw_thread_run_command()
   with prepared command line string.
*/

#define _GNU_SOURCE /* asprintf() */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>

#include "gettext.h"
#include "cdw_thread.h"
#include "cdw_config.h"
#include "cdw_string.h"
#include "cdw_xorriso.h"
#include "cdw_graftpoints.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_drive.h"
#include "cdw_erase_disc.h"
#include "cdw_libburn.h"


static char *cdw_xorriso_create_command_create_image(cdw_task_t *task);
static char *cdw_xorriso_create_command_media_info(cdw_task_t *task);
static char *cdw_xorriso_create_meta_information_string(cdw_task_t *task);
static char *cdw_xorriso_create_command_create_iso_fs(cdw_task_t *task);
static char *cdw_xorriso_create_command_burn_from_files(cdw_task_t *task, cdw_disc_t *disc);
static char *cdw_xorriso_create_command_burn_from_image(cdw_task_t *task, cdw_disc_t *disc);
static char *cdw_xorriso_create_command_erase(cdw_task_t *task, cdw_disc_t *disc);
static bool  cdw_xorriso_close_disc(cdw_task_t *task);

/**
   \brief Create iso image from files selected by user

   Create command line string (program name + arguments) for creating iso
   image from selected files, and run the command.

   Call this function only if all preconditions are met - the function
   behaves as if every condition for proper execution is met.

   Currently only creating iso image from data files is supported.

   \param task - variable describing current task
   \param disc - variable describing current disc

   \return CDW_OK when iso file is created
   \return CDW_GEN_ERROR if operation_failed
*/
cdw_rv_t cdw_xorriso_run_task(cdw_task_t *task, cdw_disc_t *disc)
{
	char *command = (char *) NULL;
	if (task->id == CDW_TASK_CREATE_IMAGE_ISO9660) {
		command = cdw_xorriso_create_command_create_image(task);
	} else if (task->id == CDW_TASK_BURN_FROM_FILES) {
		command = cdw_xorriso_create_command_burn_from_files(task, disc);
	} else if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		command = cdw_xorriso_create_command_burn_from_image(task, disc);
	} else if (task->id == CDW_TASK_MEDIA_INFO) {
		const char *drive_path = cdw_drive_get_drive_fullpath();
		cdw_rv_t crv = cdw_libburn_get_meta_info(disc, drive_path);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: libburn call returns !OK\n");
			return CDW_ERROR;
		}
		command = cdw_xorriso_create_command_media_info(task);
	} else if (task->id == CDW_TASK_ERASE_DISC) {
		command = cdw_xorriso_create_command_erase(task, disc);
	} else {
		cdw_assert (0, "ERROR: incorrect task id %lld\n", task->id);
	}

	if (command) {
		int rv = cdw_thread_run_command(command, task);
		free(command);
		command = (char *) NULL;

		if (rv == 0) {
			return CDW_OK;
		} else {
			return CDW_ERROR;
		}
	} else {
		cdw_vdm ("ERROR: failed to create xorriso command string\n");
		return CDW_ERROR;
	}
}





char *cdw_xorriso_create_command_create_image(cdw_task_t *task)
{
	cdw_assert (task->id == CDW_TASK_CREATE_IMAGE_ISO9660,
		    "ERROR: incorrect task id: %lld\n", task->id);
	cdw_assert (task->create_image.tool.id == CDW_TOOL_XORRISO,
		    "ERROR: incorrect tool id: %lld\n", task->create_image.tool.id);
	cdw_assert (task->create_image.tool.label, "ERROR: tool fullpath is NULL\n");

	char *fs_command = cdw_xorriso_create_command_create_iso_fs(task);
	if (!fs_command) {
		cdw_vdm ("ERROR: failed to create string with IS9660 parameters\n");
		return (char *) NULL;
	}
	char *command = cdw_string_concat(task->create_image.tool.label,
					  " -outdev ", task->image_file_fullpath, " ",
					  fs_command,
					  " -blank fast ", /* to start writing to image from scratch */

					  " -commit",
					  (char *) NULL);

	free(fs_command);
	fs_command = (char *) NULL;

	if (command) {
		cdw_vdm ("INFO: xorriso command for creating ISO9660 image: \"%s\"\n", command);
		return command;
	} else {
		cdw_vdm ("ERROR: failed to create xorriso command string\n");
		return (char *) NULL;
	}
}





char *cdw_xorriso_create_command_burn_from_files(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES,
		    "ERROR: incorrect task id: %lld\n", task->id);
	cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT,
		    "ERROR: session mode is not set properly\n");
	cdw_assert (task->burn.tool.id == CDW_TOOL_XORRISO,
		    "ERROR: incorrect tool id: %lld\n", task->burn.tool.id);
	cdw_assert (task->burn.tool.label != (char *) NULL,
		    "ERROR: tool fullpath is NULL\n");

	char *fs_command = cdw_xorriso_create_command_create_iso_fs(task);
	if (!fs_command) {
		cdw_vdm ("ERROR: failed to create string with IS9660 parameters\n");
		return (char *) NULL;
	}

	const char *device_fullpath = cdw_drive_get_drive_fullpath();
	cdw_assert (device_fullpath, "ERROR: \"device fullpath\" is NULL\n");

	bool close = cdw_xorriso_close_disc(task);
	char speed_string[5 + 1];
	snprintf(speed_string, 5 + 1, "%lld%s ", task->burn.speed_id, disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD ? "c" : "d");

	char *command = cdw_string_concat(task->burn.tool.label,
					  " -dev ", device_fullpath,
					  " -speed ", speed_string,
					  fs_command, /* ISO9660 main options + ISO9660 meta data options. */

					  " ", task->burn.dummy ? " -dummy on " : " ",
					  " ", task->burn.xorriso_burn_other_options ? task->burn.xorriso_burn_other_options : " ", " ",

					  close ? " -close on " : " -close off ",

					  " -commit",
					  (char *) NULL);

	free(fs_command);
	fs_command = (char *) NULL;

	if (command) {
		cdw_vdm ("INFO: xorriso command for burning from files: \"%s\"\n", command);
		return command;
	} else {
		cdw_vdm ("ERROR: failed to create xorriso command string\n");
		return (char *) NULL;
	}
}





/* xorriso -as cdrecord -vv -dev=/dev/scd0 /tmp/image.iso */
char *cdw_xorriso_create_command_burn_from_image(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: incorrect task id: %lld\n", task->id);
	cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT,
		    "ERROR: session mode is not set properly\n");
	cdw_assert (task->burn.tool.id == CDW_TOOL_XORRISO,
		    "ERROR: incorrect tool id: %lld\n", task->burn.tool.id);
	cdw_assert (task->burn.tool.label, "ERROR: tool fullpath is NULL\n");

	const char *device_fullpath = cdw_drive_get_drive_fullpath();
	cdw_assert (device_fullpath, "ERROR: \"device fullpath\" is NULL\n");

	char speed_string[5 + 1];
	snprintf(speed_string, 5 + 1, "%lld%s ", task->burn.speed_id, disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD ? "c" : "d");

	bool close = cdw_xorriso_close_disc(task);

	char *command = cdw_string_concat(task->burn.tool.label,
					  " -as cdrecord -vv dev=",
					  device_fullpath,
					  /* make sure to pass "dummy" option
					     to emulated cdrecord, don't use
					     xorriso's dummy option */
					  " ", task->burn.dummy ? " -dummy " : " ",
					  " speed=", speed_string,
					  task->image_file_fullpath,

					  " -- ",

					  /* TODO: where is "dummy" option? Answer: look few lines up, you silly. */
					  " ", task->burn.xorriso_burn_other_options ? task->burn.xorriso_burn_other_options : " ", " ",

					  close ? " -close on " : " -close off ",
					  (char *) NULL);

	if (command) {
		cdw_vdm ("INFO: xorriso command for burning from image: \"%s\"\n", command);
		return command;
	} else {
		cdw_vdm ("ERROR: failed to create xorriso command string\n");
		return (char *) NULL;
	}
}





char *cdw_xorriso_create_command_create_iso_fs(cdw_task_t *task)
{
	cdw_assert (task->id == CDW_TASK_CREATE_IMAGE_ISO9660
		    || task->id == CDW_TASK_BURN_FROM_FILES,
		    "ERROR: incorrect task id: %lld\n", task->id);
	cdw_assert (task->create_image.tool.id == CDW_TOOL_XORRISO,
		    "ERROR: incorrect tool id: %lld\n", task->create_image.tool.id);

	const char *graftpoints_fullpath = cdw_graftpoints_get_fullpath();
	cdw_assert (graftpoints_fullpath, "ERROR: graftpoints fullpath is NULL\n");

	bool useful_rr = task->create_image.iso9660.rock_ridge == CDW_ISO9660_RR_USEFUL;
	char *meta_information = cdw_xorriso_create_meta_information_string(task);

	char *command = cdw_string_concat(task->create_image.iso9660.joliet_information ? " -joliet on " : " -joliet off ",
					  task->create_image.iso9660.follow_symlinks ? " -follow link " : "",
					  " -pathspecs on ",

					  task->create_image.iso9660.pad ? " -padding 300k " : " -padding 0 ",


					  " -path_list ", graftpoints_fullpath, " ",
					  meta_information ? meta_information : "", /* already surrounded by spaces */
					  " ", task->create_image.iso9660.xorriso_iso_other_options ? task->create_image.iso9660.xorriso_iso_other_options : " ", " ",
					  useful_rr ? " -find / -true -exec mkisofs_r -- " : "",

					  (char *) NULL);

	cdw_string_delete(&meta_information);

	if (command) {
		cdw_vdm ("INFO: xorriso command string for ISO9660 fs: \"%s\"\n", command);
		return command;
	} else {
		cdw_vdm ("ERROR: failed to create xorriso command string for ISO9660 fs\n");
		return (char *) NULL;
	}
}





char *cdw_xorriso_create_meta_information_string(cdw_task_t *task)
{
	/* This data can be edited in main window of "burn from files"
	   wizard or "crate image" wizard. */
	size_t volume_id = strlen(task->create_image.iso9660.volume_id);

	/* This data can be edited in "meta data" page of "Options
	   window" in "create image" or "burn from files" wizard. */
	size_t publisher     = strlen(task->create_image.iso9660.publisher);
	size_t system_id     = strlen(task->create_image.iso9660.system_id);
	size_t volume_set_id = strlen(task->create_image.iso9660.volume_set_id);

	char *s = cdw_string_concat(" ",
				    volume_id ? " -volid \"" : "",
				    volume_id ? task->create_image.iso9660.volume_id : "",
				    volume_id ? "\" " : "",

				    publisher ? " -publisher \"" : " ",
				    publisher ? task->create_image.iso9660.publisher : "",
				    publisher ? "\" " : "",

				    system_id ? " -system_id \"" : "",
				    system_id ? task->create_image.iso9660.system_id : "",
				    system_id ? "\" " : "",

				    volume_set_id ? " -volset_id \"" : "",
				    volume_set_id ? task->create_image.iso9660.volume_set_id : "",
				    volume_set_id ? "\" " : "",

				    " ", (char *) NULL);

	cdw_vdm ("INFO: meta info string = \"%s\"\n", s);
	return s;
}





char *cdw_xorriso_create_command_media_info(cdw_task_t *task)
{
	cdw_assert (task->media_info.tool.label, "ERROR: tool fullpath is NULL\n");

	const char *device = cdw_drive_get_drive_fullpath();
	cdw_assert (device, "ERROR: \"device\" fullpath is NULL\n");

	char *command = cdw_string_concat(task->media_info.tool.label,
					  " -indev ",
					  device, " -toc ",
					  (char *) NULL);

	if (command) {
		cdw_vdm ("INFO: xorriso command string for media info: \"%s\"\n", command);
		return command;
	} else {
		cdw_vdm ("ERROR: failed to create xorriso command string for media info\n");
		return (char *) NULL;
	}
}





char *cdw_xorriso_create_command_erase(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->erase.tool.label, "ERROR: \"erase\" tool fullpath is NULL\n");

	const char *device = cdw_drive_get_drive_fullpath();
	cdw_assert (device, "ERROR: \"device\" fullpath is NULL\n");

	char blank_mode[30 + 1];
	if (disc->type == CDW_CD_RW) {
		if (task->erase.erase_mode == CDW_ERASE_MODE_FAST) {
			snprintf(blank_mode, 30 + 1, " -blank fast ");
		} else {
			snprintf(blank_mode, 30 + 1, " -blank full ");
		}
	} else {
		cdw_assert (0, "ERROR: xorriso can't work with media type %d\n", disc->type);
	}

	char speed_string[5 + 1];
	snprintf(speed_string, 5 + 1, "%lld%s ", task->erase.speed_id, disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD ? "c" : "d");


	char *command = cdw_string_concat(task->erase.tool.label,
					  " -dev ",  device, " ",
					  blank_mode,
					  " -speed ", speed_string, " ",
					  (char *) NULL);

	if (command) {
		cdw_vdm ("INFO: xorriso command string for erasing: \"%s\"\n", command);
		return command;
	} else {
		cdw_vdm ("ERROR: failed to create xorriso command string for erasing\n");
		return (char *) NULL;
	}
}





cdw_rv_t cdw_xorriso_set_disc_states(cdw_disc_t *disc)
{
	if (disc->xorriso_info.is_blank == CDW_TRUE) {
		disc->state_empty = CDW_TRUE;

		/* for empty discs xorriso prints only "is blank";
		   this information doesn't tell me if disc is writable;
		   to be 100% sure that the disc is writable I have
		   to make one more function call */
		if (cdw_disc_is_disc_type_writable(disc)) {
			disc->state_writable = CDW_TRUE;
			cdw_vdm ("INFO: disc is blank and writable\n");
		} else {
			disc->state_writable = CDW_FALSE;
			cdw_vdm ("INFO: disc is blank and not writable\n");
		}
	} else {
		cdw_vdm ("INFO: disc is not blank\n");
	}

	if (disc->xorriso_info.is_written == CDW_TRUE) {
		disc->state_empty = CDW_FALSE;
	}
	if (disc->xorriso_info.is_appendable == CDW_TRUE) {
		disc->state_writable = CDW_TRUE;
	}
	if (disc->xorriso_info.is_closed == CDW_TRUE) {
		disc->state_writable = CDW_FALSE;
	}

	return CDW_OK;
}





bool cdw_xorriso_close_disc(cdw_task_t *task)
{
	if (task->burn.session_mode == CDW_SESSION_MODE_START_MULTI
	    || task->burn.session_mode == CDW_SESSION_MODE_CONTINUE_MULTI) {
		/* note that write_mode == CDW_SESSION_MODE_START_MULTI may
		   be also true for writing image */

		return false;

	} else if (task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE
		   || task->burn.session_mode == CDW_SESSION_MODE_WRITE_FINAL) {
		/* note that write_session_mode == CDW_SESSION_MODE_CREATE_SINGLE
		   may be also true for writing image */

		/* I want to make it explicit */
		return true;
	} else {
		/* I want to make it explicit too */
		return true;
	}
}





bool cdw_xorriso_allow_dummy(__attribute__((unused)) cdw_disc_t *disc)
{
	return true;
}
