#include "Device/Coord/ICoordSystem.h"
#include "GUI/Model/Data/DataItemUtil.h"
#include "GUI/Model/Data/IntensityDataItem.h"
#include "GUI/Model/Device/InstrumentItems.h"
#include "GUI/Model/Device/RealItem.h"
#include "GUI/Model/Model/RealModel.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include "GUI/Model/Project/ProjectUtil.h"
#include "GUI/Support/Util/Path.h"
#include "Tests/GTestWrapper/google_test.h"
#include "Tests/Unit/GUI/Utils.h"
#include <QFileInfo>
#include <QSignalSpy>
#include <QUuid>

class TestProjectDocument : public ::testing::Test {
protected:
    //! helper method to modify something in a model
    void modify_models(ProjectDocument& doc)
    {
        auto* instrument = doc.instrumentModel()->instrumentItems().front();
        doc.multiNotifier()->setInstrumentName(instrument, QUuid::createUuid().toString());
    }
};

TEST_F(TestProjectDocument, projectDocument)
{
    const QString projectDir("projectDocument");
    UTest::GUI::create_dir(projectDir);
    const QString ext = QString(GUI::Project::Util::projectFileExtension);
    const QString projectFileName(projectDir + "/document" + ext);

    ProjectDocument document;
    // Checking initial document status
    EXPECT_FALSE(document.isModified());
    EXPECT_FALSE(document.hasValidNameAndPath());
    EXPECT_EQ(document.projectDir(), QString());
    EXPECT_EQ(document.projectName(), QString());
    EXPECT_EQ(document.projectFullPath(), QString());

    auto* instrument = document.instrumentModel()->addInstrumentItem<GISASInstrumentItem>();
    instrument->setInstrumentName("GISAS");

    // Checking document name and isModified status after project save
    document.saveProjectFileWithData(projectFileName);
    EXPECT_TRUE(document.hasValidNameAndPath());
    EXPECT_EQ(document.projectDir(), projectDir);
    EXPECT_EQ(document.projectName(), "document");
    EXPECT_EQ(document.projectFullPath(), projectFileName);
    EXPECT_FALSE(document.isModified());

    QSignalSpy spyDocument(&document, SIGNAL(modifiedStateChanged()));
    EXPECT_EQ(spyDocument.count(), 0);

    // Changing document and checking its status
    modify_models(document);
    EXPECT_TRUE(document.isModified());
    EXPECT_EQ(spyDocument.count(), 1);

    // Saving document
    document.saveProjectFileWithData(projectFileName);
    EXPECT_FALSE(document.isModified());
    EXPECT_EQ(spyDocument.count(), 2);

    QFileInfo info(projectFileName);
    EXPECT_TRUE(info.exists());
}

TEST_F(TestProjectDocument, projectDocumentWithData)
{
    const QString projectDir("projectDocumentWithData");
    UTest::GUI::create_dir(projectDir);
    const QString ext = QString(GUI::Project::Util::projectFileExtension);

    ProjectDocument document;
    auto* instrument = document.instrumentModel()->addInstrumentItem<GISASInstrumentItem>();
    instrument->setInstrumentName("GISAS");
    RealItem* realData = UTest::GUI::createRealData("TestData", *document.realModel());
    ASSERT(realData);
    DataItem* intensityItem = realData->dataItem();
    const auto converter =
        document.instrumentModel()->instrument2DItems().front()->createCoordSystem();
    ASSERT_TRUE(converter);
    GUI::Model::DataItemUtil::createDefaultDetectorMap(intensityItem, *converter);

    intensityItem->setSaveInBackground(false);
    document.setProjectDir(projectDir);
    document.saveProjectFileWithData(projectDir + "/untitled" + ext);

    QFileInfo info(projectDir + "/untitled" + ext);
    EXPECT_TRUE(info.exists());

    info.setFile(projectDir + "/realdata_TestData_0.int.gz");
    EXPECT_TRUE(info.exists());
}

//! Testing ProjectDocument on simple documents (no heavy data).
//! ProjectDocument should not be able to save project file.
//! Regression test for issue #1136 ("After a failed project saving, no saving takes place any more;
//! crash when changing projects")

TEST_F(TestProjectDocument, failingProjectSave)
{
    const QString projectDir("failingSaveService");
    // do NOT create dir in order to force saving to fail
    const QString ext = QString(GUI::Project::Util::projectFileExtension);
    const QString projectFileName(projectDir + "/document" + ext);

    std::unique_ptr<ProjectDocument> document(new ProjectDocument);
    auto* instrument = document->instrumentModel()->addInstrumentItem<GISASInstrumentItem>();
    instrument->setInstrumentName("GISAS");
    modify_models(*document);

    EXPECT_FALSE(QFile::exists(projectFileName));

    QSignalSpy spyDocument(document.get(), SIGNAL(modifiedStateChanged()));

    EXPECT_THROW(document->saveProjectFileWithData(projectFileName), std::runtime_error);

    EXPECT_EQ(spyDocument.count(), 0);
    EXPECT_FALSE(QFile::exists(projectFileName));

    // after failed save, document should still be in modified state
    EXPECT_TRUE(document->isModified());
}
