// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/gcm_driver/crypto/gcm_crypto_test_helpers.h"

#include <stddef.h>

#include <sstream>
#include <string>

#include "base/base64url.h"
#include "base/strings/string_util.h"
#include "components/gcm_driver/common/gcm_messages.h"
#include "components/gcm_driver/crypto/gcm_message_cryptographer.h"
#include "components/gcm_driver/crypto/p256_key_util.h"
#include "crypto/random.h"

namespace gcm {

bool CreateEncryptedPayloadForTesting(const base::StringPiece& payload,
                                      const base::StringPiece& peer_public_key,
                                      const base::StringPiece& auth_secret,
                                      IncomingMessage* message) {
  DCHECK(message);

  std::string private_key, public_key_x509, public_key;

  // Create an ephemeral key-pair for the sender.
  if (!CreateP256KeyPair(&private_key, &public_key_x509, &public_key))
    return false;

  std::string shared_secret;

  // Calculate the shared secret between the sender and its peer.
  if (!ComputeSharedP256Secret(private_key, public_key_x509, peer_public_key,
                               &shared_secret)) {
    return false;
  }

  std::string salt;

  // Generate a cryptographically secure random salt for the message.
  const size_t salt_size = GCMMessageCryptographer::kSaltSize;
  crypto::RandBytes(base::WriteInto(&salt, salt_size + 1), salt_size);

  GCMMessageCryptographer cryptographer(peer_public_key, public_key,
                                        auth_secret.as_string());

  size_t record_size;
  std::string ciphertext;

  if (!cryptographer.Encrypt(payload, shared_secret, salt, &record_size,
                             &ciphertext)) {
    return false;
  }

  std::string encoded_salt, encoded_public_key;

  // Create base64url encoded representations of the salt and local public key.
  base::Base64UrlEncode(salt, base::Base64UrlEncodePolicy::OMIT_PADDING,
                        &encoded_salt);
  base::Base64UrlEncode(public_key, base::Base64UrlEncodePolicy::OMIT_PADDING,
                        &encoded_public_key);

  // Write the Encryption header value to |*message|.
  std::stringstream encryption_header;
  encryption_header << "salt=" << encoded_salt << ";rs=" << record_size;

  message->data["encryption"] = encryption_header.str();

  // Write the Crypto-Key value to |*message|.
  std::stringstream crypto_key_header;
  crypto_key_header << "dh=" << encoded_public_key;

  message->data["crypto-key"] = crypto_key_header.str();

  message->raw_data.swap(ciphertext);
  return true;
}

}  // namespace gcm
