#! /usr/bin/env python

# Genetic
# Copyright (C) 2001 Jean-Baptiste LAMY
#
# This program is free software. See README or LICENSE for the license terms.

import random, string, math, genetic.organism, genetic.lifecycle
from genetic import organism, lifecycle


# The only thing we can currently do is minimizing a function -- Hey, if you're
# not sure, yes it means that it can do a LOT of things !!
#
# So first, we need to set the function to minimize. We'll call it FUNC.
#
# Obviously, x ** 2 + y ** 2 is minimum for (x, y) = (0.0, 0.0) !

FUNC = lambda x, y: x ** 2 + y ** 2

# Now, let create a new organism class, FUNCMinimizer, that extends the base
# organism class, organism.Organism.
# This organism has only one characteristic, FUNC, and it is computed by our
# FUNC function, with a recessive behaviour -- RECESSIVE_PHENOTYPE means that
# we choose the best gene's combination.

class FUNCMinimizer(organism.Organism):
  characteristics = [
    organism.Characteristic("FUNC", FUNC, organism.RECESSIVE_PHENOTYPE)
    ]
  
  # This compare method says how we should compare 2 organisms
  # Remember : the smaller is the better !
  # Here, comparing 2 organisms means to compare their FUNC value. We want the
  # smaller one !
  # If there was several characteristics, we would choose one or a combinate
  # them.
  def __cmp__(self, other):
    # self and other are both organisms of class FUNCMinimizer.
    # The FUNC attribute corresponds to the value given by FUNC for the
    # organism
    return cmp(self.FUNC, other.FUNC)


# But how work genetic algorythms ?
#
# It's easy : we create organisms. Each organism has some genetic stuff called 'genotype'.
# This genotype contains genes. Genes are the argument for FUNC, "x" and "y" here.
#
# Genes are organized in chromosoms (a group of genes), and the genotype is a list of pairs
# of chromosoms. Both chromosoms of a pair have the same gene, but possibly with 2 different
# values -- e.g. x = 0.0 and x = 1.0.
#
# So a single organism carries MANY VALUES (usually 2) for a given gene / variable.
#
# As we have many values for each gene, we must choose some for FUNC ! This is done by the
# phenotype function, i call it PHENOTYPE. PHENOTYPE takes a genotype (= a list of pairs of
# chromosoms) as argument, and returns the corresponding phenotype (a tuple with the value
# returned by FUNC, and the choosen genes's values).
# The lower the phenotype is, the "better" the organism is.
#
# An organism can reproduce with another (i've not implemented any "sex rextriction", so
# anyone can reproduce with anyone). Reproduction create a new organism, whose genotype is
# a mix of its parent genotype.
#
# Running a generation consists in creating a lot of children from the current organism pool,
# then all the organisms die and they are replaced by their best children -- other die too.
# Lots of deads... we are ready for a next generation. 
#
# The phenotype function.
#
# 3 functions are provided in the organism module; you can create your own if you want.
# The RECESSIVE_PHENOTYPE function correspond to a recessive phenotype = it choose the
# genes's combination that give the best result (=the smaller result, as we want to
# minimize), and often give the best result. But PERCHROMOSOME_DOMINANCY_PHENOTYPE is faster.



# Create an initial set of organisms. We need at least 2 of them.
#
# Organism can be created from a list of chromosoms pairs, or, as here, from
# a list of chromosoms. In this case, the organism is homozygote, and both
# chromosoms of each pair are the same.
# A chromosom is created by passing to its constructor its genes :
# Chromosom(gene = value, ...)
#
# You can provide as many genes as you want; but you MUST provide genes that
# have the same names that the arguments of FUNC !
#
# The value of the x and y genes are totally arbitrary !
# We may also choose to put the "x" and the "y" genes on 2 separate chromosoms.
# This choice is arbitrary too.

organismA = FUNCMinimizer([
  organism.Chromosom(x = 6.2, y = -5.4),
  ])

organismB = FUNCMinimizer([
  organism.Chromosom(x = -40.0, y = 1.0),
  ])

organisms = [organismA, organismB]


# Define some parameters

# True for elitism (= always keep the best organism)
ELITISM = 1

# Number of generation to run
NB_GENERATION = 10

# Number of children each generation made
NB_CHILDREN  = 100

# Number of organisms kept per generation (other children doesn't survive)
NB_ORGANISMS = 10


# Ready !
#
# This will make live a lot of organism, and finally print the last and "best" one -- the
# one whose phenotype minimizes FUNC !

lifecycle.run(organisms, ELITISM, NB_GENERATION, NB_CHILDREN, NB_ORGANISMS)

# The output will look like that (but values will probably differ) :
#
# FUNC('x', 'y') : 0.000643230178664(0.024210502678058316, 0.0075552457762450564)
# genotype, chromosome 0 A :
#     __break__                    : 0.00129864317455
#     __crossover__                : 0.191624101595
#     __deletion__                 : 0.00266990769512
#     __dominancy__                : 0.0238659494491
#     __duplicate__                : 0.0195951343295
#     __loss__                     : 0.0495540539135
#     __mutampl__                  : 0.315303864049
#     __mutation__                 : 0.0980921568131
#     x                            : -0.177204523541
#     y                            : 0.00755524577625
# genotype, chromosome 0 B :
#     __break__                    : 0.000186398770834
#     __crossover__                : 0.0628874380032
#     __deletion__                 : 0.00564882153852
#     __dominancy__                : 0.00724456057466
#     __duplicate__                : 0.00610509831268
#     __loss__                     : 0.0384499434339
#     __mutampl__                  : 0.0224471002322
#     __mutation__                 : 3.85230515612
#     x                            : 0.0242105026781
#     y                            : -3.87113452133
#
# The first line shows the phenotype : the minimal value found for FUNC
# (=0.000643230178664), and the corresponding genes's value (x and y).
#
# The following describes the organism's genetic stuff. Here, you can see one
# pair of 2 chromosoms (named "0 A" and "0 B"), and a line for each gene, with
# its name and its value.
#
# Don't care about the __<something>__ value for now. But notice that the x and
# y value choosen for the phenotype are the "best" ones -- the ones for which
# FUNC returns the lower number. This is because we've used the
# RECESSIVE_PHENOTYPE function.


